/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * FindBusinessQueryHelper.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.uddi.query;

import java.util.List;

import org.ow2.dragon.persistence.bo.common.CategoryBag;
import org.ow2.dragon.persistence.bo.common.KeyedReference;
import org.ow2.dragon.persistence.bo.common.Name;
import org.ow2.dragon.persistence.bo.organization.DiscoveryUrl;
import org.ow2.dragon.persistence.bo.organization.OrganizationUnit;
import org.ow2.dragon.service.uddi.FindQualifiers;
import org.ow2.dragon.util.StringHelper;

import com.trg.search.Filter;
import com.trg.search.IMutableSearch;
import com.trg.search.Search;

/**
 * @author ofabre - ebmwebsourcing
 * 
 */
public class FindBusinessQueryHelper extends CommonQueryHelper {

    public static IMutableSearch constructFindBusinessSearch(FindQualifiers findQualifiers,
            CategoryBag dragonCategoryBag, List<KeyedReference> dragonIdentifierBag,
            List<DiscoveryUrl> discoveryUrls, List<String> matchingServices, List<Name> names,
            List<String> relatedOrgsKeys, Paging paging)  {
        Class<OrganizationUnit> searchedType = OrganizationUnit.class;
        Search searchQuery = createDistinctSearch(searchedType);

        // Add discovery URLs key filter
        if (discoveryUrls != null && !discoveryUrls.isEmpty()) {
            searchQuery.addFilter(createDiscoveryURLsFilter(discoveryUrls));
        }

        // Add related orgs key filter
        if (relatedOrgsKeys != null && !relatedOrgsKeys.isEmpty()) {
            searchQuery.addFilter(createRelatedOrgsFilter(relatedOrgsKeys));
        }

        // Add category filters
        if (dragonCategoryBag != null) {
            searchQuery.addFilter(createCategoryBagFilter(findQualifiers, dragonCategoryBag));
        }

        // Add identifier filters
        if (dragonIdentifierBag != null && !dragonIdentifierBag.isEmpty()) {
            searchQuery.addFilter(createIdentifierBagFilter(findQualifiers, dragonIdentifierBag));
        }

        // Add names filter
        if (names != null && !names.isEmpty()) {
            searchQuery.addFilter(createNamesFilter(searchedType, findQualifiers, names));
        }

        // if matching services is not null or empty, the returned business must
        // own one of the services found
        if (matchingServices != null && !matchingServices.isEmpty()) {
            searchQuery.addFilter(createServiceKeysFilter(matchingServices));
        }

        // Add sort options
        addSortOptions(searchedType, searchQuery, findQualifiers);

        // Add paging options
        addPagingOptions(searchQuery, paging);

        return searchQuery;
    }

    private static Filter createDiscoveryURLsFilter(List<DiscoveryUrl> discoveryUrls) {
        Filter result = Filter.or();
        for (DiscoveryUrl discoveryURL : discoveryUrls) {
            if (!StringHelper.isNullOrEmpty(discoveryURL.getUseType())) {
                result.add(Filter.some("discoveryUrls", Filter.and(Filter.equal("url", discoveryURL
                        .getUrl()), Filter.equal("useType", discoveryURL.getUseType()))));
            } else {
                result.add(Filter.some("discoveryUrls", Filter
                        .equal("url", discoveryURL.getUrl())));
            }
        }
        return result;
    }

    private static Filter createRelatedOrgsFilter(List<String> relatedOrgsKeys) {
        return Filter.in("id", relatedOrgsKeys);
    }

    private static Filter createServiceKeysFilter(List<String> serviceKeys) {
        Filter or = Filter.or();
        for (String string : serviceKeys) {
            or.add(Filter.some("fromLinks", Filter.equal("to.id", string)));
        }
        return or;
    }
}
