/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * InquireImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.uddi.v2.impl;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;
import javax.xml.ws.handler.MessageContext;

import org.apache.cxf.jaxws.context.WebServiceContextImpl;
import org.ow2.dragon.persistence.bo.common.CategoryBag;
import org.ow2.dragon.persistence.bo.common.KeyedReference;
import org.ow2.dragon.persistence.bo.common.Name;
import org.ow2.dragon.persistence.bo.common.TModel;
import org.ow2.dragon.persistence.bo.deployment.Endpoint;
import org.ow2.dragon.persistence.bo.deployment.TechnicalService;
import org.ow2.dragon.persistence.bo.organization.DiscoveryUrl;
import org.ow2.dragon.persistence.bo.organization.OrganizationUnit;
import org.ow2.dragon.persistence.dao.UniversalUnifiedDAO;
import org.ow2.dragon.persistence.util.UDDIIdentifierGenerator;
import org.ow2.dragon.service.uddi.query.FindBindingQueryHelper;
import org.ow2.dragon.service.uddi.query.FindBusinessQueryHelper;
import org.ow2.dragon.service.uddi.query.FindRelatedBusinessesQueryHelper;
import org.ow2.dragon.service.uddi.query.FindServiceQueryHelper;
import org.ow2.dragon.service.uddi.query.FindTModelQueryHelper;
import org.ow2.dragon.service.uddi.query.Paging;
import org.ow2.dragon.service.uddi.v2.error.ErrorMessage;
import org.ow2.dragon.service.uddi.v2.error.InvalidKeyPassedException;
import org.ow2.dragon.service.uddi.v2.error.UnsupportedException;
import org.ow2.dragon.util.StringHelper;
import org.uddi.api_v2.BindingDetail;
import org.uddi.api_v2.BindingTemplate;
import org.uddi.api_v2.BusinessDetail;
import org.uddi.api_v2.BusinessDetailExt;
import org.uddi.api_v2.BusinessEntity;
import org.uddi.api_v2.BusinessList;
import org.uddi.api_v2.BusinessService;
import org.uddi.api_v2.FindBinding;
import org.uddi.api_v2.FindBusiness;
import org.uddi.api_v2.FindQualifiers;
import org.uddi.api_v2.FindRelatedBusinesses;
import org.uddi.api_v2.FindService;
import org.uddi.api_v2.FindTModel;
import org.uddi.api_v2.GetBindingDetail;
import org.uddi.api_v2.GetBusinessDetail;
import org.uddi.api_v2.GetBusinessDetailExt;
import org.uddi.api_v2.GetServiceDetail;
import org.uddi.api_v2.GetTModelDetail;
import org.uddi.api_v2.RelatedBusinessesList;
import org.uddi.api_v2.ServiceDetail;
import org.uddi.api_v2.ServiceInfo;
import org.uddi.api_v2.ServiceInfos;
import org.uddi.api_v2.ServiceList;
import org.uddi.api_v2.TModelBag;
import org.uddi.api_v2.TModelDetail;
import org.uddi.api_v2.TModelList;
import org.uddi.api_v2.Truncated;
import org.uddi.inquiry_v2.DispositionReport;
import org.uddi.inquiry_v2.Inquire;

import com.trg.search.IMutableSearch;

/**
 * @author ofabre - ebmwebsourcing
 * 
 */
public class InquireImpl implements Inquire {

    private WebServiceContextImpl webServiceContext;

    private UniversalUnifiedDAO universalUnifiedDAO;

    private UDDITransferObjectAssembler transferObjectAssembler;

    public InquireImpl(UniversalUnifiedDAO universalUnifiedDAO,
            WebServiceContextImpl webServiceContext) {
        this.universalUnifiedDAO = universalUnifiedDAO;
        this.webServiceContext = webServiceContext;
        this.transferObjectAssembler = new UDDITransferObjectAssembler(universalUnifiedDAO);
        /* this.inquiryValidator = new InquiryValidator(); */
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.uddi.inquiry_v2.Inquire#findBinding(org.uddi.api_v2.FindBinding)
     */
    public BindingDetail findBinding(FindBinding body) throws DispositionReport {
        // TODO Handle authentication

        // TODO Validate query
        // inquiryValidator.validateFindBinding(body);

        BindingDetail result = new BindingDetail();
        List<BindingTemplate> bindingTemplates = result.getBindingTemplate();

        // Validate find qualifiers
        FindQualifiersV2 findQualifiers = new FindQualifiersV2(body.getFindQualifiers());

        Paging paging = new Paging(1, body.getMaxRows());

        // Check serviceKey
        String v3ServiceKey = UDDIIdentifierGenerator.toUddiV3Id(body.getServiceKey());
        if (!StringHelper.isNullOrEmpty(v3ServiceKey)) {
            if (universalUnifiedDAO.get(TechnicalService.class, v3ServiceKey) == null) {
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.ServiceNotFound", body.getServiceKey()));
            }
        }

        // Retrieve all tModel keys related to tModelBag and find_Tmodel query
        // parts
        Set<String> tModelKeys = retrieveTModelKeys(body.getTModelBag());

        // Construct search filter
        IMutableSearch searchQuery = FindBindingQueryHelper.constructFindBindingSearch(
                findQualifiers, null, tModelKeys, v3ServiceKey, paging);

        // Search for Endpoints
        List<Endpoint> bindings = (List<Endpoint>) universalUnifiedDAO.search(Endpoint.class,
                searchQuery);

        if (bindings != null && !bindings.isEmpty()) {
            if (universalUnifiedDAO.count(Endpoint.class, searchQuery) > bindings.size()) {
                result.setTruncated(Truncated.TRUE);
            }
            for (Endpoint binding : bindings) {
                bindingTemplates.add(transferObjectAssembler.toUDDIBindingTemplate(binding));
            }
        }

        return result;
    }

    private Set<String> retrieveTModelKeys(org.uddi.api_v2.TModelBag modelBag)
            throws DispositionReport {
        Set<String> keys = new HashSet<String>();

        // Check all tmodel in tmodelBag
        if (modelBag != null) {
            if (modelBag.getTModelKey() != null) {
                for (String modelKey : modelBag.getTModelKey()) {
                    String uddiV3Key = UDDIIdentifierGenerator.toUddiV3Id(modelKey);
                    if (universalUnifiedDAO.get(TModel.class, uddiV3Key) != null) {
                        keys.add(uddiV3Key);
                    } else {
                        throw new InvalidKeyPassedException(new ErrorMessage(
                                "errors.invalidkey.TModelNotFound", modelKey));
                    }
                }
            }
        }

        return keys;
    }

    private List<String> retrieveMatchingServices(TModelBag modelBag, FindQualifiers findQualifiers)
            throws DispositionReport {
        List<String> servicesKeys = new ArrayList<String>();
        FindService includedRequest = new FindService();
        includedRequest.setTModelBag(modelBag);
        includedRequest.setFindQualifiers(findQualifiers);
        ServiceList serviceList = findService(includedRequest);
        if (serviceList != null) {
            ServiceInfos serviceInfos = serviceList.getServiceInfos();
            if (serviceInfos != null) {
                List<ServiceInfo> services = serviceInfos.getServiceInfo();
                if (services != null) {
                    for (ServiceInfo serviceInfo : services) {
                        servicesKeys.add(UDDIIdentifierGenerator.toUddiV3Id(serviceInfo
                                .getServiceKey()));
                    }
                }
            }
        }
        return servicesKeys;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.inquiry_v2.Inquire#findBusiness(org.uddi.api_v2.FindBusiness)
     */
    public BusinessList findBusiness(FindBusiness body) throws DispositionReport {
        // TODO Handle authentication

        // TODO Validate query
        // inquiryValidator.validateFindBusiness(body);

        BusinessList result = new BusinessList();

        // Validate find qualifiers
        FindQualifiersV2 findQualifiers = new FindQualifiersV2(body.getFindQualifiers());

        Paging paging = new Paging(1, body.getMaxRows());

        // Check categoryBag
        CategoryBag dragonCategoryBag = transferObjectAssembler.toDragonCategoryBag(body
                .getCategoryBag());

        // Check names
        List<Name> names = transferObjectAssembler.toDragonNames(body.getName());

        // Check identifierBag
        List<KeyedReference> dragonIdentifierBag = transferObjectAssembler
                .toDragonIdentifierBag(body.getIdentifierBag());

        // Check discoveryURLs
        List<DiscoveryUrl> discoveryUrls = transferObjectAssembler.toDragonDiscoveryUrls(body
                .getDiscoveryURLs());

        // Search for business Services matching the specified tModelKeyBag
        // and findTmodel
        List<String> serviceKeys = null;
        if (body.getTModelBag() != null) {
            serviceKeys = retrieveMatchingServices(body.getTModelBag(), body.getFindQualifiers());
        }

        // Construct search filter
        IMutableSearch searchQuery = FindBusinessQueryHelper.constructFindBusinessSearch(
                findQualifiers, dragonCategoryBag, dragonIdentifierBag, discoveryUrls, serviceKeys,
                names, null, paging);

        // Search for Services
        List<OrganizationUnit> orgs = (List<OrganizationUnit>) universalUnifiedDAO.search(
                OrganizationUnit.class, searchQuery);

        if (orgs != null && !orgs.isEmpty()) {
            if (universalUnifiedDAO.count(OrganizationUnit.class, searchQuery) > orgs.size()) {
                result.setTruncated(Truncated.TRUE);
            }
            result.setBusinessInfos(transferObjectAssembler.toUDDIBusinessInfos(orgs));
        }

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.uddi.inquiry_v2.Inquire#findRelatedBusinesses(org.uddi.api_v2.
     * FindRelatedBusinesses)
     */
    public RelatedBusinessesList findRelatedBusinesses(FindRelatedBusinesses body)
            throws DispositionReport {
        // TODO Handle authentication

        // TODO Validate query
        // inquiryValidator.validateFindRelatedBusinesses(body, false);

        RelatedBusinessesList result = new RelatedBusinessesList();

        // Validate find qualifiers
        FindQualifiersV2 findQualifiers = new FindQualifiersV2(body.getFindQualifiers());

        Paging paging = new Paging(1, body.getMaxRows());

        // Check businessKey
        String v3BusinessKey = UDDIIdentifierGenerator.toUddiV3Id(body.getBusinessKey());
        if (v3BusinessKey != null) {
            OrganizationUnit orgSource = (OrganizationUnit) universalUnifiedDAO.get(
                    OrganizationUnit.class, v3BusinessKey);
            if (orgSource == null) {
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.BusinessNotFound", body.getBusinessKey()));
            }
        }

        // Construct search filter
        IMutableSearch searchQuery = FindRelatedBusinessesQueryHelper
                .constructFindRelatedBusinessesSearch(findQualifiers, v3BusinessKey, paging);

        // Retrieve related orgs (parent org and all child orgs for the
        // moment)
        List<OrganizationUnit> relatedOrgs = (List<OrganizationUnit>) universalUnifiedDAO.search(
                OrganizationUnit.class, searchQuery);

        if (relatedOrgs != null && !relatedOrgs.isEmpty()) {
            if (universalUnifiedDAO.count(OrganizationUnit.class, searchQuery) > relatedOrgs.size()) {
                result.setTruncated(Truncated.TRUE);
            }
            result.setRelatedBusinessInfos(transferObjectAssembler
                    .toUDDIRelatedBusinessInfos(relatedOrgs));
        }
        result.setBusinessKey(body.getBusinessKey());

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.uddi.inquiry_v2.Inquire#findService(org.uddi.api_v2.FindService)
     */
    public ServiceList findService(FindService body) throws DispositionReport {
        // TODO Handle authentication

        // TODO Validate query
        // inquiryValidator.validateFindService(body);

        ServiceList result = new ServiceList();

        // Validate find qualifiers
        FindQualifiersV2 findQualifiers = new FindQualifiersV2(body.getFindQualifiers());

        Paging paging = new Paging(1, body.getMaxRows());

        // Check categoryBag
        CategoryBag dragonCategoryBag = transferObjectAssembler.toDragonCategoryBag(body
                .getCategoryBag());

        // Check Business Key
        String v3BusinessKey = UDDIIdentifierGenerator.toUddiV3Id(body.getBusinessKey());
        if (!StringHelper.isNullOrEmpty(v3BusinessKey)) {
            if (universalUnifiedDAO.get(OrganizationUnit.class, v3BusinessKey) == null) {
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.BusinessNotFound", body.getBusinessKey()));
            }
        }

        // Check names
        List<Name> names = transferObjectAssembler.toDragonNames(body.getName());

        // Retrieve all tModel keys related to tModelBag and find_Tmodel query
        // parts
        Set<String> tModelKeys = retrieveTModelKeys(body.getTModelBag());

        // Construct search filter
        IMutableSearch searchQuery = FindServiceQueryHelper.constructFindServiceSearch(
                findQualifiers, dragonCategoryBag, tModelKeys, v3BusinessKey, names, paging);

        // Search for Services
        List<TechnicalService> techServices = (List<TechnicalService>) universalUnifiedDAO.search(
                TechnicalService.class, searchQuery);

        if (techServices != null && !techServices.isEmpty()) {
            if (universalUnifiedDAO.count(TechnicalService.class, searchQuery) > techServices
                    .size()) {
                result.setTruncated(Truncated.TRUE);
            }
            result.setServiceInfos(transferObjectAssembler.toUDDIServiceInfos(techServices));
        }

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.uddi.inquiry_v2.Inquire#findTModel(org.uddi.api_v2.FindTModel)
     */
    public TModelList findTModel(FindTModel body) throws DispositionReport {
        // TODO Handle authentication

        // TODO Validate query
        // inquiryValidator.validateFindTModel(body, false);

        TModelList result = new TModelList();

        // Validate find qualifiers
        FindQualifiersV2 findQualifiers = new FindQualifiersV2(body.getFindQualifiers());

        Paging paging = new Paging(1, body.getMaxRows());

        // Check categoryBag
        CategoryBag dragonCategoryBag = transferObjectAssembler.toDragonCategoryBag(body
                .getCategoryBag());

        // Check identifierBag
        List<KeyedReference> dragonIdentifierBag = transferObjectAssembler
                .toDragonIdentifierBag(body.getIdentifierBag());

        // Check name
        Name name = transferObjectAssembler.toDragonName(body.getName());

        // Construct search filter
        IMutableSearch searchQuery = FindTModelQueryHelper.constructFindTModelSearch(
                findQualifiers, dragonCategoryBag, dragonIdentifierBag, name, paging);

        // Search for tmodel
        List<TModel> tmodels = (List<TModel>) universalUnifiedDAO.search(TModel.class, searchQuery);

        if (tmodels != null && !tmodels.isEmpty()) {
            if (universalUnifiedDAO.count(TModel.class, searchQuery) > tmodels.size()) {
                result.setTruncated(Truncated.TRUE);
            }
            result.setTModelInfos(transferObjectAssembler.toUDDITModelInfos(tmodels));
        }

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.inquiry_v2.Inquire#getBindingDetail(org.uddi.api_v2.GetBindingDetail
     * )
     */
    public BindingDetail getBindingDetail(GetBindingDetail body) throws DispositionReport {
        // TODO Validate query

        // TODO Handle authentication

        BindingDetail result = new BindingDetail();

        // Extract Keys
        List<String> keys = body.getBindingKey();
        for (String key : keys) {
            
            // Retrieve instance from Dragon DB
            Endpoint dragonEp = (Endpoint) universalUnifiedDAO.get(Endpoint.class, UDDIIdentifierGenerator.toUddiV3Id(key));
            if (dragonEp == null) {
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.BindingTemplatelNotFound", key));
            }

            setRequestOnTOA();

            // Convert Dragon instance into uddi instance
            BindingTemplate apiBindingTemplate = transferObjectAssembler
                    .toUDDIBindingTemplate(dragonEp);

            result.getBindingTemplate().add(apiBindingTemplate);
        }

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.uddi.inquiry_v2.Inquire#getBusinessDetail(org.uddi.api_v2.
     * GetBusinessDetail)
     */
    public BusinessDetail getBusinessDetail(GetBusinessDetail body) throws DispositionReport {
        // TODO Validate query

        // TODO Handle authentication

        BusinessDetail result = new BusinessDetail();

        // Extract Keys
        List<String> keys = body.getBusinessKey();
        for (String key : keys) {

            // Retrieve instance from Dragon DB
            OrganizationUnit dragonBusiness = (OrganizationUnit) universalUnifiedDAO.get(
                    OrganizationUnit.class, UDDIIdentifierGenerator.toUddiV3Id(key));
            if (dragonBusiness == null) {
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.BusinessNotFound", key));
            }

            setRequestOnTOA();

            // Convert Dragon instance into uddi instance
            BusinessEntity apiBusiness = transferObjectAssembler.toUDDIBusiness(dragonBusiness);

            result.getBusinessEntity().add(apiBusiness);
        }

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.uddi.inquiry_v2.Inquire#getBusinessDetailExt(org.uddi.api_v2.
     * GetBusinessDetailExt)
     */
    public BusinessDetailExt getBusinessDetailExt(GetBusinessDetailExt body)
            throws DispositionReport {
        throw new UnsupportedException(new ErrorMessage("errors.Unsupported",
                "getBusinessDetailExt"));
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.inquiry_v2.Inquire#getServiceDetail(org.uddi.api_v2.GetServiceDetail
     * )
     */
    public ServiceDetail getServiceDetail(GetServiceDetail body) throws DispositionReport {
        // TODO Validate query

        // TODO Handle authentication

        ServiceDetail result = new ServiceDetail();

        // Extract Keys
        List<String> keys = body.getServiceKey();
        for (String key : keys) {

            // Retrieve instance from Dragon DB
            TechnicalService dragonService = (TechnicalService) universalUnifiedDAO.get(
                    TechnicalService.class, UDDIIdentifierGenerator.toUddiV3Id(key));
            if (dragonService == null) {
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.ServiceNotFound", key));
            }

            setRequestOnTOA();

            // Convert Dragon instance into uddi instance
            BusinessService apiService = transferObjectAssembler.toUDDIService(dragonService);

            result.getBusinessService().add(apiService);
        }

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.inquiry_v2.Inquire#getTModelDetail(org.uddi.api_v2.GetTModelDetail
     * )
     */
    public TModelDetail getTModelDetail(GetTModelDetail body) throws DispositionReport {
        // TODO Validate query

        // TODO Handle authentication

        TModelDetail result = new TModelDetail();

        // Extract Keys
        List<String> keys = body.getTModelKey();
        for (String key : keys) {

            // Retrieve instance from Dragon DB
            TModel dragonTModel = (TModel) universalUnifiedDAO.get(TModel.class, UDDIIdentifierGenerator.toUddiV3Id(key));
            if (dragonTModel == null) {
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.TModelNotFound", key));
            }

            setRequestOnTOA();

            // Convert Dragon instance into uddi instance
            org.uddi.api_v2.TModel apiTModel = transferObjectAssembler.toUDDITModel(dragonTModel);

            result.getTModel().add(apiTModel);
        }

        return result;
    }

    /**
     * Retrieve servlet request to be able to generate overview docs urls
     */
    private void setRequestOnTOA() {
        HttpServletRequest request = (HttpServletRequest) webServiceContext.getMessageContext()
                .get(MessageContext.SERVLET_REQUEST);
        transferObjectAssembler.setRequest(request);
    }
}
