/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * UDDIInquiryImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.uddi.v3.impl;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.jws.WebService;
import javax.servlet.http.HttpServletRequest;
import javax.xml.ws.handler.MessageContext;

import org.apache.cxf.jaxws.context.WebServiceContextImpl;
import org.ow2.dragon.persistence.bo.common.CategoryBag;
import org.ow2.dragon.persistence.bo.common.KeyedReference;
import org.ow2.dragon.persistence.bo.common.Name;
import org.ow2.dragon.persistence.bo.common.TModel;
import org.ow2.dragon.persistence.bo.deployment.Endpoint;
import org.ow2.dragon.persistence.bo.deployment.TechnicalService;
import org.ow2.dragon.persistence.bo.organization.DiscoveryUrl;
import org.ow2.dragon.persistence.bo.organization.OrganizationUnit;
import org.ow2.dragon.persistence.dao.UniversalUnifiedDAO;
import org.ow2.dragon.service.uddi.query.FindBindingQueryHelper;
import org.ow2.dragon.service.uddi.query.FindBusinessQueryHelper;
import org.ow2.dragon.service.uddi.query.FindRelatedBusinessesQueryHelper;
import org.ow2.dragon.service.uddi.query.FindServiceQueryHelper;
import org.ow2.dragon.service.uddi.query.FindTModelQueryHelper;
import org.ow2.dragon.service.uddi.query.Paging;
import org.ow2.dragon.service.uddi.v3.error.ErrorMessage;
import org.ow2.dragon.service.uddi.v3.error.InvalidKeyPassedException;
import org.ow2.dragon.service.uddi.v3.validator.InquiryValidator;
import org.ow2.dragon.util.StringHelper;
import org.springframework.transaction.annotation.Transactional;
import org.uddi.api_v3.BindingDetail;
import org.uddi.api_v3.BindingTemplate;
import org.uddi.api_v3.BusinessDetail;
import org.uddi.api_v3.BusinessEntity;
import org.uddi.api_v3.BusinessList;
import org.uddi.api_v3.BusinessService;
import org.uddi.api_v3.FindBinding;
import org.uddi.api_v3.FindBusiness;
import org.uddi.api_v3.FindQualifiers;
import org.uddi.api_v3.FindRelatedBusinesses;
import org.uddi.api_v3.FindService;
import org.uddi.api_v3.FindTModel;
import org.uddi.api_v3.GetBindingDetail;
import org.uddi.api_v3.GetBusinessDetail;
import org.uddi.api_v3.GetOperationalInfo;
import org.uddi.api_v3.GetServiceDetail;
import org.uddi.api_v3.GetTModelDetail;
import org.uddi.api_v3.OperationalInfos;
import org.uddi.api_v3.RelatedBusinessInfo;
import org.uddi.api_v3.RelatedBusinessInfos;
import org.uddi.api_v3.RelatedBusinessesList;
import org.uddi.api_v3.ServiceDetail;
import org.uddi.api_v3.ServiceInfo;
import org.uddi.api_v3.ServiceInfos;
import org.uddi.api_v3.ServiceList;
import org.uddi.api_v3.TModelBag;
import org.uddi.api_v3.TModelDetail;
import org.uddi.api_v3.TModelInfo;
import org.uddi.api_v3.TModelInfos;
import org.uddi.api_v3.TModelList;
import org.uddi.api_v3_porttype.DispositionReportFaultMessage;
import org.uddi.api_v3_porttype.UDDIInquiryPortType;

import com.trg.search.IMutableSearch;

/**
 * @author ofabre - ebmwebsourcing
 * 
 */
@WebService(serviceName = "UDDIInquiryService", endpointInterface = "org.uddi.api_v3_porttype.UDDIInquiryPortType", targetNamespace = "urn:uddi-org:api_v3")
@Transactional(readOnly = true)
public class UDDIInquiryImpl implements UDDIInquiryPortType {

    private WebServiceContextImpl webServiceContext;

    private UniversalUnifiedDAO universalUnifiedDAO;

    private UDDITransferObjectAssembler transferObjectAssembler;

    private InquiryValidator inquiryValidator;

    public UDDIInquiryImpl(UniversalUnifiedDAO universalUnifiedDAO,
            WebServiceContextImpl webServiceContext) {
        this.universalUnifiedDAO = universalUnifiedDAO;
        this.webServiceContext = webServiceContext;
        this.transferObjectAssembler = new UDDITransferObjectAssembler(universalUnifiedDAO);
        this.inquiryValidator = new InquiryValidator();
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIInquiryPortType#findBinding(org.uddi.api_v3
     * .FindBinding)
     */
    public BindingDetail findBinding(FindBinding body) throws DispositionReportFaultMessage {
        // TODO Handle authentication

        // Validate query
        inquiryValidator.validateFindBinding(body);

        BindingDetail result = new BindingDetail();
        List<BindingTemplate> bindingTemplates = result.getBindingTemplate();

        // Validate find qualifiers
        org.ow2.dragon.service.uddi.v3.impl.FindQualifiersV3 findQualifiers = new org.ow2.dragon.service.uddi.v3.impl.FindQualifiersV3(
                body.getFindQualifiers());

        Paging paging = new Paging(body.getListHead(), body.getMaxRows());

        // Check categoryBag
        CategoryBag dragonCategoryBag = transferObjectAssembler.toDragonCategoryBag(body
                .getCategoryBag());

        // Check serviceKey
        if (!StringHelper.isNullOrEmpty(body.getServiceKey())) {
            if (universalUnifiedDAO.get(TechnicalService.class, body.getServiceKey()) == null) {
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.ServiceNotFound", body.getServiceKey()));
            }
        }

        // Retrieve all tModel keys related to tModelBag and find_Tmodel query
        // parts
        Set<String> tModelKeys = retrieveTModelKeys(body.getFindTModel(), body.getTModelBag());

        // Construct search filter
        IMutableSearch searchQuery = FindBindingQueryHelper.constructFindBindingSearch(
                findQualifiers, dragonCategoryBag, tModelKeys, body.getServiceKey(), paging);

        // Search for Endpoints
        List<Endpoint> bindings = (List<Endpoint>) universalUnifiedDAO.search(Endpoint.class,
                searchQuery);

        if (bindings != null && !bindings.isEmpty()) {
            result.setListDescription(transferObjectAssembler.toUDDIListDescription(
                    universalUnifiedDAO.count(Endpoint.class, searchQuery), bindings.size(), paging
                            .getFirstResult() + 1));
            for (Endpoint binding : bindings) {
                bindingTemplates.add(transferObjectAssembler.toUDDIBindingTemplate(binding));
            }
        }

        return result;
    }

    private Set<String> retrieveTModelKeys(FindTModel findTModel, TModelBag modelBag)
            throws DispositionReportFaultMessage {
        Set<String> keys = new HashSet<String>();

        // Check all tmodel in tmodelBag
        if (modelBag != null) {
            if (modelBag.getTModelKey() != null) {
                for (String modelKey : modelBag.getTModelKey()) {
                    if (universalUnifiedDAO.get(TModel.class, modelKey) != null) {
                        keys.add(modelKey);
                    } else {
                        throw new InvalidKeyPassedException(new ErrorMessage(
                                "errors.invalidkey.TModelNotFound", modelKey));
                    }
                }
            }
        }

        // find tmodels matching the given findTModel query
        if (findTModel != null) {
            TModelList modelList = findTModel(findTModel);
            if (modelList != null) {
                TModelInfos modelInfos = modelList.getTModelInfos();
                if (modelInfos != null) {
                    List<TModelInfo> infos = modelInfos.getTModelInfo();
                    if (infos != null) {
                        for (TModelInfo modelInfo : infos) {
                            keys.add(modelInfo.getTModelKey());
                        }
                    }
                }
            }
        }

        return keys;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIInquiryPortType#findBusiness(org.uddi.api_v3
     * .FindBusiness)
     */
    public BusinessList findBusiness(FindBusiness body) throws DispositionReportFaultMessage {
        // TODO Handle authentication

        // Validate query
        inquiryValidator.validateFindBusiness(body);

        BusinessList result = new BusinessList();

        // Validate find qualifiers
        org.ow2.dragon.service.uddi.v3.impl.FindQualifiersV3 findQualifiers = new org.ow2.dragon.service.uddi.v3.impl.FindQualifiersV3(
                body.getFindQualifiers());

        Paging paging = new Paging(body.getListHead(), body.getMaxRows());

        // Check categoryBag
        CategoryBag dragonCategoryBag = transferObjectAssembler.toDragonCategoryBag(body
                .getCategoryBag());

        // Check names
        List<Name> names = transferObjectAssembler.toDragonNames(body.getName());

        // Check identifierBag
        List<KeyedReference> dragonIdentifierBag = transferObjectAssembler
                .toDragonIdentifierBag(body.getIdentifierBag());

        // Check discoveryURLs
        List<DiscoveryUrl> discoveryUrls = transferObjectAssembler.toDragonDiscoveryUrls(body
                .getDiscoveryURLs());

        // Retrieve all related orgs keys associated to the
        // find_relatedBusinesses query part
        List<String> relatedOrgsKeys = retrieveRelatedOrgsKey(body.getFindRelatedBusinesses());

        // Search for business Services matching the specified tModelKeyBag
        // and findTmodel
        List<String> serviceKeys = null;
        if (body.getFindTModel() != null || body.getTModelBag() != null) {
            serviceKeys = retrieveMatchingServices(body.getFindTModel(), body.getTModelBag(), body
                    .getFindQualifiers());
        }

        // Construct search filter
        IMutableSearch searchQuery = FindBusinessQueryHelper.constructFindBusinessSearch(
                findQualifiers, dragonCategoryBag, dragonIdentifierBag, discoveryUrls,
                serviceKeys, names, relatedOrgsKeys, paging);

        // Search for Services
        List<OrganizationUnit> orgs = (List<OrganizationUnit>) universalUnifiedDAO.search(
                OrganizationUnit.class, searchQuery);

        if (orgs != null && !orgs.isEmpty()) {
            result.setListDescription(transferObjectAssembler.toUDDIListDescription(
                    universalUnifiedDAO.count(OrganizationUnit.class, searchQuery), orgs.size(),
                    paging.getFirstResult() + 1));
            result.setBusinessInfos(transferObjectAssembler.toUDDIBusinessInfos(orgs));
        }

        return result;
    }

    private List<String> retrieveRelatedOrgsKey(FindRelatedBusinesses findRelatedBusinesses)
            throws DispositionReportFaultMessage {
        List<String> relatedOrgs = new ArrayList<String>();
        if (findRelatedBusinesses != null) {
            RelatedBusinessesList businessesList = findRelatedBusinesses(findRelatedBusinesses);
            if (businessesList != null) {
                // Add all related orgs
                RelatedBusinessInfos businessInfos = businessesList.getRelatedBusinessInfos();
                if (businessInfos != null) {
                    List<RelatedBusinessInfo> businesses = businessInfos.getRelatedBusinessInfo();
                    if (businesses != null) {
                        for (RelatedBusinessInfo relatedBusinessInfo : businesses) {
                            relatedOrgs.add(relatedBusinessInfo.getBusinessKey());
                        }
                    }
                }
                // Add also the org itself
                relatedOrgs.add(businessesList.getBusinessKey());
            }
        }
        return relatedOrgs;
    }

    private List<String> retrieveMatchingServices(FindTModel findTModel, TModelBag modelBag,
            FindQualifiers findQualifiers) throws DispositionReportFaultMessage {
        List<String> servicesKeys = new ArrayList<String>();
        FindService includedRequest = new FindService();
        includedRequest.setTModelBag(modelBag);
        includedRequest.setFindTModel(findTModel);
        includedRequest.setFindQualifiers(findQualifiers);
        ServiceList serviceList = findService(includedRequest);
        if (serviceList != null) {
            ServiceInfos serviceInfos = serviceList.getServiceInfos();
            if (serviceInfos != null) {
                List<ServiceInfo> services = serviceInfos.getServiceInfo();
                if (services != null) {
                    for (ServiceInfo serviceInfo : services) {
                        servicesKeys.add(serviceInfo.getServiceKey());
                    }
                }
            }
        }
        return servicesKeys;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIInquiryPortType#findRelatedBusinesses(org
     * .uddi.api_v3.FindRelatedBusinesses)
     */
    public RelatedBusinessesList findRelatedBusinesses(FindRelatedBusinesses body)
            throws DispositionReportFaultMessage {
        // TODO Handle authentication

        // Validate query
        inquiryValidator.validateFindRelatedBusinesses(body, false);

        RelatedBusinessesList result = new RelatedBusinessesList();

        // Validate find qualifiers
        org.ow2.dragon.service.uddi.v3.impl.FindQualifiersV3 findQualifiers = new org.ow2.dragon.service.uddi.v3.impl.FindQualifiersV3(
                body.getFindQualifiers());

        Paging paging = new Paging(body.getListHead(), body.getMaxRows());

        // Check businessKey
        if (body.getBusinessKey() != null) {
            OrganizationUnit orgSource = (OrganizationUnit) universalUnifiedDAO.get(
                    OrganizationUnit.class, body.getBusinessKey());
            if (orgSource == null) {
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.BusinessNotFound", body.getBusinessKey()));
            }
        }

        // Construct search filter
        IMutableSearch searchQuery = FindRelatedBusinessesQueryHelper
                .constructFindRelatedBusinessesSearch(findQualifiers, body.getBusinessKey(), paging);

        // Retrieve related orgs (parent org and all child orgs for the
        // moment)
        List<OrganizationUnit> relatedOrgs = (List<OrganizationUnit>) universalUnifiedDAO.search(
                OrganizationUnit.class, searchQuery);

        if (relatedOrgs != null && !relatedOrgs.isEmpty()) {
            result.setListDescription(transferObjectAssembler.toUDDIListDescription(
                    universalUnifiedDAO.count(OrganizationUnit.class, searchQuery), relatedOrgs
                            .size(), paging.getFirstResult() + 1));
            result.setRelatedBusinessInfos(transferObjectAssembler
                    .toUDDIRelatedBusinessInfos(relatedOrgs));
        }
        result.setBusinessKey(body.getBusinessKey());

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIInquiryPortType#findService(org.uddi.api_v3
     * .FindService)
     */
    public ServiceList findService(FindService body) throws DispositionReportFaultMessage {
        // TODO Handle authentication

        // Validate query
        inquiryValidator.validateFindService(body);

        ServiceList result = new ServiceList();

        // Validate find qualifiers
        org.ow2.dragon.service.uddi.v3.impl.FindQualifiersV3 findQualifiers = new org.ow2.dragon.service.uddi.v3.impl.FindQualifiersV3(
                body.getFindQualifiers());

        Paging paging = new Paging(body.getListHead(), body.getMaxRows());

        // Check categoryBag
        CategoryBag dragonCategoryBag = transferObjectAssembler.toDragonCategoryBag(body
                .getCategoryBag());

        // Check Business Key
        if (!StringHelper.isNullOrEmpty(body.getBusinessKey())) {
            if (universalUnifiedDAO.get(OrganizationUnit.class, body.getBusinessKey()) == null) {
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.BusinessNotFound", body.getBusinessKey()));
            }
        }

        // Check names
        List<Name> names = transferObjectAssembler.toDragonNames(body.getName());

        // Retrieve all tModel keys related to tModelBag and find_Tmodel query
        // parts
        Set<String> tModelKeys = retrieveTModelKeys(body.getFindTModel(), body.getTModelBag());

        // Construct search filter
        IMutableSearch searchQuery = FindServiceQueryHelper
                .constructFindServiceSearch(findQualifiers, dragonCategoryBag, tModelKeys, body
                        .getBusinessKey(), names, paging);

        // Search for Services
        List<TechnicalService> techServices = (List<TechnicalService>) universalUnifiedDAO.search(
                TechnicalService.class, searchQuery);

        if (techServices != null && !techServices.isEmpty()) {
            result.setListDescription(transferObjectAssembler.toUDDIListDescription(
                    universalUnifiedDAO.count(TechnicalService.class, searchQuery), techServices
                            .size(), paging.getFirstResult() + 1));
            result.setServiceInfos(transferObjectAssembler.toUDDIServiceInfos(techServices));
        }

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIInquiryPortType#findTModel(org.uddi.api_v3
     * .FindTModel)
     */
    public TModelList findTModel(FindTModel body) throws DispositionReportFaultMessage {

        // TODO Handle authentication

        // Validate query
        inquiryValidator.validateFindTModel(body, false);

        TModelList result = new TModelList();

        // Validate find qualifiers
        org.ow2.dragon.service.uddi.v3.impl.FindQualifiersV3 findQualifiers = new org.ow2.dragon.service.uddi.v3.impl.FindQualifiersV3(
                body.getFindQualifiers());

        Paging paging = new Paging(body.getListHead(), body.getMaxRows());

        // Check categoryBag
        CategoryBag dragonCategoryBag = transferObjectAssembler.toDragonCategoryBag(body
                .getCategoryBag());

        // Check identifierBag
        List<KeyedReference> dragonIdentifierBag = transferObjectAssembler
                .toDragonIdentifierBag(body.getIdentifierBag());

        // Check name
        Name name = transferObjectAssembler.toDragonName(body.getName());

        // Construct search filter
        IMutableSearch searchQuery = FindTModelQueryHelper.constructFindTModelSearch(
                findQualifiers, dragonCategoryBag, dragonIdentifierBag, name, paging);

        // Search for tmodel
        List<TModel> tmodels = (List<TModel>) universalUnifiedDAO.search(TModel.class, searchQuery);

        if (tmodels != null && !tmodels.isEmpty()) {
            result.setListDescription(transferObjectAssembler.toUDDIListDescription(
                    universalUnifiedDAO.count(TModel.class, searchQuery), tmodels.size(), paging
                            .getFirstResult() + 1));
            result.setTModelInfos(transferObjectAssembler.toUDDITModelInfos(tmodels));
        }

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIInquiryPortType#getBindingDetail(org.uddi
     * .api_v3.GetBindingDetail)
     */
    public BindingDetail getBindingDetail(GetBindingDetail body)
            throws DispositionReportFaultMessage {
        // TODO Validate query

        // TODO Handle authentication

        BindingDetail result = new BindingDetail();

        // Extract Keys
        List<String> keys = body.getBindingKey();
        for (String key : keys) {

            // Retrieve instance from Dragon DB
            Endpoint dragonEp = (Endpoint) universalUnifiedDAO.get(Endpoint.class, key);
            if (dragonEp == null) {
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.BindingTemplatelNotFound", key));
            }

            setRequestOnTOA();

            // Convert Dragon instance into uddi instance
            BindingTemplate apiBindingTemplate = transferObjectAssembler
                    .toUDDIBindingTemplate(dragonEp);

            result.getBindingTemplate().add(apiBindingTemplate);
        }

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIInquiryPortType#getBusinessDetail(org.uddi
     * .api_v3.GetBusinessDetail)
     */
    public BusinessDetail getBusinessDetail(GetBusinessDetail body)
            throws DispositionReportFaultMessage {
        // TODO Validate query

        // TODO Handle authentication

        BusinessDetail result = new BusinessDetail();

        // Extract Keys
        List<String> keys = body.getBusinessKey();
        for (String key : keys) {

            // Retrieve instance from Dragon DB
            OrganizationUnit dragonBusiness = (OrganizationUnit) universalUnifiedDAO.get(
                    OrganizationUnit.class, key);
            if (dragonBusiness == null) {
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.BusinessNotFound", key));
            }

            setRequestOnTOA();

            // Convert Dragon instance into uddi instance
            BusinessEntity apiBusiness = transferObjectAssembler.toUDDIBusiness(dragonBusiness);

            result.getBusinessEntity().add(apiBusiness);
        }

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIInquiryPortType#getOperationalInfo(org.uddi
     * .api_v3.GetOperationalInfo)
     */
    public OperationalInfos getOperationalInfo(GetOperationalInfo body)
            throws DispositionReportFaultMessage {
        // TODO Auto-generated method stub
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIInquiryPortType#getServiceDetail(org.uddi
     * .api_v3.GetServiceDetail)
     */
    public ServiceDetail getServiceDetail(GetServiceDetail body)
            throws DispositionReportFaultMessage {
        // TODO Validate query

        // TODO Handle authentication

        ServiceDetail result = new ServiceDetail();

        // Extract Keys
        List<String> keys = body.getServiceKey();
        for (String key : keys) {

            // Retrieve instance from Dragon DB
            TechnicalService dragonService = (TechnicalService) universalUnifiedDAO.get(
                    TechnicalService.class, key);
            if (dragonService == null) {
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.ServiceNotFound", key));
            }

            setRequestOnTOA();

            // Convert Dragon instance into uddi instance
            BusinessService apiService = transferObjectAssembler.toUDDIService(dragonService);

            result.getBusinessService().add(apiService);
        }

        return result;
    }

    /**
     * Retrieve servlet request to be able to generate overview docs urls
     */
    private void setRequestOnTOA() {
        HttpServletRequest request = (HttpServletRequest) webServiceContext.getMessageContext()
                .get(MessageContext.SERVLET_REQUEST);
        transferObjectAssembler.setRequest(request);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIInquiryPortType#getTModelDetail(org.uddi
     * .api_v3.GetTModelDetail)
     */
    public TModelDetail getTModelDetail(GetTModelDetail body) throws DispositionReportFaultMessage {
        // TODO Validate query

        // TODO Handle authentication

        TModelDetail result = new TModelDetail();

        // Extract Keys
        List<String> keys = body.getTModelKey();
        for (String key : keys) {

            // Retrieve instance from Dragon DB
            TModel dragonTModel = (TModel) universalUnifiedDAO.get(TModel.class, key);
            if (dragonTModel == null) {
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.TModelNotFound", key));
            }

            setRequestOnTOA();

            // Convert Dragon instance into uddi instance
            org.uddi.api_v3.TModel apiTModel = transferObjectAssembler.toUDDITModel(dragonTModel);

            result.getTModel().add(apiTModel);
        }

        return result;
    }

}
