/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * PublicationValidator.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.uddi.v3.validator;

import java.util.List;

import javax.xml.bind.JAXBElement;

import org.ow2.dragon.persistence.util.UDDIIdentifierGenerator;
import org.ow2.dragon.service.uddi.v3.error.ErrorMessage;
import org.ow2.dragon.service.uddi.v3.error.FatalErrorException;
import org.ow2.dragon.service.uddi.v3.error.InvalidKeyPassedException;
import org.ow2.dragon.service.uddi.v3.error.ValueNotAllowedException;
import org.uddi.api_v3.ObjectFactory;
import org.uddi.api_v3.SaveTModel;
import org.uddi.api_v3_porttype.DispositionReportFaultMessage;

/**
 * @author ofabre
 * 
 */
public class PublicationValidator {

    public void validateSaveTModel(SaveTModel body) throws DispositionReportFaultMessage {

        // No null input
        if (body == null)
            throw new FatalErrorException(new ErrorMessage("errors.NullInput"));

        // No null or empty list
        List<org.uddi.api_v3.TModel> entityList = body.getTModel();
        if (entityList == null || entityList.size() == 0)
            throw new ValueNotAllowedException(new ErrorMessage("errors.savetmodel.NoInput"));

        for (org.uddi.api_v3.TModel entity : entityList) {
            validateTModel(entity);
        }
    }

    public void validateTModel(org.uddi.api_v3.TModel tModel) throws DispositionReportFaultMessage {
        // A supplied tModel can't be null
        if (tModel == null)
            throw new ValueNotAllowedException(new ErrorMessage("errors.tmodel.NullInput"));

        String entityKey = tModel.getTModelKey();

        if (entityKey != null) {
            entityKey = entityKey.toLowerCase();
            tModel.setTModelKey(entityKey);
            
            // Validate Key for this registry
            if (!UDDIIdentifierGenerator.isUddiV3ValidId(entityKey)){
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.MalformedKey", entityKey));
            }
        }

        if (tModel.getName() == null)
            throw new ValueNotAllowedException(new ErrorMessage("errors.tmodel.NoName"));

        validateCategoryBag(tModel.getCategoryBag());
        validateIdentifierBag(tModel.getIdentifierBag());

        List<org.uddi.api_v3.OverviewDoc> overviewDocList = tModel.getOverviewDoc();
        if (overviewDocList != null) {
            for (org.uddi.api_v3.OverviewDoc overviewDoc : overviewDocList)
                validateOverviewDoc(overviewDoc);
        }

    }
    
    public void validateCategoryBag(org.uddi.api_v3.CategoryBag categories) throws DispositionReportFaultMessage {
        
        // Category bag is optional
        if (categories == null)
            return;
        
        // If category bag does exist, it must have at least one element
        List<JAXBElement<?>> elems = categories.getContent();
        if (elems == null || elems.size() == 0)
            throw new ValueNotAllowedException(new ErrorMessage("errors.categorybag.NoInput"));
        
        for (JAXBElement<?> elem : elems) {
            validateKeyedReferenceTypes(elem);
        }
    }
    
    public void validateKeyedReferenceTypes(JAXBElement<?> elem) throws DispositionReportFaultMessage {
        if (elem == null || elem.getValue() == null)
            throw new ValueNotAllowedException(new ErrorMessage("errors.keyedreference.NullInput"));
        
        // Keyed reference groups must contain a tModelKey
        if (elem.getValue() instanceof org.uddi.api_v3.KeyedReferenceGroup) {
            org.uddi.api_v3.KeyedReferenceGroup krg = (org.uddi.api_v3.KeyedReferenceGroup)elem.getValue();
            if (krg.getTModelKey() == null || krg.getTModelKey().length() == 0)
                throw new ValueNotAllowedException(new ErrorMessage("errors.keyedreference.NoTModelKey"));
        }
        // Keyed references must contain a tModelKey and keyValue
        else if (elem.getValue() instanceof org.uddi.api_v3.KeyedReference) {
            org.uddi.api_v3.KeyedReference kr = (org.uddi.api_v3.KeyedReference)elem.getValue();
            if (kr.getTModelKey() == null || kr.getTModelKey().length() == 0)
                throw new ValueNotAllowedException(new ErrorMessage("errors.keyedreference.NoTModelKey"));
            
            if (kr.getKeyValue() == null || kr.getKeyValue().length() == 0)
                throw new ValueNotAllowedException(new ErrorMessage("errors.keyedreference.NoKeyValue"));
            
        }
    }
    
    public void validateIdentifierBag(org.uddi.api_v3.IdentifierBag identifiers) throws DispositionReportFaultMessage {
        
        // Identifier bag is optional
        if (identifiers == null)
            return;
        
        // If category bag does exist, it must have at least one element
        List<org.uddi.api_v3.KeyedReference> keyedRefList = identifiers.getKeyedReference();
        if (keyedRefList == null || keyedRefList.size() == 0)
            throw new ValueNotAllowedException(new ErrorMessage("errors.identifierbag.NoInput"));
        
        for (org.uddi.api_v3.KeyedReference keyedRef : keyedRefList) {
            validateKeyedReferenceTypes(new ObjectFactory().createKeyedReference(keyedRef));
        }
    }
    
    public void validateOverviewDoc(org.uddi.api_v3.OverviewDoc overviewDoc) throws DispositionReportFaultMessage {
        // OverviewDoc can't be null
        if (overviewDoc == null)
            throw new ValueNotAllowedException(new ErrorMessage("errors.overviewdoc.NullInput"));
        
        // At least one description or overview URL must be supplied
        List<JAXBElement<?>> elems = overviewDoc.getContent();
        if (elems == null || elems.size() == 0)
            throw new ValueNotAllowedException(new ErrorMessage("errors.overviewdoc.NoDescOrUrl"));
    }

}
