/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * ListOfTechnicalServicesBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.deployment;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.SortCriteria;
import org.ow2.dragon.api.to.deployment.ServiceSearchProperties;
import org.ow2.dragon.api.to.deployment.TechnicalServiceTO;
import org.ow2.dragon.ui.businessdelegate.factory.DragonServiceFactory;
import org.ow2.dragon.ui.utils.StringHelper;
import org.ow2.dragon.util.XMLUtil;
import org.ow2.opensuit.core.error.IError;
import org.ow2.opensuit.core.error.LocalizedError;
import org.ow2.opensuit.core.impl.multiparts.IFileUploadPart;

/**
 * This bean holds a list of technical services.
 * 
 * It is initialized either by loading all technical services from the database,
 * or by searching technical services by name.
 * 
 * <p>
 * This version of the bean performs the search when loading, then lets the
 * framework manage sort and pagination. This design may only be used if you're
 * sure the amount of data is "reasonable" and under control.
 */
public class ListOfTechnicalServicesBean {

    private Logger logger = Logger.getLogger(this.getClass());

    // ==========================================================
    // === Attributes
    // ==========================================================
    private List<TechnicalServiceTO> techServices;

    private String searchCriteria;

    private List<String> searchedProperties;

    private String fileDescriptionURL;

    private String[] serviceIds;

    /*
     * Added By TIAR Abderrahmane We can also search a technical service by tag.
     */
    private String tag;

    /**
     * @return the tag.
     */
    public String getTag() {
        return this.tag;
    }

    /**
     * @param tag
     *            the tag to set
     */
    public void setTag(String tag) {
        this.tag = tag;
    }

    /**
     * @param request
     *            contains the parameter Tag which is associated with the name
     *            of the tag.
     * @return "success"
     * @throws LocalizedError
     */
    public String loadServicesByTag(HttpServletRequest request) throws LocalizedError {
        // We also set the attribute tag, useful to reload the page.
        tag = request.getParameter("tag");
        try {
            techServices = DragonServiceFactory.getInstance().getTechServiceManager()
                    .getTechServicesByTag(tag);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    /**
     * @param request
     *            contains the parameters Tag and serviceId, representing each
     *            the tag name and the technical service we want to remove the
     *            tag from.
     * @return "success"
     * @throws LocalizedError
     */
    public String removeTagFromService(HttpServletRequest request) throws LocalizedError {
        String serviceId = request.getParameter("serviceId");
        tag = request.getParameter("tag");

        try {
            DragonServiceFactory.getInstance().getTechServiceManager().removeTag(tag, serviceId);
            // reload the techServices.
            techServices = DragonServiceFactory.getInstance().getTechServiceManager()
                    .getTechServicesByTag(tag);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        return "success";
    }

    // End of what's been added by TIAR Abderrahmane.

    // ==========================================================
    // === Control Methods
    // ==========================================================

    public String loadAll() throws LocalizedError {
        try {
            techServices = DragonServiceFactory.getInstance().getTechServiceManager()
                    .getAllTechServices(createSortOption(SortCriteria.SERVICE_NAME, true));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    public void reset() {

    }

    public String loadNew() throws LocalizedError {
        try {
            return loadNew(DragonServiceFactory.getInstance().getWSDLImporter()
                    .importServiceDefFile(URI.create(fileDescriptionURL)));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

    }

    public String loadNew(String[] serviceIds) throws LocalizedError {
        try {
            techServices = DragonServiceFactory.getInstance().getTechServiceManager()
                    .loadServicesFromWSDL(serviceIds,
                            createSortOption(SortCriteria.SERVICE_NAME, true));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        String result = "success";
        if (techServices == null || techServices.isEmpty()) {
            result = "no";
        }

        return result;
    }

    public void loadNew(IFileUploadPart file) throws LocalizedError {

        String fileName = extractShortFilename(file);

        try {
            serviceIds = DragonServiceFactory.getInstance().getWSDLImporter().importServiceDefFile(
                    XMLUtil.loadDocument(file.getFileInputStream()), fileName);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    private String extractShortFilename(IFileUploadPart file) {
        String fileName = file.getFileName();
        if (fileName != null) {
            if (fileName.contains("/")) {
                // it's a full file name, so extract only the local part
                String[] parts = fileName.split("/");
                fileName = parts[parts.length - 1];
            } else if (fileName.contains("\\")) {
                // it's a full file name, so extract only the local part
                String[] parts = fileName.split("\\u005C+");
                fileName = parts[parts.length - 1];
            }

        }
        return fileName;
    }

    /**
     * Searches organizations by business domain, location, name, type
     */
    public void search(HttpServletRequest request) throws LocalizedError {
        boolean all = Boolean.parseBoolean(request.getParameter("all"));
        if (all) {
            loadAll();
        } else {
            String searchCriteria = request.getParameter("searchCriteria");
            String searchedPropertiesParam = request.getParameter("searchedProperties");
            List<String> searchedProperties = StringHelper
                    .urlParameterToArray(searchedPropertiesParam);
            List<ServiceSearchProperties> properties = convertSearchedProperties(searchedProperties);
            try {
                techServices = DragonServiceFactory.getInstance().getTechServiceManager()
                        .searchTechService(searchCriteria, properties,
                                createSortOption(SortCriteria.SERVICE_NAME, true));
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
                throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
            }
            if (techServices == null || techServices.size() == 0) {
                LocalizedError error = new LocalizedError("search_tech_serv_no_result");
                // NonLocalizedError eroor = new NonLocalizedError("",)
                error.setType(IError.FUNCTIONAL_ERROR);
                // error.setStackTrace(null);
                throw error;
            }
        }
    }

    private List<ServiceSearchProperties> convertSearchedProperties(List<String> searchedProperties2) {
        List<ServiceSearchProperties> result = new ArrayList<ServiceSearchProperties>();
        for (String prop : searchedProperties2) {
            ServiceSearchProperties propEnum = ServiceSearchProperties.fromString(prop);
            if (propEnum != null) {
                result.add(propEnum);
            }
        }
        return result;
    }

    private RequestOptionsTO createSortOption(SortCriteria sortOn, boolean ascendingly) {
        RequestOptionsTO requestOptionsTO = new RequestOptionsTO();
        requestOptionsTO.setSortAscendingly(ascendingly);
        requestOptionsTO.setSortCriteria(sortOn);
        return requestOptionsTO;
    }

    // ==========================================================
    // === Attributes Getters & Setters
    // ==========================================================

    /**
     * @return the searchCriteria
     */
    public String getSearchCriteria() {
        return searchCriteria;
    }

    public List<TechnicalServiceTO> getTechServices() {
        return techServices;
    }

    public void setTechServices(List<TechnicalServiceTO> techServices) {
        this.techServices = techServices;
    }

    /**
     * @param searchCriteria
     *            the searchCriteria to set
     */
    public void setSearchCriteria(String searchCriteria) {
        this.searchCriteria = searchCriteria;
    }

    /**
     * @return the fileDescriptionURL
     */
    public String getFileDescriptionURL() {
        return fileDescriptionURL;
    }

    /**
     * @param fileDescriptionURL
     *            the fileDescriptionURL to set
     */
    public void setFileDescriptionURL(String fileDescriptionURL) {
        this.fileDescriptionURL = fileDescriptionURL;
    }

    public List<String> getSearchedProperties() {
        return searchedProperties;
    }

    public void setSearchedProperties(List<String> searchedProperties) {
        this.searchedProperties = searchedProperties;
    }

    public String[] getServiceIds() {
        return serviceIds;
    }

    public void setServiceIds(String[] serviceIds) {
        this.serviceIds = serviceIds;
    }

}
