/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * OrganizationBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.organization;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.service.organization.OrganizationException;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.SortCriteria;
import org.ow2.dragon.api.to.common.KeyedRefTO;
import org.ow2.dragon.api.to.organization.OrganizationUnitTO;
import org.ow2.dragon.api.to.organization.PersonTO;
import org.ow2.dragon.api.to.organization.PostTO;
import org.ow2.dragon.ui.businessdelegate.DragonDelegateException;
import org.ow2.dragon.ui.businessdelegate.factory.DragonServiceFactory;
import org.ow2.opensuit.core.error.IError;
import org.ow2.opensuit.core.error.LocalizedError;
import org.ow2.opensuit.core.error.NonLocalizedError;
import org.ow2.opensuit.core.validation.ValidationErrors;

/**
 * 
 * @author ambarthe, ofabre - eBM Websourcing
 * 
 */
public class OrganizationBean extends PartyBean {

    private Logger logger = Logger.getLogger(this.getClass());

    // ==========================================================
    // === Attributes
    // ==========================================================
    private OrganizationUnitTO organization;

    private List<PersonTO> persons;

    private List<PostTO> orgPosts;

    private String name;

    private String type;

    private boolean isLegalEntity;

    private String discoveryURL;

    private String emailAddress;

    private String motherOrganizationId;

    private String streetNumber;

    private String street;

    private String complement;

    private String zipcode;

    private String city;

    private String state;

    private String country;

    private List<KeyedRefTO> categories;

    private List<String> selectedCategories;

    private List<KeyedRefTO> identifiers;

    private List<String> selectedIdentifiers;

    /**
     * Default Constructor
     * 
     */
    public OrganizationBean() {

    }

    public OrganizationBean(OrganizationUnitTO orgTO) throws LocalizedError {
        try {
            organization = orgTO;
            persons = DragonServiceFactory.getInstance().getPersonManager()
                    .getPersonsByOrganization(organization.getId(), null);
            loadOrganizationPosts();
            reset();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        reset();
    }

    // ==========================================================
    // === Control Methods
    // ==========================================================
    /**
     * Loads an organization from the request parameter 'id'.
     */
    @Override
    public void load(HttpServletRequest iRequest) throws LocalizedError {
        String id = iRequest.getParameter("organizationId");
        try {
            organization = DragonServiceFactory.getInstance().getOrganizationManager()
                    .getOrganization(id);
            if (organization != null) {
                loadOrganizationPersons();
                loadOrganizationPosts();
            }
            reset();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    public void loadOrganizationPosts() throws OrganizationException, DragonDelegateException {
        // sort ascendingly on name
        RequestOptionsTO requestOptionsTO = new RequestOptionsTO();
        requestOptionsTO.setSortAscendingly(true);
        requestOptionsTO.setSortCriteria(SortCriteria.POST_NAME);
        orgPosts = DragonServiceFactory.getInstance().getOrganizationManager()
                .getPostsByOrganization(organization.getId(), requestOptionsTO);
    }

    public void loadOrganizationPersons() throws OrganizationException, DragonDelegateException {
        persons = DragonServiceFactory.getInstance().getPersonManager().getPersonsByOrganization(
                organization.getId(), null);
    }

    /**
     * Saves the current organization with changes.
     */
    public String save() throws IError {

        OrganizationUnitTO motherOrganization = null;
        try {
            if (motherOrganizationId != null && !motherOrganizationId.equals("")
                    && !motherOrganizationId.equals("null")) {
                motherOrganization = DragonServiceFactory.getInstance().getOrganizationManager()
                        .getOrganization(motherOrganizationId);
            }
        } catch (OrganizationException e) {
            // No mother org found for the given id, simply log error
            logger.error(e.getMessage(), e);
        } catch (DragonDelegateException e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        OrganizationUnitTO newOrganization = createOrganizationUnitTO(motherOrganization);

        try {
            DragonServiceFactory.getInstance().getOrganizationManager().updateOrganization(
                    newOrganization);
        } catch (OrganizationException e) {
            NonLocalizedError error = new NonLocalizedError(
                    "Can't update the selected organization.", e.getMessage(), e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        } catch (Throwable e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        organization = newOrganization;
        return "success";
    }

    private OrganizationUnitTO createOrganizationUnitTO(OrganizationUnitTO motherOrganization) {
        OrganizationUnitTO newOrganization = new OrganizationUnitTO();
        newOrganization.setId(organization.getId());
        newOrganization.setDiscoveryURL(discoveryURL);
        newOrganization.setEmailAddress(emailAddress);
        newOrganization.setIsLegalEntity(isLegalEntity);
        newOrganization.setMotherOrganization(motherOrganization);
        newOrganization.setName(name);
        newOrganization.setType(type);
        newOrganization.setStreetNumber(streetNumber);
        newOrganization.setStreet(street);
        newOrganization.setComplement(complement);
        newOrganization.setZipcode(zipcode);
        newOrganization.setCity(city);
        newOrganization.setState(state);
        newOrganization.setCountry(country);
        return newOrganization;
    }

    public boolean isPostToAdd(HttpServletRequest iRequest) throws LocalizedError {
        List<PostTO> posts = new ArrayList<PostTO>();
        boolean result = false;
        String orgId = iRequest.getParameter("organizationId");
        try {
            posts = DragonServiceFactory.getInstance().getPostManager()
                    .getPostsNotLinkedToOrganization(orgId, null);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        if (posts != null && !posts.isEmpty()) {
            result = true;
        }
        return result;

    }

    public boolean isPostInPostList(HttpServletRequest request) {
        boolean result = false;
        if (orgPosts != null && !orgPosts.isEmpty()) {
            result = true;
        }
        return result;
    }

    /**
     * Try to retrieve a Person of this Organization which is linked to the Post
     * to delete. If it exist, the post can't be deleted
     * 
     * @param postBean
     * 
     * @return
     * 
     * @throws LocalizedError
     */
    public String canDeletePost(PostTO postTo) throws LocalizedError {
        List<PersonTO> persons = new ArrayList<PersonTO>();
        try {
            persons = DragonServiceFactory.getInstance().getPersonManager().getPersonsByOrgAndPost(
                    organization.getId(), postTo.getId(), null);
        } catch (DragonDelegateException e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        if (!(persons != null && !persons.isEmpty())) {
            return "yes";
        }
        return "no";
    }

    public String deletePost(String postId) throws LocalizedError {
        try {
            DragonServiceFactory.getInstance().getOrganizationManager().removePost(
                    organization.getId(), postId);
            loadOrganizationPosts();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    public String deletePerson(String personId) throws LocalizedError {
        try {
            DragonServiceFactory.getInstance().getPersonManager().removePerson(personId);
            loadOrganizationPersons();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    /**
     * Creates an organization object with current attributes
     */
    public String create() throws IError {
        OrganizationUnitTO motherOrganization = null;
        try {
            if (motherOrganizationId != null && !motherOrganizationId.equals("")
                    && !motherOrganizationId.equals("null")) {
                motherOrganization = DragonServiceFactory.getInstance().getOrganizationManager()
                        .getOrganization(motherOrganizationId);
            }
        } catch (OrganizationException e) {
            // No mother org found for the given id, simply log error
            logger.error(e.getMessage(), e);
        } catch (DragonDelegateException e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        OrganizationUnitTO newOrganization = createOrganizationUnitTO(motherOrganization);

        String orgId = "";
        try {
            orgId = DragonServiceFactory.getInstance().getOrganizationManager().createOrganization(
                    newOrganization);
        } catch (OrganizationException e) {
            NonLocalizedError error = new NonLocalizedError("Can't create organization.", e
                    .getMessage(), e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;

        } catch (Throwable e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        try {
            organization = DragonServiceFactory.getInstance().getOrganizationManager()
                    .getOrganization(orgId);
        } catch (Throwable e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    /**
     * Deletes the current organization from the database
     */
    public String delete() throws LocalizedError {
        try {
            DragonServiceFactory.getInstance().getOrganizationManager().removeOrganization(
                    organization.getId());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        return "success";
    }

    /**
     * Resets changes made to the current organization attributes.
     */
    public void reset() {
        discoveryURL = organization.getDiscoveryURL();
        isLegalEntity = organization.getIsLegalEntity();
        name = organization.getName();
        type = organization.getType();
        if (organization.getMotherOrganization() != null) {
            motherOrganizationId = organization.getMotherOrganization().getId();
        } else {
            motherOrganizationId = "";
        }
        categories = organization.getCategories();
        identifiers = organization.getIdentifiers();
        emailAddress = organization.getEmailAddress();
        streetNumber = organization.getStreetNumber();
        street = organization.getStreet();
        complement = organization.getComplement();
        zipcode = organization.getZipcode();
        city = organization.getCity();
        state = organization.getState();
        country = organization.getCountry();
    }

    /**
     * Clears the organization.
     */
    public void clear() {
        organization = new OrganizationUnitTO();
        reset();
    }

    public void loadCategories() throws LocalizedError {
        try {
            categories = DragonServiceFactory.getInstance().getOrganizationManager()
                    .getCategoriesForOrg(organization.getId());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    public void loadIdentifiers() throws LocalizedError {
        try {
            identifiers = DragonServiceFactory.getInstance().getOrganizationManager()
                    .getIdentifiersForOrg(organization.getId());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    public String removeCategories() throws LocalizedError {
        try {
            DragonServiceFactory.getInstance().getOrganizationManager().removeCategories(
                    organization.getId(), selectedCategories);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        loadCategories();
        return "success";
    }

    public String removeIdentifiers() throws LocalizedError {
        try {
            DragonServiceFactory.getInstance().getOrganizationManager().removeIdentifiers(
                    organization.getId(), selectedIdentifiers);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        loadIdentifiers();
        return "success";
    }

    /**
     * Validate the data put for the organization object
     * 
     * @throws ValidationErrors
     */
    public void validate() throws ValidationErrors {
        // --- no check for now
    }

    /*
     * auto complete sample public List<String> autocompleteOrgName(String
     * value) { if(value == null) return null; value = value.trim();
     * if(value.length() < 1) // --- auto complete for more than 1 character
     * return null; ArrayList<String> completedValues = new ArrayList<String>();
     * // retrieve org beans with the given name try { List<OrganizationUnitTO>
     * organizations =
     * DragonServiceFactory.getInstance().getOrganizationManager()
     * .searchOrganization(value, true, false, false, false, null); } catch
     * (OrganizationException e) {
     * logger.warn("error during org name autocompletion", e); } catch
     * (DragonDelegateException e) {
     * logger.warn("error during org name autocompletion", e); } // add
     * completed values /// ..... return completedValues; }
     */

    // ==========================================================
    // === Attributes Getters & Setters
    // ==========================================================
    @Override
    public String getId() {
        if (organization == null)
            return "";
        return organization.getId();
    }

    /**
     * @return the persons
     */
    public List<PersonTO> getPersons() {
        return persons;
    }

    /**
     * @return the posts
     */
    public List<PostTO> getOrgPosts() {
        return orgPosts;
    }

    /**
     * @return the discoveryURL
     */
    public String getDiscoveryURL() {
        return discoveryURL;
    }

    /**
     * @param discoveryURL
     *            the discoveryURL to set
     */
    public void setDiscoveryURL(String discoveryURL) {
        this.discoveryURL = discoveryURL;
    }

    /**
     * @return the name
     */
    public String getName() {
        return name;
    }

    /**
     * @param name
     *            the name to set
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * @return the type
     */
    public String getType() {
        return type;
    }

    /**
     * @param type
     *            the type to set
     */
    public void setType(String type) {
        this.type = type;
    }

    /**
     * @return the isLegalEntity
     */
    public boolean getIsLegalEntity() {
        return isLegalEntity;
    }

    /**
     * @param isLegalEntity
     *            the isLegalEntity to set
     */
    public void setIsLegalEntity(boolean isLegalEntity) {
        this.isLegalEntity = isLegalEntity;
    }

    /**
     * @return the motherOrganizationId
     */
    public String getMotherOrganizationId() {
        return motherOrganizationId;
    }

    /**
     * @param motherOrganizationId
     *            the motherOrganizationId to set
     */
    public void setMotherOrganizationId(String motherOrganizationId) {
        this.motherOrganizationId = motherOrganizationId;
    }

    /**
     * @return the streetNumber
     */
    public String getStreetNumber() {
        return streetNumber;
    }

    /**
     * @param streetNumber
     *            the streetNumber to set
     */
    public void setStreetNumber(String streetNumber) {
        this.streetNumber = streetNumber;
    }

    /**
     * @return the street
     */
    public String getStreet() {
        return street;
    }

    /**
     * @param street
     *            the street to set
     */
    public void setStreet(String street) {
        this.street = street;
    }

    /**
     * @return the complement
     */
    public String getComplement() {
        return complement;
    }

    /**
     * @param complement
     *            the complement to set
     */
    public void setComplement(String complement) {
        this.complement = complement;
    }

    /**
     * @return the zipcode
     */
    public String getZipcode() {
        return zipcode;
    }

    /**
     * @param zipcode
     *            the zipcode to set
     */
    public void setZipcode(String zipcode) {
        this.zipcode = zipcode;
    }

    /**
     * @return the city
     */
    public String getCity() {
        return city;
    }

    /**
     * @param city
     *            the city to set
     */
    public void setCity(String city) {
        this.city = city;
    }

    /**
     * @return the state
     */
    public String getState() {
        return state;
    }

    /**
     * @param state
     *            the state to set
     */
    public void setState(String state) {
        this.state = state;
    }

    /**
     * @return the country
     */
    public String getCountry() {
        return country;
    }

    /**
     * @param country
     *            the country to set
     */
    public void setCountry(String country) {
        this.country = country;
    }

    public String getEmailAddress() {
        return emailAddress;
    }

    public void setEmailAddress(String emailAddress) {
        this.emailAddress = emailAddress;
    }

    @Override
    public String getGenericName() {
        return this.name;
    }

    @Override
    public String toString() {
        return this.getGenericName();
    }

    public List<KeyedRefTO> getCategories() {
        return categories;
    }

    public void setCategories(List<KeyedRefTO> categories) {
        this.categories = categories;
    }

    public List<String> getSelectedCategories() {
        return selectedCategories;
    }

    public void setSelectedCategories(List<String> selectedCategories) {
        this.selectedCategories = selectedCategories;
    }

    public OrganizationUnitTO getOrganization() {
        return organization;
    }

    public void setOrganization(OrganizationUnitTO organization) {
        this.organization = organization;
    }

    public List<KeyedRefTO> getIdentifiers() {
        return identifiers;
    }

    public void setIdentifiers(List<KeyedRefTO> identifiers) {
        this.identifiers = identifiers;
    }

    public List<String> getSelectedIdentifiers() {
        return selectedIdentifiers;
    }

    public void setSelectedIdentifiers(List<String> selectedIdentifiers) {
        this.selectedIdentifiers = selectedIdentifiers;
    }
}
