/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * ListOfRuntimeManagersBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.technology;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.service.technology.TechnologyException;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.SortCriteria;
import org.ow2.dragon.api.to.technology.ExecEnvManagerSearchProperties;
import org.ow2.dragon.api.to.technology.ExecutionEnvironmentManagerTO;
import org.ow2.dragon.ui.businessdelegate.DragonDelegateException;
import org.ow2.dragon.ui.businessdelegate.factory.DragonServiceFactory;
import org.ow2.dragon.ui.utils.StringHelper;
import org.ow2.opensuit.core.error.IError;
import org.ow2.opensuit.core.error.LocalizedError;

/**
 * @author ofabre - eBM Websourcing
 * 
 */
public class ListOfRuntimeManagersBean {

    private Logger logger = Logger.getLogger(this.getClass());

    private List<ExecutionEnvironmentManagerTO> runtimeManagers;

    private String searchCriteria;

    private List<String> searchedProperties;

    private String runtimeAddress;

    private String newLoadedRuntimeManagerId;

    /**
     * Loads all runtime managers from the database
     */
    public void loadAll() throws LocalizedError {
        try {
            runtimeManagers = DragonServiceFactory.getInstance().getTechnologyManager()
                    .getAllExecutionEnvironmentManagers(
                            createSortOption(SortCriteria.EXEC_ENV_MAN_NAME, true));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    public void search(HttpServletRequest request) throws LocalizedError {
        boolean all = Boolean.parseBoolean(request.getParameter("all"));
        if (all) {
            loadAll();
        } else {
            String searchCriteria = request.getParameter("searchCriteria");
            String searchedPropertiesParam = request.getParameter("searchedProperties");
            List<String> searchedProperties = StringHelper
                    .urlParameterToArray(searchedPropertiesParam);
            List<ExecEnvManagerSearchProperties> properties = convertSearchedProperties(searchedProperties);
            try {
                runtimeManagers = DragonServiceFactory.getInstance().getTechnologyManager()
                        .searchExecutionEnvironmentManagers(searchCriteria, properties,
                                createSortOption(SortCriteria.EXEC_ENV_MAN_NAME, true));
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
                throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
            }

            if (runtimeManagers == null || runtimeManagers.size() == 0) {
                LocalizedError error = new LocalizedError("search_runtime_manager_no_result");
                // NonLocalizedError eroor = new NonLocalizedError("",)
                error.setType(IError.FUNCTIONAL_ERROR);
                // error.setStackTrace(null);
                throw error;
            }
        }
    }

    private List<ExecEnvManagerSearchProperties> convertSearchedProperties(
            List<String> searchedProperties2) {
        List<ExecEnvManagerSearchProperties> result = new ArrayList<ExecEnvManagerSearchProperties>();
        for (String prop : searchedProperties2) {
            ExecEnvManagerSearchProperties propEnum = ExecEnvManagerSearchProperties
                    .fromString(prop);
            if (propEnum != null) {
                result.add(propEnum);
            }
        }
        return result;
    }

    public String loadNew() throws LocalizedError {
        try {
            newLoadedRuntimeManagerId = DragonServiceFactory.getInstance().getTechnologyManager()
                    .addRuntimeManager(runtimeAddress);
        } catch (DragonDelegateException e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        } catch (TechnologyException e) {
            LocalizedError error = new LocalizedError("add_runtime_manager_exception");
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        }
        return "success";
    }

    private RequestOptionsTO createSortOption(SortCriteria sortOn, boolean ascendingly) {
        RequestOptionsTO requestOptionsTO = new RequestOptionsTO();
        requestOptionsTO.setSortAscendingly(ascendingly);
        requestOptionsTO.setSortCriteria(sortOn);
        return requestOptionsTO;
    }

    public String getRuntimeAddress() {
        return runtimeAddress;
    }

    public void setRuntimeAddress(String runtimeAddress) {
        this.runtimeAddress = runtimeAddress;
    }

    public String getNewLoadedRuntimeManagerId() {
        return newLoadedRuntimeManagerId;
    }

    public void setNewLoadedRuntimeManagerId(String newLoadedRuntimeManagerId) {
        this.newLoadedRuntimeManagerId = newLoadedRuntimeManagerId;
    }

    public List<ExecutionEnvironmentManagerTO> getRuntimeManagers() {
        return runtimeManagers;
    }

    public void setRuntimeManagers(List<ExecutionEnvironmentManagerTO> runtimeManagers) {
        this.runtimeManagers = runtimeManagers;
    }

    public String getSearchCriteria() {
        return searchCriteria;
    }

    public void setSearchCriteria(String searchCriteria) {
        this.searchCriteria = searchCriteria;
    }

    public List<String> getSearchedProperties() {
        return searchedProperties;
    }

    public void setSearchedProperties(List<String> searchedProperties) {
        this.searchedProperties = searchedProperties;
    }

}
