/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * EndpointBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.deployment;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.service.organization.OrganizationException;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.common.KeyedRefTO;
import org.ow2.dragon.api.to.deployment.BindingOperationTO;
import org.ow2.dragon.api.to.deployment.CommentTO;
import org.ow2.dragon.api.to.deployment.EndpointTO;
import org.ow2.dragon.api.to.deployment.RatingTO;
import org.ow2.dragon.api.to.deployment.TechnicalServiceTO;
import org.ow2.dragon.api.to.organization.OrgToEndpointTO;
import org.ow2.dragon.api.to.organization.PersonToEndpointTO;
import org.ow2.dragon.api.to.specification.ServiceSpecificationTO;
import org.ow2.dragon.api.to.technology.ExecutionEnvironmentTO;
import org.ow2.dragon.api.to.technology.ProcessorTO;
import org.ow2.dragon.ui.businessdelegate.DragonDelegateException;
import org.ow2.dragon.ui.businessdelegate.factory.DragonServiceFactory;
import org.ow2.dragon.ui.uibeans.organization.OrganizationBean;
import org.ow2.dragon.ui.uibeans.organization.PartyBean;
import org.ow2.dragon.ui.uibeans.organization.PersonBean;
import org.ow2.dragon.util.StringHelper;
import org.ow2.opensuit.core.error.LocalizedError;

/**
 * 
 * @author ofabre - eBM Websourcing
 * 
 */
public class EndpointBean {

    private Logger logger = Logger.getLogger(this.getClass());

    // ==========================================================
    // === Attributes
    // ==========================================================
    private EndpointTO endpoint;

    private List<PersonToEndpointTO> personRoles;

    private List<OrgToEndpointTO> organizationRoles;

    private List<BindingOperationTO> operations;

    private ProcessorTO hostProcessor;

    private ExecutionEnvironmentTO hostExecEnv;

    private TechnicalServiceTO service;

    private String bindingId;

    private String name;

    private String networkAddress;

    private List<KeyedRefTO> categories;

    private List<String> selectedCategories;

    private List<ServiceSpecificationTO> specsTO;

    // Added By TIAR Abderrahmane

    // The technical service rating of the user connected.
    private int myRating;

    // The global rating of the technical service.
    private float globalRating;

    // Tags are listed as a hashSet.
    private Set<String> tags = new HashSet<String>();

    // All the comments of the technical service.
    private List<CommentTO> comments;

    // This String will be useful to get an eventual comment to add.
    private String newComment;

    // ////////////////////////////////////////////////////
    // setters and getters for the Community part
    // ////////////////////////////////////////////////////
    /**
     * @param rating
     *            the user rating to set for this service.
     */
    public void setMyRating(int rating) {
        this.myRating = rating;
    }

    /**
     * @return the user rating of this service.
     */
    public int getMyRating() {
        return this.myRating;
    }

    /**
     * @param globalRating
     *            the global rating to set for this service.
     */
    public void setGlobalRating(float globalRating) {
        this.globalRating = globalRating;
    }

    /**
     * @param globalRating
     *            a String representing the global rating to set.
     */
    public void setGlobalRating(String globalRating) {
        this.globalRating = Float.parseFloat(globalRating);
    }

    /**
     * @return the global rating in a String format.
     */
    public String getGlobalRating() {
        String toPrint = Float.toString(this.globalRating);
        // We need to print only one digit after the decimal.
        return toPrint.substring(0, 3);
    }

    /**
     * @param tags
     *            a String representing the new tags to add ("tag1, tag2...")
     */
    // the tags follows the shape: tag1, tag2 ....
    public void setTags(String tags) {
        // Remove all spaces.
        String[] pieces = tags.split(",");

        for (int i = 0; i <= pieces.length - 1; i++) {
            // No need to add white spaces as tags.
            pieces[i] = pieces[i].trim();
            if (!StringHelper.isNullOrEmpty(pieces[i]))
                this.tags.add(pieces[i]);
        }
    }

    /**
     * @param tags
     *            tags to set.
     */
    public void setTags(Set<String> tags) {
        this.tags = tags;
    }

    /**
     * @return technical service tags in a string format.
     */
    public String getTags() {
        return this.tags.toString();
    }

    /**
     * @return the new comment to add.
     */
    public String getNewComment() {
        return newComment;
    }

    /**
     * @param newComment
     *            the new comment to set.
     */
    public void setNewComment(String newComment) {
        this.newComment = newComment;
    }

    /**
     * @return the list of this technical service comments
     */
    public List<CommentTO> getComments() {
        return this.comments;
    }

    /**
     * @param the
     *            list of comments to set to this service.
     */
    public void setComments(List<CommentTO> comments) {
        this.comments = comments;
    }

    /**
     * This function adds a new comment to the technical service.
     * 
     * @param user
     * @return "success"
     * @throws LocalizedError
     */
    public String addComment(String user) throws LocalizedError {
        String cmtTemp = newComment.trim();
        // no need to add a white space comment.
        if (!StringHelper.isNullOrEmpty(cmtTemp)) {
            // Creating a new Transfer Object comment.
            CommentTO newCommentTO = new CommentTO();
            newCommentTO.setContent(newComment);
            newCommentTO.setUserName(user);
            newCommentTO.setDate(new Date());

            // Try saving the comment in the Technical Service
            try {
                DragonServiceFactory.getInstance().getEndpointManager().updateComments(
                        endpoint.getId(), newCommentTO);

                /*
                 * We Need the id of the comment that has just been saved.
                 * Useful if the Admin wants to delete the new comment without
                 * reloading the page to get the id.
                 */
                final EndpointTO endpointTemp = DragonServiceFactory.getInstance()
                        .getEndpointManager().getEndpoint(endpoint.getId(), null);
                // set this comments.
                comments = endpointTemp.getComments();
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
                throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
            }
        }
        // Setting the new Comment to an empty string
        newComment = "";
        return "success";
    }

    /**
     * remove a comment from the technical service.
     * 
     * @param request
     *            contains the id of the comment to remove.
     * @return "success"
     * @throws LocalizedError
     */
    public String removeComment(HttpServletRequest request) throws LocalizedError {
        // get the id of the comment.
        String id = request.getParameter("comment");

        try {
            DragonServiceFactory.getInstance().getEndpointManager().removeComment(endpoint.getId(),
                    id);

            // getting the new list of comments after the removal.
            final EndpointTO endpointTemp = DragonServiceFactory.getInstance().getEndpointManager()
                    .getEndpoint(endpoint.getId(), null);
            // set this comments.
            comments = endpointTemp.getComments();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    // Have a look at load, save, create, reset
    // End of what's been added By TIAR Abderrahmane

    // ==========================================================
    // === Control Methods
    // ==========================================================
    /**
     * Loads an endpoint from the request parameter 'id'.
     */
    public void load(HttpServletRequest iRequest) throws LocalizedError {
        String id = iRequest.getParameter("endpointId");

        // We add the userName for the ratings;
        String user = iRequest.getUserPrincipal().getName();

        try {
            endpoint = DragonServiceFactory.getInstance().getEndpointManager()
                    .getEndpoint(id, user);
            loadOrgRoles();
            loadPersonRoles();
            loadServiceSpecs();
            loadHostProcessor();
            loadHostExecEnv();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        reset();
    }

    private void loadHostProcessor() throws LocalizedError {
        if (endpoint.getHostProcessorId() != null) {
            try {
                hostProcessor = DragonServiceFactory.getInstance().getTechnologyManager()
                        .getProcessor(endpoint.getHostProcessorId());
            } catch (DragonDelegateException e) {
                logger.error(e.getMessage(), e);
                throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
            }
        }
    }

    private void loadHostExecEnv() throws LocalizedError {
        if (endpoint.getHostExecEnvId() != null) {
            try {
                hostExecEnv = DragonServiceFactory.getInstance().getTechnologyManager()
                        .getExecutionEnvironment(endpoint.getHostExecEnvId());
            } catch (DragonDelegateException e) {
                logger.error(e.getMessage(), e);
                throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
            }
        }
    }

    public void loadServiceSpecs() throws LocalizedError {
        try {
            List<ServiceSpecificationTO> result = DragonServiceFactory.getInstance()
                    .getWSDLImporter().getServSpecsForEndpoint(endpoint.getId());

            if (result != null) {
                for (ServiceSpecificationTO serviceSpecificationTO : result) {
                    // Fill file name if empty
                    if (serviceSpecificationTO.getFileName() == null) {
                        serviceSpecificationTO.setFileName("wsdlFile.wsdl");
                    }
                }
            }

            this.specsTO = result;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    public void loadWSDLDesc(HttpServletResponse response, String serviceSpecId)
            throws LocalizedError, IOException {

        InputStream input = null;
        try {
            input = DragonServiceFactory.getInstance().getWSDLImporter().getWsdlDesc(serviceSpecId);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        OutputStream output = response.getOutputStream();

        response.setContentType("text/xml");
        response.setHeader("Cache-Control", "no-cache"); // HTTP 1.1
        response.setHeader("Pragma", "no-cache"); // HTTP 1.0
        response.setDateHeader("Expires", 0); // prevents caching at the proxy

        byte[] buffer = new byte[512];
        int len = 0;
        while ((len = input.read(buffer)) >= 0) {
            output.write(buffer, 0, len);
        }
        output.flush();
        output.close();
    }

    /**
     * Saves the current endpoint with changes.
     */
    public String save() throws LocalizedError {
        EndpointTO newEndpoint = new EndpointTO();
        newEndpoint.setId(endpoint.getId());
        newEndpoint.setName(name);
        newEndpoint.setNetworkAddress(networkAddress);
        newEndpoint.setTechService(service);
        newEndpoint.setProtocols(endpoint.getProtocols());
        newEndpoint.setTransports(endpoint.getTransports());

        // P.S: The Binding should also be set. for the moment, we
        // just check if it's null in the function Reset()

        // Added By TIAR Abderrahmane
        // Adding to newtechService the comments, tags and rating.
        RatingTO ratingTO = new RatingTO();
        ratingTO.setMyRating(myRating);
        ratingTO.setUserName(endpoint.getRatingTO().getUserName());
        ratingTO.setId(endpoint.getRatingTO().getid());
        newEndpoint.setRatingTO(ratingTO);
        newEndpoint.setTags(tags);

        newEndpoint.setComments(comments);
        // End

        try {
            DragonServiceFactory.getInstance().getEndpointManager().updateEndpoint(newEndpoint);

            // getting the new global rating after an eventual rating
            globalRating = DragonServiceFactory.getInstance().getEndpointManager()
                    .updateGlobalRating(newEndpoint.getId());
            newEndpoint.setGlobalRating(globalRating);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        endpoint = newEndpoint;
        return "success";
    }

    /**
     * Creates an endpoint object with current attributes
     */
    public String create() throws LocalizedError {
        EndpointTO newEndpoint = new EndpointTO();
        newEndpoint.setName(name);
        newEndpoint.setNetworkAddress(networkAddress);
        newEndpoint.setTechService(service);

        String endpointId = "";

        // Added By TIAR Abderrahmane
        RatingTO rating = new RatingTO();
        rating.setMyRating(myRating);
        rating.setUserName(endpoint.getRatingTO().getUserName());
        rating.setId("");
        newEndpoint.setRatingTO(rating);

        newEndpoint.setGlobalRating(globalRating);
        newEndpoint.setTags(tags);

        newEndpoint.setComments(comments);
        // END

        try {
            endpointId = DragonServiceFactory.getInstance().getEndpointManager().createEndpoint(
                    newEndpoint);

            // For the rating
            endpoint = DragonServiceFactory.getInstance().getEndpointManager().getEndpoint(
                    endpointId, newEndpoint.getRatingTO().getUserName());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        return "success";
    }

    /**
     * Deletes the current endpoint from the database
     */
    public String delete() throws LocalizedError {
        try {
            DragonServiceFactory.getInstance().getEndpointManager()
                    .removeEndpoint(endpoint.getId());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    /**
     * Resets changes made to the current endpoint attributes.
     */
    public void reset() {
        name = endpoint.getName();
        networkAddress = endpoint.getNetworkAddress();
        service = endpoint.getTechService();
        if (endpoint.getBinding() != null) {
            bindingId = endpoint.getBinding().getId();
            operations = endpoint.getBinding().getBindingOps();
        }
        categories = endpoint.getCategories();

        // Added By TIAR Abderrahmane
        myRating = endpoint.getRatingTO().getMyRating();
        globalRating = endpoint.getGlobalRating();

        comments = endpoint.getComments();
        tags = endpoint.getTags();
        // End
    }

    /**
     * Clears the endpoint.
     */
    public void clear() {
        endpoint = new EndpointTO();
        reset();
    }

    public List<String> getEndpointProvidersString() throws OrganizationException,
            DragonDelegateException, LocalizedError {
        List<String> res = new ArrayList<String>();
        for (PartyBean p : getEndpointProviders()) {
            res.add(p.getGenericName());
        }
        if (res.size() == 0) {
            res.add("anonymous_provider");
        }
        return res;
    }

    public List<PartyBean> getEndpointProviders() throws OrganizationException,
            DragonDelegateException, LocalizedError {
        List<PartyBean> res = new ArrayList<PartyBean>();
        loadPersonRoles();
        loadOrgRoles();
        if (personRoles != null) {
            for (PersonToEndpointTO personRole : personRoles) {
                if (personRole.getType().equals("Manager")) {
                    res.add(new PersonBean(personRole.getPersonTO()));
                }
            }
        }
        if (organizationRoles != null) {
            for (OrgToEndpointTO organizationRole : organizationRoles) {
                if (organizationRole.getType().equals("Provider")) {
                    res.add(new OrganizationBean(organizationRole.getOrganizationUnitTO()));
                }
            }
        }
        return res;
    }

    public void loadCategories() throws LocalizedError {
        try {
            categories = DragonServiceFactory.getInstance().getEndpointManager()
                    .getCategoriesForEndpoint(endpoint.getId());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    public String removeCategories() throws LocalizedError {
        try {
            DragonServiceFactory.getInstance().getEndpointManager().removeCategories(
                    endpoint.getId(), selectedCategories);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        loadCategories();
        return "success";
    }

    public String authorizeSLACreation(String type) {
        String res = null;

        if ((type.equals("Consumer")) || (type.equals("User"))) {
            res = "Create or Submit Agreement Offer";
        } else if ((type.equals("Provider")) || (type.equals("Manager"))) {
            res = "Create Agreement Template or Accept Offer";
        }

        return res;
    }

    public void loadPersonRoles() throws OrganizationException, DragonDelegateException {
        personRoles = DragonServiceFactory.getInstance().getRoleOfPartyManager()
                .getPersonRolesByEndpoint(endpoint.getId(), createSortOption("type", true));
    }

    public void loadOrgRoles() throws OrganizationException, DragonDelegateException {
        organizationRoles = DragonServiceFactory.getInstance().getRoleOfPartyManager()
                .getOrgRolesByEndpoint(endpoint.getId(), createSortOption("type", true));
    }

    /**
     * Delete a role of party
     * 
     * @return
     * @throws LocalizedError
     */
    public String deleteRole(HttpServletRequest request) throws LocalizedError {
        String roleToDeleteId = request.getParameter("roleToDelete");
        try {
            DragonServiceFactory.getInstance().getRoleOfPartyManager().removeRole(roleToDeleteId);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        load(request);
        return "success";
    }

    private RequestOptionsTO createSortOption(String sortOn, boolean ascendingly) {
        RequestOptionsTO requestOptionsTO = new RequestOptionsTO();
        requestOptionsTO.setSortAscendingly(ascendingly);
        requestOptionsTO.setSortCriteria(sortOn);
        return requestOptionsTO;
    }

    // ==========================================================
    // === Attributes Getters & Setters
    // ==========================================================
    public String getId() {
        if (endpoint == null)
            return "";
        return endpoint.getId();
    }

    /**
     * @return the endpoint
     */
    public EndpointTO getEndpoint() {
        return endpoint;
    }

    /**
     * @param endpoint
     *            the endpoint to set
     */
    public void setEndpoint(EndpointTO endpoint) {
        this.endpoint = endpoint;
    }

    /**
     * @return the bindingId
     */
    public String getBindingId() {
        return bindingId;
    }

    /**
     * @param bindingId
     *            the bindingId to set
     */
    public void setBindingId(String bindingId) {
        this.bindingId = bindingId;
    }

    /**
     * @return the name
     */
    public String getName() {
        return name;
    }

    /**
     * @param name
     *            the name to set
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * @return the networkAddress
     */
    public String getNetworkAddress() {
        return networkAddress;
    }

    /**
     * @param networkAddress
     *            the networkAddress to set
     */
    public void setNetworkAddress(String networkAddress) {
        this.networkAddress = networkAddress;
    }

    /**
     * @return the protocol
     */
    public String getProtocol() {
        return endpoint.getProtocol();
    }

    public List<PersonToEndpointTO> getPersonRoles() {
        return personRoles;
    }

    public void setPersonRoles(List<PersonToEndpointTO> personRoles) {
        this.personRoles = personRoles;
    }

    public List<OrgToEndpointTO> getOrganizationRoles() {
        return organizationRoles;
    }

    public void setOrganizationRoles(List<OrgToEndpointTO> organizationRoles) {
        this.organizationRoles = organizationRoles;
    }

    public List<BindingOperationTO> getOperations() {
        return operations;
    }

    public void setOperations(List<BindingOperationTO> operations) {
        this.operations = operations;
    }

    public List<ServiceSpecificationTO> getSpecsTO() {
        return specsTO;
    }

    public void setSpecsTO(List<ServiceSpecificationTO> specsTO) {
        this.specsTO = specsTO;
    }

    public ProcessorTO getHostProcessor() {
        return hostProcessor;
    }

    public void setHostProcessor(ProcessorTO hostProcessor) {
        this.hostProcessor = hostProcessor;
    }

    public ExecutionEnvironmentTO getHostExecEnv() {
        return hostExecEnv;
    }

    public void setHostExecEnv(ExecutionEnvironmentTO hostExecEnv) {
        this.hostExecEnv = hostExecEnv;
    }

    public TechnicalServiceTO getService() {
        return service;
    }

    public void setService(TechnicalServiceTO service) {
        this.service = service;
    }

    public List<KeyedRefTO> getCategories() {
        return categories;
    }

    public void setCategories(List<KeyedRefTO> categories) {
        this.categories = categories;
    }

    public List<String> getSelectedCategories() {
        return selectedCategories;
    }

    public void setSelectedCategories(List<String> selectedCategories) {
        this.selectedCategories = selectedCategories;
    }

}
