package org.ow2.dragon.ui.bricks;

/**
 * Open SUIT - Simple User Interface Toolkit
 * 
 * Copyright (c) 2009 France Telecom, http://www.francetelecom.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
import java.io.PrintWriter;
import java.lang.reflect.Type;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.ow2.opensuit.core.util.ReflectionHelper;
import org.ow2.opensuit.core.util.ReflectionHelper.NumberOutOfBoundsException;
import org.ow2.opensuit.core.validation.LocalizedValidationError;
import org.ow2.opensuit.core.validation.ValidationError;
import org.ow2.opensuit.xml.base.html.form.FormRenderingContext;
import org.ow2.opensuit.xml.base.html.form.FormSubmitContext;
import org.ow2.opensuit.xml.base.html.form.field.BaseSingleValueField;
import org.ow2.opensuit.xmlmap.annotations.XmlDoc;
import org.ow2.opensuit.xmlmap.annotations.XmlElement;
import org.ow2.opensuit.xmlmap.interfaces.IInitializable;
import org.ow2.opensuit.xmlmap.interfaces.IInitializationSupport;
import org.ow2.opensuit.xmlmap.interfaces.IInstantiationContext;


@XmlDoc("A global rating editor form field for floats.<br/>" +
        "Supported value types are: any number type or class.")
@XmlElement
public class GlobalRatingField extends BaseSingleValueField implements IInitializable
{
    private Class<?> bindingType;
    
    public void initialize(IInitializationSupport initSupport, IInstantiationContext instContext)
    {
        super.initialize(initSupport, instContext);
        
        if(initSupport.initialize(bind))
        {
            bindingType = bind.getType();
            
            // --- check binding type
            if(!bind.isConvertible(String.class))
                initSupport.addValidationMessage(this, "Bind", IInitializationSupport.ERROR, "Expression "+bind+" has invalid type ("+bind.getType()+"): int, Integer or BigInteger expected.");
        }
    }
    public Type getValueGenericType()
    {
        return getValueType();
    }
    public Class<?> getValueType()
    {
        if(bindingType != null)
            return bindingType;
        return String.class;
    }

    protected Object getValueFromRequest(HttpServletRequest iRequest, FormSubmitContext submitContext) throws ValidationError
    {
        String val = submitContext.getSubmitValue(getFieldName());
        try
        {
            return ReflectionHelper.parse2Type(val, bind.getType());
        }
        catch(NumberFormatException e)
        {
            throw new LocalizedValidationError("validation.int.not_an_int", "opensuit");
        }
        catch(NumberOutOfBoundsException e)
        {
            throw new LocalizedValidationError("validation.int.out_of_bounds", "opensuit", new Object[]{e.getMin(), e.getMax()});
        }
    }
    public void preRender(HttpServletRequest iRequest) throws Exception
    {
    }
    protected void renderInput(HttpServletRequest iRequest, HttpServletResponse iResponse, FormRenderingContext iContext) throws Exception
    {
        PrintWriter iWriter = iResponse.getWriter();
        boolean editable = iContext.isEditable() && isEditable(iRequest);
        
        String val = null;
        if(editable && iContext.getSubmitContext() != null && iContext.getSubmitContext().isSubmitAction())
        {
            // --- after a submit: retrieve value from the request
            val = iContext.getSubmitContext().getSubmitValue(getFieldName());
        }
        else
        {
            // --- first display: retrieve value from the bean
            val = String.valueOf(bind.invoke(iRequest));
        }
        if(val == null)
            // --- initialization on empty
            val = "";
        
        // The id of the bean used. 
        String id = bind.getID() ;
        // the width of the stars to show depending on the value.
        int nbPixels =  (int)(17*Float.parseFloat(val)) ; // a star is 17 pixels width.

        iWriter.println("<div id=\"star\">") ;
        iWriter.println("<ul id=\"star0\" class=\"star\">");
        iWriter.println("<li id=\"starCur0\" class=\"curr\" title=\""+val+"\" style=\"width: "+nbPixels+"px;\"></li>"); 
        iWriter.println("</ul>");
        if (nbPixels > 0) {
            iWriter.println("<div id=\"starUser0\" class=\"user\">"+val+"/5</div>");
        } else {
            iWriter.println("<div id=\"starUser0\" class=\"user\">No rates.</div>");
        }
        iWriter.println("<input id=\"main_"+id+"\" type=\"hidden\" value=\""+val+"\" name=\""+id+"\"/>") ;
        iWriter.println("</div>");
    }
}