/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * ListOfEndpointsBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.deployment;

import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.SortCriteria;
import org.ow2.dragon.api.to.deployment.EndpointTO;
import org.ow2.dragon.ui.businessdelegate.factory.DragonServiceFactory;
import org.ow2.opensuit.core.error.LocalizedError;

/**
 * This bean holds a list of endpoints.
 * 
 * It is initialized either by loading all technical services from the database,
 * or by searching endpoints by name or ever.
 * 
 * <p>
 * This version of the bean performs the search when loading, then lets the
 * framework manage sort and pagination. This design may only be used if you're
 * sure the amount of data is "reasonable" and under control.
 */
public class ListOfEndpointsBean {

    private Logger logger = Logger.getLogger(this.getClass());

    // ==========================================================
    // === Attributes
    // ==========================================================
    private List<EndpointTO> endpoints;

    private String searchCriteria;

    private boolean isSearchOnName;

    private boolean isSearchOnService;

    private boolean isSearchOnProvider;

    private String serviceId;

    /*
     * Added By TIAR Abderrahmane We can also search an endpoint by tag.
     */
    private String tag;

    /**
     * @return the tag.
     */
    public String getTag() {
        return this.tag;
    }

    /**
     * @param tag
     *            the tag to set
     */
    public void setTag(String tag) {
        this.tag = tag;
    }

    /**
     * @return the list of the endpoints after the search.
     */
    public List<EndpointTO> getEndpoints() {
        return this.endpoints;
    }

    /**
     * @param request
     *            contains the parameter Tag which is associated with the name
     *            of the tag.
     * @return "success"
     * @throws LocalizedError
     */
    public String loadEndpointsByTag(HttpServletRequest request) throws LocalizedError {
        // We also set the attribute tag, useful to reload the page.
        tag = request.getParameter("tag");
        try {
            endpoints = DragonServiceFactory.getInstance().getEndpointManager().getEndpointsByTag(
                    tag);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    /**
     * @param request
     *            contains the parameters Tag and endpointId, representing each
     *            the tag name and the endpoint we want to remove the tag from.
     * @return "success"
     * @throws LocalizedError
     */
    public String removeTagFromEndpoint(HttpServletRequest request) throws LocalizedError {
        String endpointId = request.getParameter("endpointId");
        tag = request.getParameter("tag");

        try {
            DragonServiceFactory.getInstance().getEndpointManager().removeTag(tag, endpointId);
            // reload the techServices.
            endpoints = DragonServiceFactory.getInstance().getEndpointManager().getEndpointsByTag(
                    tag);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        return "success";
    }

    // End of what's been added by TIAR Abderrahmane.

    // ==========================================================
    // === Control Methods
    // ==========================================================
    /**
     * Loads all organizations from the database
     */
    public void loadAll() throws LocalizedError {
        try {
            endpoints = DragonServiceFactory.getInstance().getEndpointManager().getAllEndpoints(
                    createSortOption(SortCriteria.ENDPOINT_NAME, true));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    // Load the endpoints corresponding to a service
    // Mock
    public List<EndpointTO> loadByTechService() throws LocalizedError {

        try {
            endpoints = DragonServiceFactory.getInstance().getEndpointManager()
                    .getEndpointsByTechService(serviceId,
                            createSortOption(SortCriteria.ENDPOINT_NAME, true));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return endpoints;

    }

    private RequestOptionsTO createSortOption(SortCriteria sortOn, boolean ascendingly) {
        RequestOptionsTO requestOptionsTO = new RequestOptionsTO();
        requestOptionsTO.setSortAscendingly(ascendingly);
        requestOptionsTO.setSortCriteria(sortOn);
        return requestOptionsTO;
    }

    // ==========================================================
    // === Attributes Getters & Setters
    // ==========================================================

    public String getSearchCriteria() {
        return searchCriteria;
    }

    public void setSearchCriteria(String searchCriteria) {
        this.searchCriteria = searchCriteria;
    }

    public boolean isSearchOnName() {
        return isSearchOnName;
    }

    public void setSearchOnName(boolean isSearchOnName) {
        this.isSearchOnName = isSearchOnName;
    }

    public boolean isSearchOnService() {
        return isSearchOnService;
    }

    public void setSearchOnService(boolean isSearchOnService) {
        this.isSearchOnService = isSearchOnService;
    }

    public boolean isSearchOnProvider() {
        return isSearchOnProvider;
    }

    public void setSearchOnProvider(boolean isSearchOnProvider) {
        this.isSearchOnProvider = isSearchOnProvider;
    }

}
