/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * ListOfOrganizationsBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.organization;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.SortCriteria;
import org.ow2.dragon.api.to.organization.OrganizationSearchProperties;
import org.ow2.dragon.api.to.organization.OrganizationUnitTO;
import org.ow2.dragon.ui.businessdelegate.DragonDelegateException;
import org.ow2.dragon.ui.businessdelegate.factory.DragonServiceFactory;
import org.ow2.dragon.ui.utils.StringHelper;
import org.ow2.opensuit.core.error.IError;
import org.ow2.opensuit.core.error.LocalizedError;

/**
 * 
 * @author ambarthe, ofabre - eBM Websourcing
 * 
 */
public class ListOfOrganizationsBean {

    private Logger logger = Logger.getLogger(this.getClass());

    // ==========================================================
    // === Attributes
    // ==========================================================
    private List<OrganizationUnitTO> organizations;

    private String searchCriteria;

    private List<String> searchedProperties;

    // ==========================================================
    // === Control Methods
    // ==========================================================
    /**
     * Loads all organizations from the database
     */
    public void loadAll() throws LocalizedError {
        try {
            organizations = DragonServiceFactory.getInstance().getOrganizationManager()
                    .getAllOrganizations(createSortOption(SortCriteria.PARTY_NAME, true));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

    }

    public void loadAllWithoutMeAndMyChildren(HttpServletRequest request) throws LocalizedError {
        String initialOrg = request.getParameter("organizationId");

        try {
            organizations = DragonServiceFactory.getInstance().getOrganizationManager()
                    .getAllOrgsWithoutMeAndMyChildren(initialOrg);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

    }

    public boolean isOrganizationInDB() throws LocalizedError {
        List<OrganizationUnitTO> organizationsInDB = null;
        try {
            organizationsInDB = DragonServiceFactory.getInstance().getOrganizationManager()
                    .getAllOrganizations(null);
        } catch (DragonDelegateException e) {
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        if (organizationsInDB == null || organizationsInDB.size() == 0) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Searches organizations by business domain, location, name, type
     */
    public void search(HttpServletRequest request) throws LocalizedError {
        boolean all = Boolean.parseBoolean(request.getParameter("all"));
        if (all) {
            loadAll();
        } else {
            String searchCriteria = request.getParameter("searchCriteria");
            String searchedPropertiesParam = request.getParameter("searchedProperties");
            List<String> searchedProperties = StringHelper
                    .urlParameterToArray(searchedPropertiesParam);
            List<OrganizationSearchProperties> properties = convertSearchedProperties(searchedProperties);
            try {
                organizations = DragonServiceFactory.getInstance().getOrganizationManager()
                        .searchOrganization(searchCriteria, properties,
                                createSortOption(SortCriteria.PARTY_NAME, true));
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
                throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
            }

            if (organizations == null || organizations.size() == 0) {
                LocalizedError error = new LocalizedError("search_org_no_result");
                // NonLocalizedError eroor = new NonLocalizedError("",)
                error.setType(IError.FUNCTIONAL_ERROR);
                // error.setStackTrace(null);
                throw error;
            }
        }
    }

    private List<OrganizationSearchProperties> convertSearchedProperties(
            List<String> searchedProperties2) {
        List<OrganizationSearchProperties> result = new ArrayList<OrganizationSearchProperties>();
        for (String prop : searchedProperties2) {
            OrganizationSearchProperties propEnum = OrganizationSearchProperties.fromString(prop);
            if (propEnum != null) {
                result.add(propEnum);
            }
        }
        return result;
    }

    private RequestOptionsTO createSortOption(SortCriteria sortOn, boolean ascendingly) {
        RequestOptionsTO requestOptionsTO = new RequestOptionsTO();
        requestOptionsTO.setSortAscendingly(ascendingly);
        requestOptionsTO.setSortCriteria(sortOn);
        return requestOptionsTO;
    }

    public List<OrganizationUnitTO> getMotherOrganizations() {
        List<OrganizationUnitTO> result = new ArrayList<OrganizationUnitTO>();
        OrganizationUnitTO nullOrganization = new OrganizationUnitTO();
        nullOrganization.setId(null);
        nullOrganization.setName("No mother organization");
        if (organizations != null) {
            result.add(nullOrganization);
            result.addAll(organizations);
        } else {
            result.add(nullOrganization);
        }
        return result;
    }

    public List<OrganizationUnitTO> getOrganizationsForPerson() {
        List<OrganizationUnitTO> result = new ArrayList<OrganizationUnitTO>();
        OrganizationUnitTO nullOrganization = new OrganizationUnitTO();
        nullOrganization.setId("");
        nullOrganization.setName("No organization selected");
        if (organizations != null) {
            result.add(nullOrganization);
            result.addAll(organizations);
        } else {
            result.add(nullOrganization);
        }
        return result;
    }

    // ==========================================================
    // === Attributes Getters & Setters
    // ==========================================================

    /**
     * @return the searchCriteria
     */
    public String getSearchCriteria() {
        return searchCriteria;
    }

    /**
     * @param searchCriteria
     *            the searchCriteria to set
     */
    public void setSearchCriteria(String searchCriteria) {
        this.searchCriteria = searchCriteria;
    }

    public List<OrganizationUnitTO> getOrganizations() {
        return organizations;
    }

    public List<String> getSearchedProperties() {
        return searchedProperties;
    }

    public void setSearchedProperties(List<String> searchedProperties) {
        this.searchedProperties = searchedProperties;
    }
}
