/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * PersonBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.organization;

import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.service.organization.OrganizationException;
import org.ow2.dragon.api.to.organization.OrganizationUnitTO;
import org.ow2.dragon.api.to.organization.PersonTO;
import org.ow2.dragon.api.to.organization.PostTO;
import org.ow2.dragon.ui.businessdelegate.DragonDelegateException;
import org.ow2.dragon.ui.businessdelegate.factory.DragonServiceFactory;
import org.ow2.opensuit.core.error.IError;
import org.ow2.opensuit.core.error.LocalizedError;
import org.ow2.opensuit.core.error.NonLocalizedError;
import org.ow2.opensuit.core.validation.ValidationErrors;

/**
 * This is a UI bean that represents a person.
 * 
 * @author ambarthe, ofabre - eBMWebsourcing
 */
public class PersonBean extends PartyBean {

    private Logger logger = Logger.getLogger(this.getClass());

    // ==========================================================
    // === Attributes
    // ==========================================================
    private PersonTO person;

    private String organizationId;

    private String postId;

    private String title;

    private String firstname;

    private String middlename;

    private String lastname;

    private String email;

    private String phone;

    private String localization;

    private boolean contact;

    /**
     * Default Constructor
     * 
     */
    public PersonBean() {

    }

    public PersonBean(PersonTO p) {
        person = p;
        reset();
    }

    // ==========================================================
    // === Control Methods
    // ==========================================================
    /**
     * Loads a person from the request parameter 'id'.
     */
    @Override
    public void load(HttpServletRequest iRequest) throws LocalizedError {
        String id = iRequest.getParameter("personId");
        try {
            person = DragonServiceFactory.getInstance().getPersonManager().getPerson(id);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        reset();
    }

    /**
     * Loads a person from the request parameter 'id'.
     */
    public String loadNew(HttpServletRequest iRequest) throws LocalizedError {
        String id = iRequest.getParameter("organizationId");
        try {
            OrganizationUnitTO organization = DragonServiceFactory.getInstance()
                    .getOrganizationManager().getOrganization(id);
            person = new PersonTO();
            person.setOrganization(organization);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        reset();
        return "success";
    }

    /**
     * Saves the current person with changes.
     */
    public String save() throws IError {
        OrganizationUnitTO organization = null;
        try {
            organization = DragonServiceFactory.getInstance().getOrganizationManager()
                    .getOrganization(organizationId);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        PostTO post = null;
        try {
            post = DragonServiceFactory.getInstance().getPostManager().getPost(postId);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        PersonTO newPerson = new PersonTO();
        newPerson.setId(person.getId());
        newPerson.setOrganization(organization);
        newPerson.setEmail(email);
        newPerson.setTitle(title);
        newPerson.setFirstname(firstname);
        newPerson.setLastname(lastname);
        newPerson.setLocalization(localization);
        newPerson.setMiddlename(middlename);
        newPerson.setPhone(phone);
        newPerson.setPost(post);
        newPerson.setContact(contact);

        try {
            DragonServiceFactory.getInstance().getPersonManager().updatePerson(newPerson);
        } catch (OrganizationException e) {
            NonLocalizedError error = new NonLocalizedError("Can't create person", e.getMessage(),
                    e);
            error.setType(IError.FUNCTIONAL_ERROR);
            // error.setNextUrl("");
            throw error;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        person = newPerson;
        return "success";
    }

    /**
     * Creates a person object with current attributes
     */
    public String create(HttpServletRequest request, OrganizationBean orgBean) throws IError {
        OrganizationUnitTO organization = null;
        try {
            organization = DragonServiceFactory.getInstance().getOrganizationManager()
                    .getOrganization(organizationId);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        PostTO post = null;
        try {
            post = DragonServiceFactory.getInstance().getPostManager().getPost(postId);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        PersonTO newPerson = new PersonTO();
        newPerson.setOrganization(organization);
        newPerson.setEmail(email);
        newPerson.setFirstname(firstname);
        newPerson.setLastname(lastname);
        newPerson.setLocalization(localization);
        newPerson.setMiddlename(middlename);
        newPerson.setPhone(phone);
        newPerson.setPost(post);
        newPerson.setTitle(title);
        newPerson.setContact(contact);

        String personId = "";

        try {
            personId = DragonServiceFactory.getInstance().getPersonManager()
                    .createPerson(newPerson);
        } catch (OrganizationException e) {
            NonLocalizedError error = new NonLocalizedError("Can't create person", e.getMessage(),
                    e);
            error.setType(IError.FUNCTIONAL_ERROR);
            // error.setNextUrl("");
            throw error;
        } catch (DragonDelegateException e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        try {
            person = DragonServiceFactory.getInstance().getPersonManager().getPerson(personId);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        // Renew person list in org Bean.
        orgBean.load(request);
        return "success";
    }

    /**
     * Deletes the current person from the database
     */
    public String delete() throws LocalizedError {
        try {
            DragonServiceFactory.getInstance().getPersonManager().removePerson(person.getId());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    /**
     * Resets changes made to the current movie attributes.
     */
    public void reset() {
        if (person.getOrganization() != null) {
            organizationId = person.getOrganization().getId();
        } else {
            organizationId = "";
        }
        if (person.getPost() != null) {
            postId = person.getPost().getId();
        } else {
            postId = "";
        }
        email = person.getEmail();
        title = person.getTitle();
        firstname = person.getFirstname();
        lastname = person.getLastname();
        localization = person.getLocalization();
        middlename = person.getMiddlename();
        phone = person.getPhone();
        contact = person.isContact();
    }

    /**
     * Clears the movie.
     */
    public void clear() {
        person = new PersonTO();
        reset();
    }

    public void validate() throws ValidationErrors {
        /*
         * Validaton sample ValidationErrors errors = new ValidationErrors();
         * errors.addItemError("personBean.firstname", new IValidationMessage(){
         * public String getMessage(HttpServletRequest request) { return
         * "firstname already in use!"; } });
         * errors.addItemError("personBean.email", new IValidationMessage(){
         * public String getMessage(HttpServletRequest request) { return
         * "email did not respond!"; } });
         * 
         * throw errors;
         */
    }

    public String orgPostExistence() throws LocalizedError {
        List<OrganizationUnitTO> organizations;
        List<PostTO> posts;
        String result = "";

        try {
            organizations = DragonServiceFactory.getInstance().getOrganizationManager()
                    .getAllOrganizations(null);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        try {
            posts = DragonServiceFactory.getInstance().getPostManager().getAllPosts(null);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        if (organizations == null || organizations.isEmpty() || posts == null || posts.isEmpty()) {
            result = "zero";
        } else {
            result = "success";
        }

        return result;

    }

    // ==========================================================
    // === Attributes Getters & Setters
    // ==========================================================
    @Override
    public String getId() {
        if (person == null)
            return "";
        return person.getId();
    }

    /**
     * @return the organizationId
     */
    public String getOrganizationId() {
        return organizationId;
    }

    /**
     * @param organizationId
     *            the organizationId to set
     */
    public void setOrganizationId(String organizationId) {
        this.organizationId = organizationId;
    }

    /**
     * Returns the person's firstname
     * 
     * @return
     */
    public String getFirstname() {
        return firstname;
    }

    /**
     * Sets the person's firstname
     * 
     * @param firstname
     */
    public void setFirstname(String firstname) {
        this.firstname = firstname;
    }

    /**
     * Returns the person's middlename
     * 
     * @return
     */
    public String getMiddlename() {
        return middlename;
    }

    /**
     * Sets the person's middlename
     * 
     * @param middlename
     */
    public void setMiddlename(String middlename) {
        this.middlename = middlename;
    }

    /**
     * Returns the person's lastname
     * 
     * @return
     */
    public String getLastname() {
        return lastname;
    }

    /**
     * Sets the person's lastname
     * 
     * @param lastname
     */
    public void setLastname(String lastname) {
        this.lastname = lastname;
    }

    /**
     * Returns the person's email
     * 
     * @return
     */
    public String getEmail() {
        return email;
    }

    /**
     * Sets the person's email
     * 
     * @param email
     */
    public void setEmail(String email) {
        this.email = email;
    }

    /**
     * Returns the person's phone
     * 
     * @return
     */
    public String getPhone() {
        return phone;
    }

    /**
     * Sets the person's phone
     * 
     * @param email
     */
    public void setPhone(String phone) {
        this.phone = phone;
    }

    /**
     * Returns the person's localization
     * 
     * @return
     */
    public String getLocalization() {
        return localization;
    }

    /**
     * Sets the person's localization
     * 
     * @param localization
     */
    public void setLocalization(String localization) {
        this.localization = localization;
    }

    /**
     * Returns the person's post
     * 
     * @return
     */
    public String getPostId() {
        return postId;
    }

    /**
     * @param postId
     *            the postId to set
     */
    public void setPostId(String postId) {
        this.postId = postId;
    }

    /**
     * Returns the person's title
     * 
     * @return
     */
    public String getTitle() {
        return title;
    }

    /**
     * Sets the person's title
     * 
     * @param title
     */
    public void setTitle(String title) {
        this.title = title;
    }

    @Override
    public String getGenericName() {
        return this.lastname + "_" + this.getFirstname();
    }

    @Override
    public String toString() {
        return this.getGenericName();
    }

    public boolean isContact() {
        return contact;
    }

    public void setContact(boolean contact) {
        this.contact = contact;
    }
}
