/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * PostBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.organization;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.service.organization.OrganizationException;
import org.ow2.dragon.api.to.organization.PostTO;
import org.ow2.dragon.ui.businessdelegate.factory.DragonServiceFactory;
import org.ow2.opensuit.core.error.IError;
import org.ow2.opensuit.core.error.LocalizedError;
import org.ow2.opensuit.core.error.NonLocalizedError;
import org.ow2.opensuit.core.validation.ValidationErrors;

/**
 * 
 * @author ambarthe, ofabre - eBM Websourcing
 * 
 */
public class PostBean {

    private Logger logger = Logger.getLogger(this.getClass());

    // ==========================================================
    // === Attributes
    // ==========================================================
    private PostTO post;

    private String nature;

    private String description;

    private String name;

    private boolean contact;

    // ==========================================================
    // === Control Methods
    // ==========================================================
    /**
     * Loads a post from the request parameter 'id'.
     */
    public void load(HttpServletRequest iRequest) throws LocalizedError {
        String id = iRequest.getParameter("postId");
        try {
            post = DragonServiceFactory.getInstance().getPostManager().getPost(id);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        reset();
    }

    public void load(String postId) throws LocalizedError {
        try {
            post = DragonServiceFactory.getInstance().getPostManager().getPost(postId);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        reset();
    }

    /**
     * Saves the current post with changes.
     */
    public String save() throws IError {
        PostTO newPost = new PostTO();
        newPost.setId(post.getId());
        newPost.setName(name);
        newPost.setNature(nature);
        newPost.setDescription(description);
        newPost.setContact(contact);

        try {
            DragonServiceFactory.getInstance().getPostManager().updatePost(newPost);
        } catch (OrganizationException e) {
            NonLocalizedError error = new NonLocalizedError("Can't create post.", e.getMessage(), e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        post = newPost;
        return "success";
    }

    /**
     * Creates a post object with current attributes
     */
    public String create(HttpServletRequest iRequest, OrganizationBean orgBean,
            ListOfPostsBean listOfPosts) throws IError {
        PostTO newPost = new PostTO();
        newPost.setName(name);
        newPost.setNature(nature);
        newPost.setDescription(description);
        newPost.setContact(contact);

        String postId = "";

        try {
            postId = DragonServiceFactory.getInstance().getPostManager().createPost(newPost);
            DragonServiceFactory.getInstance().getOrganizationManager().addPost(orgBean.getId(),
                    postId);
            orgBean.loadOrganizationPosts();
            listOfPosts.loadFree(iRequest);
        } catch (OrganizationException e) {
            NonLocalizedError error = new NonLocalizedError("Can't create post.", e.getMessage(), e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        try {
            post = DragonServiceFactory.getInstance().getPostManager().getPost(postId);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    /**
     * Deletes the current post from the database
     */
    public String delete() throws LocalizedError {
        try {
            DragonServiceFactory.getInstance().getPostManager().removePost(post.getId());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    /**
     * Resets changes made to the current post attributes.
     */
    public void reset() {
        name = post.getName();
        nature = post.getNature();
        description = post.getDescription();
        contact = post.isContact();
    }

    /**
     * Clears the post.
     */
    public String clear() {
        post = new PostTO();
        reset();
        return "success";
    }

    public void validate() throws ValidationErrors {

    }

    // ==========================================================
    // === Attributes Getters & Setters
    // ==========================================================
    public String getId() {
        if (post == null)
            return "";
        return post.getId();
    }

    /**
     * @return the nature
     */
    public String getNature() {
        return nature;
    }

    /**
     * @param nature
     *            the nature to set
     */
    public void setNature(String nature) {
        this.nature = nature;
    }

    /**
     * @return the description
     */
    public String getDescription() {
        return description;
    }

    /**
     * @param description
     *            the description to set
     */
    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * @return the name
     */
    public String getName() {
        return name;
    }

    /**
     * @param name
     *            the name to set
     */
    public void setName(String name) {
        this.name = name;
    }

    public boolean isContact() {
        return contact;
    }

    public void setContact(boolean contact) {
        this.contact = contact;
    }

}
