/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * RoleBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.administration;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.service.administration.RoleGroupException;
import org.ow2.dragon.api.to.administration.RoleGroupTO;
import org.ow2.dragon.api.to.administration.UserTO;
import org.ow2.dragon.ui.businessdelegate.factory.DragonServiceFactory;
import org.ow2.opensuit.core.error.IError;
import org.ow2.opensuit.core.error.LocalizedError;
import org.ow2.opensuit.core.error.NonLocalizedError;
import org.ow2.opensuit.core.session.OpenSuitSession;

/**
 * 
 * @author strino - eBM Websourcing
 * 
 */

public class RoleBean {

    private Logger logger = Logger.getLogger(this.getClass());

    // ==========================================================
    // === Attributes
    // ==========================================================

    // Attributes for RoleGroup
    private String id;

    private String name;

    private List<RoleGroupTO> roles = new ArrayList<RoleGroupTO>();

    private RoleGroupTO role;

    // Attributes for Users
    private List<UserTO> allUsers;

    private List<UserTO> listUsersTO = new ArrayList<UserTO>();

    // Attributes for Roles
    private List<String> listRoles = new ArrayList<String>();

    private static String[] ALL_ROLES = { "MANAGEADMINISTRATION", "MANAGEORGA", "VIEWORGA",
            "MANAGESERVICE", "VIEWSERVICE", "MANAGERUNTIMEMANAGER", "VIEWRUNTIMEMANAGER" };

    /**
     * Default Constructor
     * 
     */

    public RoleBean() {

    }

    // ==========================================================
    // === Control Methods
    // ==========================================================

    /**
     * Load selected roles group
     * 
     */
    public void load(HttpServletRequest iRequest) throws LocalizedError {
        String idRole = iRequest.getParameter("roleId");

        try {
            role = DragonServiceFactory.getInstance().getRoleGroupManager().getRoleGroup(idRole);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        name = role.getName();

        id = idRole;

        listRoles = role.getListRoles();

        listUsersTO = role.getListUsers();

    }

    /**
     * load all Roles
     * 
     */
    public void loadAllRoles() throws LocalizedError {
        try {
            roles = DragonServiceFactory.getInstance().getRoleGroupManager().getAllRoleGroup();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

    }

    /**
     * load users not associate with the current roles group
     * 
     */
    public void loadUsersNotInRole() throws LocalizedError {
        try {
            allUsers = DragonServiceFactory.getInstance().getUserManager()
                    .getUserNotInRoleGroup(id);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

    }

    /**
     * create a new roles group
     * 
     */
    public void create() throws IError {
        role = new RoleGroupTO();
        role.setName(name);

        role.setListRoles(listRoles);
        role.setListUsers(listUsersTO);

        try {
            id = DragonServiceFactory.getInstance().getRoleGroupManager().createRoleGroup(role);
        } catch (RoleGroupException e) {
            NonLocalizedError error = new NonLocalizedError("Can't create role.", e.getMessage(), e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        } catch (Throwable e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

    }

    /**
     * save roles group modifications
     * 
     */
    public void save() throws IError {
        RoleGroupTO roleTmp = new RoleGroupTO();

        roleTmp.setId(id);

        roleTmp.setName(name);

        roleTmp.setListRoles(listRoles);

        if (listUsersTO != null) {
            roleTmp.setListUsers(listUsersTO);
        }

        try {
            id = DragonServiceFactory.getInstance().getRoleGroupManager().updateRoleGroup(roleTmp);
        } catch (RoleGroupException e) {
            // need to rollback local changes
            load(OpenSuitSession.getCurrentRequest());
            name = role.getName();
            NonLocalizedError error = new NonLocalizedError("Can't save role.", e.getMessage(), e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        } catch (Throwable e) {
            // need to rollback local changes
            load(OpenSuitSession.getCurrentRequest());
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

    }

    /**
     * delete roles group
     * 
     */
    public void delete(HttpServletRequest iRequest) throws IError {
        String idRole = iRequest.getParameter("roleId");

        try {
            DragonServiceFactory.getInstance().getRoleGroupManager().removeRoleGroup(idRole);
            loadAllRoles();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    /**
     * refresh list <User> for current roles group
     * 
     */
    public void refresh() throws LocalizedError {
        try {
            role = DragonServiceFactory.getInstance().getRoleGroupManager().getRoleGroup(id);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        name = role.getName();

        listRoles = role.getListRoles();

        listUsersTO = role.getListUsers();
    }

    public void clear() {

    }

    public void reset() {
        // do nothing
    }

    /**
     * remove user from roles group List<User>
     * 
     */
    public void removeUser(HttpServletRequest iRequest) throws IError {
        String idUser = iRequest.getParameter("userId");

        try {
            DragonServiceFactory.getInstance().getRoleGroupManager().removeUser(idUser, id);
        } catch (RoleGroupException e) {
            NonLocalizedError error = new NonLocalizedError("Can't create role.", e.getMessage(), e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        } catch (Throwable e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        refresh();

    }

    /**
     * remove all user from roles group List<User>
     * 
     */
    public void removeAllUsers() throws IError {
        for (UserTO us : listUsersTO) {
            String idUser = us.getId();

            try {
                DragonServiceFactory.getInstance().getRoleGroupManager().removeUser(idUser, id);
            } catch (RoleGroupException e) {
                NonLocalizedError error = new NonLocalizedError("Can't create role.", e
                        .getMessage(), e);
                error.setType(IError.FUNCTIONAL_ERROR);
                throw error;
            } catch (Throwable e) {
                logger.error(e.getMessage(), e);
                throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
            }
        }
        refresh();
    }

    /**
     * add user in roles group List<User>
     * 
     */
    public void addUser(HttpServletRequest iRequest) throws IError {
        String idUser = iRequest.getParameter("userId");

        try {
            DragonServiceFactory.getInstance().getRoleGroupManager().addUser(idUser, id);
        } catch (RoleGroupException e) {
            NonLocalizedError error = new NonLocalizedError("Can't create role.", e.getMessage(), e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        } catch (Throwable e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        loadUsersNotInRole();
        refresh();
    }

    /**
     * add all user in roles group List<User>
     * 
     */
    public void addAllUsers() throws IError {

        for (UserTO us : allUsers) {
            String idUser = us.getId();

            try {
                DragonServiceFactory.getInstance().getRoleGroupManager().addUser(idUser, id);
            } catch (RoleGroupException e) {
                NonLocalizedError error = new NonLocalizedError("Can't create role.", e
                        .getMessage(), e);
                error.setType(IError.FUNCTIONAL_ERROR);
                throw error;
            } catch (Throwable e) {
                logger.error(e.getMessage(), e);
                throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
            }
        }

        refresh();

    }

    /**
     * the roles group admin suppression disabled
     * 
     */
    public boolean canDeleteOrUpadateRole(String roleName) {
        boolean verif;

        if (roleName.equals("admin")) {
            verif = false;
        } else {
            verif = true;
        }

        return verif;
    }

    // ==========================================================
    // === Attributes Getters & Setters
    // ==========================================================

    /**
     * @return the id
     */
    public String getId() {
        return id;
    }

    /**
     * @param setId
     *            the id to set
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * @return the name
     */
    public String getName() {
        return name;
    }

    /**
     * @param setName
     *            the name to set
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * @return the roles
     */
    public List<RoleGroupTO> getRoles() {
        return roles;
    }

    /**
     * @param setRoles
     *            the roles to set
     */
    public void setRoles(List<RoleGroupTO> roles) {
        this.roles = roles;
    }

    /**
     * @return the role
     */
    public RoleGroupTO getRole() {
        return role;
    }

    /**
     * @param setRole
     *            the role to set
     */
    public void setRole(RoleGroupTO role) {
        this.role = role;
    }

    /**
     * @return the allUsers
     */
    public List<UserTO> getAllUsers() {
        return allUsers;
    }

    /**
     * @param setAllUsers
     *            the allUsers to set
     */
    public void setAllUsers(List<UserTO> allUsers) {
        this.allUsers = allUsers;
    }

    /**
     * @return the ListUsersTO
     */
    public List<UserTO> getListUsersTO() {
        return listUsersTO;
    }

    /**
     * @param setListUsersTO
     *            the listUsersTO to set
     */
    public void setListUsersTO(List<UserTO> listUsersTO) {
        this.listUsersTO = listUsersTO;
    }

    /**
     * @return the listRoles
     */
    public List<String> getListRoles() {
        return listRoles;
    }

    /**
     * @param setListRoles
     *            the listRoles to set
     */
    public void setListRoles(List<String> listRoles) {
        this.listRoles = listRoles;
    }

    public static String[] getAllRoles() {
        return ALL_ROLES;
    }

}
