/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * UserBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.administration;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.service.administration.UserException;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.SortCriteria;
import org.ow2.dragon.api.to.administration.RoleGroupTO;
import org.ow2.dragon.api.to.administration.UserTO;
import org.ow2.dragon.api.to.organization.OrganizationUnitTO;
import org.ow2.dragon.api.to.organization.PersonTO;
import org.ow2.dragon.ui.businessdelegate.factory.DragonServiceFactory;
import org.ow2.dragon.util.StringHelper;
import org.ow2.opensuit.core.error.IError;
import org.ow2.opensuit.core.error.LocalizedError;
import org.ow2.opensuit.core.error.NonLocalizedError;
import org.ow2.opensuit.core.validation.LocalizedValidationError;
import org.ow2.opensuit.core.validation.ValidationErrors;

/**
 * 
 * @author strino - eBM Websourcing
 * 
 */

public class UserBean {
    private Logger logger = Logger.getLogger(this.getClass());

    // ==========================================================
    // === Attributes
    // ==========================================================

    // Attributes for User
    private String id;

    private String login;

    private String pass;

    private String confirmPass;

    private String newPass;

    private String oldPass;

    private UserTO user;

    private List<UserTO> users;

    // Attributes for Organizations/Persons
    private List<OrganizationUnitTO> organizations;

    private List<PersonTO> persons = new ArrayList<PersonTO>();

    private String idOrga;

    private String idPerson;

    private boolean visible = false;

    // Attributes for Role Group
    private List<RoleGroupTO> allRoles;

    private List<RoleGroupTO> roles = new ArrayList<RoleGroupTO>();

    /**
     * Default Constructor
     * 
     */
    public UserBean() {

    }

    // ==========================================================
    // === Control Methods
    // ==========================================================

    /**
     * Load selected user
     * 
     */
    public void load(HttpServletRequest iRequest) throws LocalizedError {
        id = iRequest.getParameter("userId");

        try {
            user = DragonServiceFactory.getInstance().getUserManager().getUser(id);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        login = user.getName();
        pass = user.getPassword();
        roles = user.getListRoleGroup();
        if (user.getPerson() != null) {
            loadOrganizations();
            idOrga = user.getPerson().getOrganization().getId();
            loadPersons();
            idPerson = user.getPerson().getId();

            visible = true;
        } else {
            loadOrganizations();
            idOrga = null;
            visible = false;
        }

    }

    /**
     * load all users
     * 
     */
    public void loadUsers() throws LocalizedError {
        try {
            users = DragonServiceFactory.getInstance().getUserManager().getAllUser();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

    }

    /**
     * load roles not associate with the current user
     * 
     */
    public void loadRoleNotInUser() throws LocalizedError {
        try {
            allRoles = DragonServiceFactory.getInstance().getRoleGroupManager()
                    .getRoleGroupNotInUser(id);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

    }

    /**
     * load all Organizations
     * 
     */
    public void loadOrganizations() throws LocalizedError {
        try {
            OrganizationUnitTO org = new OrganizationUnitTO();
            org.setId("");
            org.setName("select an organization");

            organizations = DragonServiceFactory.getInstance().getOrganizationManager()
                    .getAllOrganizations(createSortOption(SortCriteria.PARTY_NAME, true));
            organizations.add(0, org);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    /**
     * load Persons for selected Organization
     * 
     */
    public boolean loadPersons() throws LocalizedError {
        if (StringHelper.isNullOrEmpty(idOrga)) {
            idPerson = null;
            visible = false;
        } else {
            try {
                persons = DragonServiceFactory.getInstance().getPersonManager()
                        .getPersonsByOrganization(idOrga,
                                createSortOption(SortCriteria.PERSON_LASTNAME, true));

                if (persons == null || persons.isEmpty()) {
                    PersonTO pers = new PersonTO();
                    pers.setId("");
                    pers.setFirstname("No persons in this organization");
                    pers.setLastname("");
                    persons.add(pers);
                    idPerson = null;
                }
                visible = true;
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
                visible = false;
                throw new LocalizedError("database_technical", new Object[] { e.getMessage() });

            }

        }
        return visible;
    }

    /**
     * create the new user
     * 
     */
    public void create() throws IError {
        user = new UserTO();
        user.setName(login);
        pass = passwordCrypting(pass);
        user.setPassword(pass);
        PersonTO pers = new PersonTO();
        if (!StringHelper.isNullOrEmpty(idPerson)) {
            pers.setId(idPerson);
            user.setPerson(pers);
        }

        try {
            id = DragonServiceFactory.getInstance().getUserManager().createUser(user);
        } catch (UserException e) {
            NonLocalizedError error = new NonLocalizedError("Can't create role.", e.getMessage(), e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        } catch (Throwable e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    /**
     * save user modifications
     * 
     */
    public void save() throws IError {
        UserTO userTmp = new UserTO();
        userTmp.setName(login);
        userTmp.setId(id);
        userTmp.setPassword(pass);
        userTmp.setListRoleGroup(roles);
        PersonTO pers = new PersonTO();
        if (!StringHelper.isNullOrEmpty(idPerson)) {
            pers.setId(idPerson);
            userTmp.setPerson(pers);
        } else {
            userTmp.setPerson(null);
        }

        try {
            id = DragonServiceFactory.getInstance().getUserManager().updateUser(userTmp);

        } catch (UserException e) {
            login = user.getName();
            NonLocalizedError error = new NonLocalizedError("Can't save role.", e.getMessage(), e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        } catch (Throwable e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

    }

    /**
     * delete user
     * 
     */
    public void delete(HttpServletRequest iRequest) throws IError {
        String idRole = iRequest.getParameter("userID");

        try {
            DragonServiceFactory.getInstance().getUserManager().removeUser(idRole);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        loadUsers();
    }

    public void reset() {
        // do nothing
    }

    /**
     * refresh user's list<RoleGoup>
     * 
     */
    public void refresh() throws IError {
        try {
            user = DragonServiceFactory.getInstance().getUserManager().getUser(id);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        login = user.getName();
        pass = user.getPassword();
        roles = user.getListRoleGroup();
    }

    public void clear() {
        try {
            organizations = DragonServiceFactory.getInstance().getOrganizationManager()
                    .getAllOrganizations(null);
        } catch (Exception e) {
            e.printStackTrace();
        }
        pass = null;
        confirmPass = null;

    }

    /**
     * add role in user's List<RoleGroup>
     * 
     */
    public void addRole(HttpServletRequest iRequest) throws IError {
        String idRole = iRequest.getParameter("ID");

        try {
            DragonServiceFactory.getInstance().getUserManager().addRoleGroup(id, idRole);

        } catch (UserException e) {
            NonLocalizedError error = new NonLocalizedError("Can't save user.", e.getMessage(), e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        } catch (Throwable e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        refresh();

    }

    /**
     * add all roles in user's List<RoleGroup>
     * 
     */
    public void addAllRoles() throws IError {
        for (RoleGroupTO rG : allRoles) {
            String idRole = rG.getId();
            try {
                DragonServiceFactory.getInstance().getUserManager().addRoleGroup(id, idRole);

            } catch (UserException e) {
                NonLocalizedError error = new NonLocalizedError("Can't save user.", e.getMessage(),
                        e);
                error.setType(IError.FUNCTIONAL_ERROR);
                throw error;
            } catch (Throwable e) {
                logger.error(e.getMessage(), e);
                throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
            }
        }
        refresh();
    }

    /**
     * remove role in user's List<RoleGroup>
     * 
     */
    public void removeRole(HttpServletRequest iRequest) throws IError {
        String idRole = iRequest.getParameter("ID");

        try {
            DragonServiceFactory.getInstance().getUserManager().removeRoleGroup(id, idRole);

        } catch (UserException e) {
            NonLocalizedError error = new NonLocalizedError("Can't save role.", e.getMessage(), e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        } catch (Throwable e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        refresh();

    }

    /**
     * remove all roles in user's List<RoleGroup>
     * 
     */
    public void removeAllRoles() throws IError {
        for (RoleGroupTO rG : roles) {
            String idRole = rG.getId();
            try {
                DragonServiceFactory.getInstance().getUserManager().removeRoleGroup(id, idRole);

            } catch (UserException e) {
                NonLocalizedError error = new NonLocalizedError("Can't save role.", e.getMessage(),
                        e);
                error.setType(IError.FUNCTIONAL_ERROR);
                throw error;
            } catch (Throwable e) {
                logger.error(e.getMessage(), e);
                throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
            }
        }
        refresh();
    }

    /**
     * the user master suppression disabled
     * 
     */
    public boolean canDeleteOrUpadateUser(String userName) {
        boolean verif;

        if (userName.equals("master")) {
            verif = false;
        } else {
            verif = true;
        }
        return verif;
    }

    /**
     * password verification
     * 
     */
    public void validate() throws ValidationErrors {
        ValidationErrors errors = new ValidationErrors();

        if (!pass.equalsIgnoreCase(confirmPass)) {
            errors.addItemError("userBean.confirmPass", new LocalizedValidationError(
                    "validation.password"));
        }

        if (errors.hasErrors())
            throw errors;
    }

    /**
     * change password verification
     * 
     */
    public void validateChange() throws ValidationErrors {
        ValidationErrors errors = new ValidationErrors();

        oldPass = passwordCrypting(oldPass);

        if (!newPass.equalsIgnoreCase(confirmPass)) {
            errors.addItemError("userBean.confirmPass", new LocalizedValidationError(
                    "validation.password"));
        } else if (!oldPass.equalsIgnoreCase(pass)) {
            errors.addItemError("userBean.oldPass", new LocalizedValidationError(
                    "validation.oldPassword"));
        }

        if (errors.hasErrors())
            throw errors;
    }

    /**
     * save user's modification password
     * 
     */
    public void changePassword(HttpServletRequest iRequest) throws IError {
        user = new UserTO();
        user.setName(login);
        user.setId(id);
        newPass = passwordCrypting(newPass);
        user.setPassword(newPass);
        user.setListRoleGroup(roles);
        try {
            id = DragonServiceFactory.getInstance().getUserManager().updateUser(user);

        } catch (UserException e) {
            NonLocalizedError error = new NonLocalizedError("Can't save role.", e.getMessage(), e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        } catch (Throwable e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    /**
     * encrypt user's password
     * 
     */
    private String passwordCrypting(String password) {

        byte[] uniqueKey = password.getBytes();
        byte[] hash = null;

        try {
            hash = MessageDigest.getInstance("MD5").digest(uniqueKey);
        } catch (NoSuchAlgorithmException e) {
            throw new Error("No MD5 support in this VM.");
        }

        StringBuilder hashString = new StringBuilder();
        for (int i = 0; i < hash.length; i++) {
            String hex = Integer.toHexString(hash[i]);
            if (hex.length() == 1) {
                hashString.append('0');
                hashString.append(hex.charAt(hex.length() - 1));
            } else
                hashString.append(hex.substring(hex.length() - 2));
        }
        return hashString.toString();

    }

    private RequestOptionsTO createSortOption(SortCriteria sortOn, boolean ascendingly) {
        RequestOptionsTO requestOptionsTO = new RequestOptionsTO();
        requestOptionsTO.setSortAscendingly(ascendingly);
        requestOptionsTO.setSortCriteria(sortOn);
        return requestOptionsTO;
    }

    // ==========================================================
    // === Attributes Getters & Setters
    // ==========================================================

    /**
     * @return the id
     */
    public String getId() {
        return id;
    }

    /**
     * @param setId
     *            the id to set
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * @return the login
     */
    public String getLogin() {
        return login;
    }

    /**
     * @param setLogin
     *            the login to set
     */
    public void setLogin(String login) {
        this.login = login;
    }

    /**
     * @return the pass
     */
    public String getPass() {
        return pass;
    }

    /**
     * @param setPass
     *            the pass to set
     */
    public void setPass(String pass) {
        this.pass = pass;
    }

    /**
     * @return the confirmPass
     */
    public String getConfirmPass() {
        return confirmPass;
    }

    /**
     * @param setConfirmPass
     *            the confirmPass to set
     */
    public void setConfirmPass(String confirmPass) {
        this.confirmPass = confirmPass;
    }

    /**
     * @return the newPass
     */
    public String getNewPass() {
        return newPass;
    }

    /**
     * @param setNewPass
     *            the newPass to set
     */
    public void setNewPass(String newPass) {
        this.newPass = newPass;
    }

    /**
     * @return the oldPass
     */
    public String getOldPass() {
        return oldPass;
    }

    /**
     * @param setOldPass
     *            the oldPass to set
     */
    public void setOldPass(String oldPass) {
        this.oldPass = oldPass;
    }

    /**
     * @return the user
     */
    public UserTO getUser() {
        return user;
    }

    /**
     * @param setUser
     *            the user to set
     */
    public void setUser(UserTO user) {
        this.user = user;
    }

    /**
     * @return the users
     */
    public List<UserTO> getUsers() {
        return users;
    }

    /**
     * @param setUsers
     *            the users to set
     */
    public void setUsers(List<UserTO> users) {
        this.users = users;
    }

    /**
     * @return the organizations
     */
    public List<OrganizationUnitTO> getOrganizations() {
        return organizations;
    }

    /**
     * @param setOrganizations
     *            the organizations to set
     */
    public void setOrganizations(List<OrganizationUnitTO> organizations) {
        this.organizations = organizations;
    }

    /**
     * @return the persons
     */
    public List<PersonTO> getPersons() {
        return persons;
    }

    /**
     * @param setPersons
     *            the persons to set
     */
    public void setPersons(List<PersonTO> persons) {
        this.persons = persons;
    }

    /**
     * return the Organization id
     * 
     * @return the idOrga
     */
    public String getIdOrga() {
        return idOrga;
    }

    /**
     * @param setIdOrga
     *            the idOrga to set
     */
    public void setIdOrga(String idOrga) {
        this.idOrga = idOrga;
    }

    /**
     * return the Person id
     * 
     * @return the idPerson
     */
    public String getIdPerson() {
        return idPerson;
    }

    /**
     * @param setIdPerson
     *            the idPerson to set
     */
    public void setIdPerson(String idPerson) {
        this.idPerson = idPerson;
    }

    /**
     * @return allRoles
     */
    public List<RoleGroupTO> getAllRoles() {
        return allRoles;
    }

    /**
     * @param setAllRoles
     *            the allRoles to set
     */
    public void setAllRoles(List<RoleGroupTO> allRoles) {
        this.allRoles = allRoles;
    }

    /**
     * @return roles
     */
    public List<RoleGroupTO> getRoles() {
        return roles;
    }

    /**
     * @param setRoles
     *            the roles to set
     */
    public void setRoles(List<RoleGroupTO> roles) {
        this.roles = roles;
    }

    public boolean isVisible() {
        return visible;
    }

    public void setVisible(boolean visible) {
        this.visible = visible;
    }

}
