/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * TechnicalServiceBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.deployment;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.service.deployment.DeploymentException;
import org.ow2.dragon.api.service.organization.OrganizationException;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.SortCriteria;
import org.ow2.dragon.api.to.common.KeyedRefTO;
import org.ow2.dragon.api.to.deployment.CommentTO;
import org.ow2.dragon.api.to.deployment.EndpointTO;
import org.ow2.dragon.api.to.deployment.RatingTO;
import org.ow2.dragon.api.to.deployment.TechnicalServiceTO;
import org.ow2.dragon.api.to.metadata.SimpleFileTO;
import org.ow2.dragon.api.to.organization.OrgToTechServiceTO;
import org.ow2.dragon.api.to.organization.PersonToTechServiceTO;
import org.ow2.dragon.api.to.specification.ServiceSpecificationTO;
import org.ow2.dragon.ui.businessdelegate.DragonDelegateException;
import org.ow2.dragon.ui.businessdelegate.factory.DragonServiceFactory;
import org.ow2.dragon.util.ContentType;
import org.ow2.dragon.util.InputStreamUtil;
import org.ow2.dragon.util.StringHelper;
import org.ow2.dragon.util.XMLUtil;
import org.ow2.opensuit.core.error.IError;
import org.ow2.opensuit.core.error.LocalizedError;
import org.ow2.opensuit.core.error.NonLocalizedError;
import org.ow2.opensuit.core.impl.multiparts.IFileUploadPart;
import org.w3c.dom.Document;

/**
 * 
 * @author ofabre - eBM Websourcing
 * 
 */
public class TechnicalServiceBean {

    private final Logger logger = Logger.getLogger(this.getClass());

    // ==========================================================
    // === Attributes
    // ==========================================================
    private TechnicalServiceTO techService;

    private List<PersonToTechServiceTO> personRoles;

    private List<OrgToTechServiceTO> organizationRoles;

    private List<EndpointTO> endpoints;

    private String name;

    private String purpose;

    private List<KeyedRefTO> categories;

    private String wsdlUrl;

    private List<ServiceSpecificationTO> specsTO;

    private List<SimpleFileTO> relatedDocsTO;

    private List<String> selectedCategories;

    // Add WSDL Attributes
    private String fileDescriptionURL;

    private String fileDescriptionName;

    private boolean storeEntities;

    private Document wsdlDocContent;

    private String wsdlDocFileName;

    // Added By TIAR Abderrahmane

    // The technical service rating of the user connected.
    private int myRating;

    // The global rating of the technical service.
    private float globalRating;

    // Tags are listed as a hashSet.
    private Set<String> tags = new HashSet<String>();

    // All the comments of the technical service.
    private List<CommentTO> comments;

    // This String will be useful to get an eventual comment to add.
    private String newComment;

    // ////////////////////////////////////////////////////
    // setters and getters for the Community part
    // ////////////////////////////////////////////////////
    /**
     * @param rating
     *            the user rating to set for this service.
     */
    public void setMyRating(int rating) {
        this.myRating = rating;
    }

    /**
     * @return the user rating of this service.
     */
    public int getMyRating() {
        return this.myRating;
    }

    /**
     * @param globalRating
     *            the global rating to set for this service.
     */
    public void setGlobalRating(float globalRating) {
        this.globalRating = globalRating;
    }

    /**
     * @param globalRating
     *            a String representing the global rating to set.
     */
    public void setGlobalRating(String globalRating) {
        this.globalRating = Float.parseFloat(globalRating);
    }

    /**
     * @return the global rating in a String format.
     */
    public String getGlobalRating() {
        String toPrint = Float.toString(this.globalRating);
        // We need to print only one digit after the decimal.
        return toPrint.substring(0, 3);
    }

    /**
     * @param tags
     *            a String representing the new tags to add ("tag1, tag2...")
     */
    // the tags follows the shape: tag1, tag2 ....
    public void setTags(String tags) {
        // Remove all spaces.
        String[] pieces = tags.split(",");

        for (int i = 0; i <= pieces.length - 1; i++) {
            // No need to add white spaces as tags.
            pieces[i] = pieces[i].trim();
            if (!StringHelper.isNullOrEmpty(pieces[i]))
                this.tags.add(pieces[i]);
        }
    }

    /**
     * @param tags
     *            tags to set.
     */
    public void setTags(Set<String> tags) {
        this.tags = tags;
    }

    /**
     * @return technical service tags in a string format.
     */
    public String getTags() {
        return this.tags.toString();
    }

    /**
     * @return the new comment to add.
     */
    public String getNewComment() {
        return newComment;
    }

    /**
     * @param newComment
     *            the new comment to set.
     */
    public void setNewComment(String newComment) {
        this.newComment = newComment;
    }

    /**
     * @return the list of this technical service comments
     */
    public List<CommentTO> getComments() {
        return this.comments;
    }

    /**
     * @param the
     *            list of comments to set to this service.
     */
    public void setComments(List<CommentTO> comments) {
        this.comments = comments;
    }

    /**
     * This function adds a new comment to the technical service.
     * 
     * @param user
     * @return "success"
     * @throws LocalizedError
     */
    public String addComment(String user) throws LocalizedError {
        String cmtTemp = newComment.trim();
        // no need to add a white space comment.
        if (!StringHelper.isNullOrEmpty(cmtTemp)) {
            // Creating a new Transfer Object comment.
            CommentTO newCommentTO = new CommentTO();
            newCommentTO.setContent(newComment);
            newCommentTO.setUserName(user);
            newCommentTO.setDate(new Date());

            // Try saving the comment in the Technical Service
            try {
                DragonServiceFactory.getInstance().getTechServiceManager().updateComments(
                        techService.getId(), newCommentTO);

                /*
                 * We Need the id of the comment that has just been saved.
                 * Useful if the Admin wants to delete the new comment without
                 * reloading the page to get the id.
                 */
                final TechnicalServiceTO techServiceTemp = DragonServiceFactory.getInstance()
                        .getTechServiceManager().getTechService(techService.getId(), null);
                // set this comments.
                comments = techServiceTemp.getComments();
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
                throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
            }
        }
        // Setting the new Comment to an empty string
        newComment = "";
        return "success";
    }

    /**
     * remove a comment from the technical service.
     * 
     * @param request
     *            contains the id of the comment to remove.
     * @return "success"
     * @throws LocalizedError
     */
    public String removeComment(HttpServletRequest request) throws LocalizedError {
        // get the id of the comment.
        String id = request.getParameter("comment");

        try {
            DragonServiceFactory.getInstance().getTechServiceManager().removeComment(
                    techService.getId(), id);

            // getting the new list of comments after the removal.
            final TechnicalServiceTO techServiceTemp = DragonServiceFactory.getInstance()
                    .getTechServiceManager().getTechService(techService.getId(), null);
            // set this comments.
            comments = techServiceTemp.getComments();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    // Have a look at load, save, create, reset
    // End of what's been added By TIAR Abderrahmane.

    /**
     * @return the wsdlUrl
     */
    public String getWsdlUrl() {
        return wsdlUrl;
    }

    // ==========================================================
    // === Control Methods
    // ==========================================================
    /**
     * Loads a technical service from the request parameter 'id'.
     */
    public void load(HttpServletRequest iRequest) throws LocalizedError {
        String id = iRequest.getParameter("serviceId");

        // We add the userName for the ratings.
        String user = iRequest.getUserPrincipal().getName();

        try {
            techService = DragonServiceFactory.getInstance().getTechServiceManager()
                    .getTechService(id, user);
            loadOrgRoles();
            loadPersonRoles();
            loadEndpoints();
            loadServiceSpecs();
            loadRelatedDocs();

        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        reset();
    }

    public void addRelatedDoc(IFileUploadPart file) throws IError {
        try {
            DragonServiceFactory.getInstance().getTechServiceManager().registerRelatedDoc(
                    techService.getId(), file.getFileContentType(),
                    InputStreamUtil.getBytes(file.getFileInputStream()), file.getFileName());
        } catch (DeploymentException e) {
            logger.error(e.getMessage(), e);
            NonLocalizedError error = new NonLocalizedError("Document content type not supported.",
                    "Supported types are: DOC, PDF, XML, HTML, XLS, PPT, TXT, RTF, ODT, ODS, ODP",
                    e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    public String addServiceDesc() throws LocalizedError {
        try {
            DragonServiceFactory.getInstance().getTechServiceManager().addServiceDefFile(
                    URI.create(fileDescriptionURL), fileDescriptionName, techService.getId(),
                    storeEntities);
            if (storeEntities) {
                loadEndpoints();
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return loadServiceSpecs();
    }

    public void uploadServDesc(IFileUploadPart file) throws LocalizedError {
        try {
            wsdlDocContent = XMLUtil.loadDocument(file.getFileInputStream());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        wsdlDocFileName = extractShortFilename(file);
    }

    public String addServiceDescFromFile() throws LocalizedError {

        String fileName;
        if (StringHelper.isNullOrEmpty(fileDescriptionName)) {
            fileName = wsdlDocFileName;
        } else {
            fileName = fileDescriptionName;
        }

        try {
            DragonServiceFactory.getInstance().getTechServiceManager().addServiceDefFile(
                    wsdlDocContent, fileName, techService.getId(), storeEntities);
            if (storeEntities) {
                loadEndpoints();
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return loadServiceSpecs();
    }

    public String removeRelatedDocument(String documentId) throws LocalizedError {
        try {
            DragonServiceFactory.getInstance().getTechServiceManager().removeRelatedDoc(
                    techService.getId(), documentId);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        loadRelatedDocs();
        return "success";
    }

    public String removeEndpoint(String endpointId) throws IError {
        try {
            DragonServiceFactory.getInstance().getEndpointManager().removeEndpoint(endpointId);
            loadEndpoints();
        } catch (DeploymentException e) {
            NonLocalizedError error = new NonLocalizedError("Can't create endpoint.", e
                    .getMessage(), e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    public String removeServSpec(String documentId) throws LocalizedError {
        try {
            DragonServiceFactory.getInstance().getTechServiceManager().removeServiceSpec(
                    techService.getId(), documentId);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return loadServiceSpecs();
    }

    public String removeCategories() throws LocalizedError {
        try {
            DragonServiceFactory.getInstance().getTechServiceManager().removeCategories(
                    techService.getId(), selectedCategories);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        loadCategories();
        return "success";
    }

    private void loadEndpoints() throws DeploymentException, DragonDelegateException {
        endpoints = DragonServiceFactory.getInstance().getEndpointManager()
                .getEndpointsByTechService(techService.getId(),
                        createSortOption(SortCriteria.ENDPOINT_NAME, true));
    }

    public String loadServiceSpecs() throws LocalizedError {
        try {
            List<ServiceSpecificationTO> result = DragonServiceFactory.getInstance()
                    .getWSDLImporter().getServSpecsForTechServ(techService.getId());

            if (result != null) {
                for (ServiceSpecificationTO serviceSpecificationTO : result) {
                    // Fill file name if empty
                    if (serviceSpecificationTO.getFileName() == null) {
                        serviceSpecificationTO.setFileName("wsdlFile.wsdl");
                    }
                }
            }

            this.specsTO = result;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        return "success";
    }

    public void loadCategories() throws LocalizedError {
        try {
            categories = DragonServiceFactory.getInstance().getTechServiceManager()
                    .getCategoriesForTechServ(techService.getId());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    public String loadRelatedDocs() throws LocalizedError {
        List<SimpleFileTO> result = null;
        try {
            result = DragonServiceFactory.getInstance().getTechServiceManager()
                    .getRelatedDocsForTechServ(techService.getId());
            if (result != null) {
                for (SimpleFileTO simpleFileTO : result) {
                    // Fill file names
                    if (StringHelper.isNullOrEmpty(simpleFileTO.getFileName())) {
                        String suffix = ContentType.getRelatedSuffix(ContentType
                                .fromString(simpleFileTO.getFileType()));
                        simpleFileTO.setFileName("Document" + suffix);
                    }
                    // Fill empty titles
                    if (StringHelper.isNullOrEmpty(simpleFileTO.getTitle())) {
                        simpleFileTO.setTitle("NoTitle");
                    }
                }
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        this.relatedDocsTO = result;
        return "success";
    }

    /**
     * Saves the current techService with changes.
     */
    public String save() throws IError {
        TechnicalServiceTO newTechService = new TechnicalServiceTO();
        newTechService.setId(techService.getId());
        newTechService.setName(name);
        newTechService.setPurpose(purpose);
        newTechService.setCategories(categories);

        // Added By TIAR Abderrahmane
        // Adding to newtechService the comments, tags and rating.
        RatingTO ratingTO = new RatingTO();
        ratingTO.setMyRating(myRating);
        ratingTO.setUserName(techService.getRatingTO().getUserName());
        ratingTO.setId(techService.getRatingTO().getid());
        newTechService.setRatingTO(ratingTO);
        newTechService.setTags(tags);

        newTechService.setComments(comments);
        // End

        try {
            DragonServiceFactory.getInstance().getTechServiceManager().updateTechService(
                    newTechService);

            // getting the new global rating after an eventual new rating
            globalRating = DragonServiceFactory.getInstance().getTechServiceManager()
                    .updateGlobalRating(newTechService.getId());
            newTechService.setGlobalRating(globalRating);

        } catch (DeploymentException e) {
            NonLocalizedError error = new NonLocalizedError("Can't update service.",
                    e.getMessage(), e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        techService = newTechService;
        return "success";
    }

    /**
     * Creates a new techservice object with current attributes
     */
    public String create() throws IError {

        TechnicalServiceTO newTechService = new TechnicalServiceTO();

        newTechService.setName(name);
        newTechService.setPurpose(purpose);

        try {
            techService.setId(DragonServiceFactory.getInstance().getTechServiceManager()
                    .createTechService(newTechService));
        } catch (DeploymentException e) {
            NonLocalizedError error = new NonLocalizedError("Can't create service.",
                    e.getMessage(), e);
            error.setType(IError.FUNCTIONAL_ERROR);
            throw error;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    /**
     * Deletes the current endpoint from the database
     */
    public String delete() throws LocalizedError {
        try {
            DragonServiceFactory.getInstance().getTechServiceManager().removeTechService(
                    techService.getId());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    /**
     * Resets changes made to the current endpoint attributes.
     */
    public void reset() {

        name = techService.getName();
        purpose = techService.getPurpose();
        categories = techService.getCategories();

        // Added By TIAR Abderrahmane
        myRating = techService.getRatingTO().getMyRating();
        globalRating = techService.getGlobalRating();

        comments = techService.getComments();
        tags = techService.getTags();
        // End
    }

    /**
     * Clears the endpoint.
     */
    public void clear() {
        techService = new TechnicalServiceTO();
        reset();
    }

    /**
     * Clears the add wsdl attributes.
     */
    public void clearAddWsdl() {
        fileDescriptionURL = null;
        fileDescriptionName = null;
        storeEntities = false;
        wsdlDocContent = null;
        wsdlDocFileName = null;
    }

    /**
     * Validate the endpoint
     */
    public void validate() {

    }

    public void loadPersonRoles() throws OrganizationException, DragonDelegateException {
        personRoles = DragonServiceFactory.getInstance().getRoleOfPartyManager()
                .getPersonRolesByTechService(techService.getId(),
                        createSortOption(SortCriteria.LINK_TYPE, true));
    }

    public void loadOrgRoles() throws OrganizationException, DragonDelegateException {
        organizationRoles = DragonServiceFactory.getInstance().getRoleOfPartyManager()
                .getOrgRolesByTechService(techService.getId(),
                        createSortOption(SortCriteria.LINK_TYPE, true));
    }

    /**
     * Delete a role of party
     * 
     * @return
     * @throws LocalizedError
     */
    public String deleteRole(HttpServletRequest request) throws LocalizedError {
        String roleToDeleteId = request.getParameter("roleToDeleteId");
        try {
            DragonServiceFactory.getInstance().getRoleOfPartyManager().removeRole(roleToDeleteId);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        load(request);
        return "success";
    }

    public void loadRelatedDoc(HttpServletRequest request, HttpServletResponse response,
            String relatedDocId, String contentType) throws LocalizedError, IOException {

        InputStream input = null;
        try {
            input = DragonServiceFactory.getInstance().getTechServiceManager()
                    .getRelatedDocContent(relatedDocId);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        OutputStream output = response.getOutputStream();

        response.setContentType(contentType);
        response.setHeader("Cache-Control", "no-cache"); // HTTP 1.1
        response.setHeader("Pragma", "no-cache"); // HTTP 1.0
        response.setDateHeader("Expires", 0); // prevents caching at the proxy

        byte[] buffer = new byte[512];
        int len = 0;
        while ((len = input.read(buffer)) >= 0) {
            output.write(buffer, 0, len);
        }
        output.flush();
        output.close();
    }

    public void loadWSDLDesc(HttpServletRequest request, HttpServletResponse response,
            String serviceSpecId) throws LocalizedError, IOException {

        InputStream input = null;
        try {
            input = DragonServiceFactory.getInstance().getWSDLImporter().getWsdlDesc(serviceSpecId);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        OutputStream output = response.getOutputStream();

        response.setContentType("text/xml");
        response.setHeader("Cache-Control", "no-cache"); // HTTP 1.1
        response.setHeader("Pragma", "no-cache"); // HTTP 1.0
        response.setDateHeader("Expires", 0); // prevents caching at the proxy

        byte[] buffer = new byte[512];
        int len = 0;
        while ((len = input.read(buffer)) >= 0) {
            output.write(buffer, 0, len);
        }
        output.flush();
        output.close();
    }

    private RequestOptionsTO createSortOption(SortCriteria sortOn, boolean ascendingly) {
        RequestOptionsTO requestOptionsTO = new RequestOptionsTO();
        requestOptionsTO.setSortAscendingly(ascendingly);
        requestOptionsTO.setSortCriteria(sortOn);
        return requestOptionsTO;
    }

    // ==========================================================
    // === Attributes Getters & Setters
    // ==========================================================
    public String getId() {
        if (techService == null)
            return "";
        return techService.getId();
    }

    public List<PersonToTechServiceTO> getPersonRoles() {
        return personRoles;
    }

    public List<OrgToTechServiceTO> getOrganizationRoles() {
        return organizationRoles;
    }

    /**
     * @return the name
     */
    public String getName() {
        return name;
    }

    /**
     * @param name
     *            the name to set
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * @return the purpose
     */
    public String getPurpose() {
        return purpose;
    }

    /**
     * @param purpose
     *            the purpose to set
     */
    public void setPurpose(String purpose) {
        this.purpose = purpose;
    }

    public List<EndpointTO> getEndpoints() {
        return endpoints;
    }

    public void setEndpoints(List<EndpointTO> endpoints) {
        this.endpoints = endpoints;
    }

    public List<ServiceSpecificationTO> getSpecsTO() {
        return specsTO;
    }

    public void setSpecsTO(List<ServiceSpecificationTO> specsTO) {
        this.specsTO = specsTO;
    }

    public List<SimpleFileTO> getRelatedDocsTO() {
        return relatedDocsTO;
    }

    public void setRelatedDocsTO(List<SimpleFileTO> relatedDocsTO) {
        this.relatedDocsTO = relatedDocsTO;
    }

    public TechnicalServiceTO getTechService() {
        return techService;
    }

    public void setTechService(TechnicalServiceTO techService) {
        this.techService = techService;
    }

    public List<String> getSelectedCategories() {
        return selectedCategories;
    }

    public void setSelectedCategories(List<String> selectedCategories) {
        this.selectedCategories = selectedCategories;
    }

    public List<KeyedRefTO> getCategories() {
        return categories;
    }

    public void setCategories(List<KeyedRefTO> categories) {
        this.categories = categories;
    }

    /**
     * @return the fileDescriptionURL
     */
    public String getFileDescriptionURL() {
        return fileDescriptionURL;
    }

    /**
     * @param fileDescriptionURL
     *            the fileDescriptionURL to set
     */
    public void setFileDescriptionURL(String fileDescriptionURL) {
        this.fileDescriptionURL = fileDescriptionURL;
    }

    /**
     * @return the fileDescriptionName
     */
    public String getFileDescriptionName() {
        return fileDescriptionName;
    }

    /**
     * @param fileDescriptionName
     *            the fileDescriptionName to set
     */
    public void setFileDescriptionName(String fileDescriptionName) {
        this.fileDescriptionName = fileDescriptionName;
    }

    /**
     * @return the storeEntities
     */
    public boolean isStoreEntities() {
        return storeEntities;
    }

    /**
     * @param storeEntities
     *            the storeEntities to set
     */
    public void setStoreEntities(boolean storeEntities) {
        this.storeEntities = storeEntities;
    }

    private String extractShortFilename(IFileUploadPart file) {
        String fileName = file.getFileName();
        if (fileName != null) {
            if (fileName.contains("/")) {
                // it's a full file name, so extract only the local part
                String[] parts = fileName.split("/");
                fileName = parts[parts.length - 1];
            } else if (fileName.contains("\\")) {
                // it's a full file name, so extract only the local part
                String[] parts = fileName.split("\\u005C+");
                fileName = parts[parts.length - 1];
            }

        }
        return fileName;
    }

}
