/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * ListOfPersonsBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.organization;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.SortCriteria;
import org.ow2.dragon.api.to.organization.PersonSearchProperties;
import org.ow2.dragon.api.to.organization.PersonTO;
import org.ow2.dragon.ui.businessdelegate.factory.DragonServiceFactory;
import org.ow2.dragon.ui.utils.StringHelper;
import org.ow2.opensuit.core.error.IError;
import org.ow2.opensuit.core.error.LocalizedError;

/**
 * This bean holds a list of persons.
 * 
 * It is initialized either by loading all persons from the database, or by
 * searching persons by name.
 * 
 * <p>
 * This version of the bean performs the search when loading, then lets the
 * framework manage sort and pagination. This design may only be used if you're
 * sure the amount of data is "reasonable" and under control.
 * 
 * @author ambarthe, ofabre
 */
public class ListOfPersonsBean {

    private Logger logger = Logger.getLogger(this.getClass());

    // ==========================================================
    // === Attributes
    // ==========================================================
    private String organizationId;

    List<PersonTO> persons;

    private String searchCriteria;

    private List<String> searchedProperties;

    // ==========================================================
    // === Control Methods
    // ==========================================================
    /**
     * Loads all persons from the database
     */
    public void loadAll() throws LocalizedError {
        try {
            persons = DragonServiceFactory.getInstance().getPersonManager().getAllPersons(
                    createSortOption(SortCriteria.PERSON_LASTNAME, true));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    /**
     * Loads all the persons of an organization from the database
     */
    public void loadByOrganization(String orgId) throws LocalizedError {
        this.organizationId = orgId;
        try {
            persons = DragonServiceFactory.getInstance().getPersonManager()
                    .getPersonsByOrganization(organizationId,
                            createSortOption(SortCriteria.PERSON_LASTNAME, true));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    public List<PersonTO> getPersonsToLink() {
        List<PersonTO> personsToLink = new ArrayList<PersonTO>();
        if (persons != null && !persons.isEmpty()) {
            personsToLink = persons;
        } else {
            PersonTO emptyPerson = new PersonTO();
            emptyPerson.setId("");
            emptyPerson.setLastname("No person to link");
            personsToLink.add(emptyPerson);
        }
        return personsToLink;
    }

    /**
     * Searches persons by firstname, lastname, location, organisation, post
     */
    public void search(HttpServletRequest request) throws LocalizedError {
        boolean all = Boolean.parseBoolean(request.getParameter("all"));
        if (all) {
            loadAll();
        } else {
            String searchCriteria = request.getParameter("searchCriteria");
            String searchedPropertiesParam = request.getParameter("searchedProperties");
            List<String> searchedProperties = StringHelper
                    .urlParameterToArray(searchedPropertiesParam);
            List<PersonSearchProperties> properties = convertSearchedProperties(searchedProperties);
            try {
                persons = DragonServiceFactory.getInstance().getPersonManager().searchPersons(
                        searchCriteria, properties,
                        createSortOption(SortCriteria.PERSON_LASTNAME, true));
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
                throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
            }

            if (persons == null || persons.size() == 0) {
                LocalizedError error = new LocalizedError("search_person_no_result");
                // NonLocalizedError eroor = new NonLocalizedError("",)
                error.setType(IError.FUNCTIONAL_ERROR);
                // error.setStackTrace(null);
                throw error;
            }
        }
    }

    private List<PersonSearchProperties> convertSearchedProperties(List<String> searchedProperties2) {
        List<PersonSearchProperties> result = new ArrayList<PersonSearchProperties>();
        for (String prop : searchedProperties2) {
            PersonSearchProperties propEnum = PersonSearchProperties.fromString(prop);
            if (propEnum != null) {
                result.add(propEnum);
            }
        }
        return result;
    }

    private RequestOptionsTO createSortOption(SortCriteria sortOn, boolean ascendingly) {
        RequestOptionsTO requestOptionsTO = new RequestOptionsTO();
        requestOptionsTO.setSortAscendingly(ascendingly);
        requestOptionsTO.setSortCriteria(sortOn);
        return requestOptionsTO;
    }

    // ==========================================================
    // === Attributes Getters & Setters
    // ==========================================================

    /**
     * @return the name
     */

    /**
     * @return the searchCriteria
     */
    public String getSearchCriteria() {
        return searchCriteria;
    }

    public List<PersonTO> getPersons() {
        return persons;
    }

    /**
     * @param searchCriteria
     *            the searchCriteria to set
     */
    public void setSearchCriteria(String searchCriteria) {
        this.searchCriteria = searchCriteria;
    }

    public List<String> getSearchedProperties() {
        return searchedProperties;
    }

    public void setSearchedProperties(List<String> searchedProperties) {
        this.searchedProperties = searchedProperties;
    }
}
