/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * ListOfPostsBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.organization;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.SortCriteria;
import org.ow2.dragon.api.to.organization.PostSearchProperties;
import org.ow2.dragon.api.to.organization.PostTO;
import org.ow2.dragon.ui.businessdelegate.factory.DragonServiceFactory;
import org.ow2.dragon.ui.utils.StringHelper;
import org.ow2.opensuit.core.error.IError;
import org.ow2.opensuit.core.error.LocalizedError;

/**
 * This bean holds a list of posts.
 * 
 * It is initialized either by loading all posts from the database, or by
 * searching directors by keywords on various fields.
 * 
 * <p>
 * This version of the bean performs the search when loading, then lets the
 * framework manage sort and pagination. This design may only be used if you're
 * sure the amount of data is "reasonable" and under control.
 * 
 * @author ambarthe, ofabre
 */
public class ListOfPostsBean {

    private Logger logger = Logger.getLogger(this.getClass());

    // ==========================================================
    // === Attributes
    // ==========================================================
    private List<PostTO> posts;

    private List<PostTO> freePosts;

    private List<PostTO> orgPosts;

    private String searchCriteria;

    private List<String> searchedProperties;

    // ==========================================================
    // === Control Methods
    // ==========================================================
    /**
     * Loads all posts from the database
     */
    public void loadAll() throws LocalizedError {
        try {
            posts = DragonServiceFactory.getInstance().getPostManager().getAllPosts(
                    createSortOption(SortCriteria.POST_NAME, true));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    /**
     * Loads all posts from the database
     */
    public void loadByOrganization(String orgId) throws LocalizedError {
        try {
            posts = DragonServiceFactory.getInstance().getOrganizationManager()
                    .getPostsByOrganization(orgId, createSortOption(SortCriteria.POST_NAME, true));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    public void deletePost(String postId, HttpServletRequest request) throws LocalizedError {
        try {
            DragonServiceFactory.getInstance().getPostManager().removePost(postId);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        search(request);
    }

    public String postInUse(PostTO post) throws LocalizedError {
        return post.isPostInUse() ? "yes" : "no";
    }

    /**
     * Searches post by business name, nature, description
     */
    public void search(HttpServletRequest request) throws LocalizedError {
        boolean all = Boolean.parseBoolean(request.getParameter("all"));
        if (all) {
            loadAll();
        } else {
            String searchCriteria = request.getParameter("searchCriteria");
            String searchedPropertiesParam = request.getParameter("searchedProperties");
            List<String> searchedProperties = StringHelper
                    .urlParameterToArray(searchedPropertiesParam);
            List<PostSearchProperties> properties = convertSearchedProperties(searchedProperties);
            try {
                posts = DragonServiceFactory.getInstance().getPostManager().searchPost(
                        searchCriteria, properties, createSortOption(SortCriteria.POST_NAME, true));
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
                throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
            }

            if (posts == null || posts.size() == 0) {
                LocalizedError error = new LocalizedError("search_post_no_result");
                // NonLocalizedError eroor = new NonLocalizedError("",)
                error.setType(IError.FUNCTIONAL_ERROR);
                // error.setStackTrace(null);
                throw error;
            }
        }
    }

    private List<PostSearchProperties> convertSearchedProperties(List<String> searchedProperties2) {
        List<PostSearchProperties> result = new ArrayList<PostSearchProperties>();
        for (String prop : searchedProperties2) {
            PostSearchProperties propEnum = PostSearchProperties.fromString(prop);
            if (propEnum != null) {
                result.add(propEnum);
            }
        }
        return result;
    }

    /**
     * Loads all posts from the database
     */
    public void loadFree(HttpServletRequest request) throws LocalizedError {
        String orgId = request.getParameter("organizationId");
        try {
            posts = DragonServiceFactory.getInstance().getPostManager()
                    .getPostsNotLinkedToOrganization(orgId,
                            createSortOption(SortCriteria.POST_NAME, true));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
    }

    public List<PostTO> getPostsToAdd() {
        List<PostTO> postsToAdd = new ArrayList<PostTO>();
        if (posts != null && !posts.isEmpty()) {
            postsToAdd = posts;
        } else {
            PostTO emptyPost = new PostTO();
            emptyPost.setId("");
            emptyPost.setName("No post to add");
            postsToAdd.add(emptyPost);
        }
        return postsToAdd;
    }

    /**
     * TODO to be replaced by a more elegant solution
     * 
     * @return
     */
    public void noOrgSelected() {
        PostTO emptyPost = new PostTO();
        emptyPost.setName("Select an org first");
        List<PostTO> result = new ArrayList<PostTO>();
        result.add(emptyPost);
        posts = result;
    }

    private RequestOptionsTO createSortOption(SortCriteria sortOn, boolean ascendingly) {
        RequestOptionsTO requestOptionsTO = new RequestOptionsTO();
        requestOptionsTO.setSortAscendingly(ascendingly);
        requestOptionsTO.setSortCriteria(sortOn);
        return requestOptionsTO;
    }

    // ==========================================================
    // === Attributes Getters & Setters
    // ==========================================================

    public List<PostTO> getPosts() {
        return posts;
    }

    public List<PostTO> getFreePosts() {
        return freePosts;
    }

    public List<PostTO> getOrgPosts() {
        return orgPosts;
    }

    /**
     * @return the searchCriteria
     */
    public String getSearchCriteria() {
        return searchCriteria;
    }

    /**
     * @param searchCriteria
     *            the searchCriteria to set
     */
    public void setSearchCriteria(String searchCriteria) {
        this.searchCriteria = searchCriteria;
    }

    public List<String> getSearchedProperties() {
        return searchedProperties;
    }

    public void setSearchedProperties(List<String> searchedProperties) {
        this.searchedProperties = searchedProperties;
    }

}
