/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * AgreementBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.sla;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.xml.namespace.QName;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.to.deployment.BindingOperationTO;
import org.ow2.dragon.api.to.deployment.EndpointTO;
import org.ow2.dragon.ui.businessdelegate.factory.DragonServiceFactory;
import org.ow2.dragon.ui.uibeans.deployment.EndpointBean;
import org.ow2.dragon.ui.uibeans.deployment.TechnicalServiceBean;
import org.ow2.opensuit.core.error.LocalizedError;

import com.ebmwebsourcing.addressing.addressing4agreement.monitoring.api.EndpointReferenceType;
import com.ebmwebsourcing.addressing.addressing4agreement.monitoring.impl.WSAddressing4AgreementFactory;
import com.ebmwebsourcing.agreement.definition.api.Agreement;
import com.ebmwebsourcing.agreement.definition.api.AgreementTemplate;
import com.ebmwebsourcing.agreement.definition.api.GuaranteeTerms;
import com.ebmwebsourcing.wsstar.agreement.service.impl.AgreementServiceImpl;

/**
 * 
 * @author nsaladge - eBM Websourcing
 *
 */
public class AgreementBean {

	private Logger logger = Logger.getLogger(this.getClass());

	// ==========================================================
	// === Attributes
	// ==========================================================
	private Agreement agreementTO;

	private ManagedAgreementBean mab;

	private String name;

	private String clientEndpoint;

	private String providerEndpoint;

	private Date expirationTime;

	private ServiceDescriptionTermBean serviceDescriptionBean = new ServiceDescriptionTermBean();

	private ServiceReferenceBean serviceReferenceBean = new ServiceReferenceBean();

	private ServicePropertiesBean servicePropertiesBean = new ServicePropertiesBean();

	private List<GuaranteeBean> guaranteesBean = new ArrayList<GuaranteeBean>();

	private String templateId;

	/**
	 * selectedProvider: the selected provider name of the endpoint
	 */
	private String selectedProvider;

	/**
	 * selectedTemplate: the selected template 
	 */
	private String selectedTemplate = null;


	public AgreementBean() {

	}

	public AgreementBean(Agreement agTO, ManagedAgreementBean mab) throws LocalizedError {
		agreementTO = agTO;
		this.mab = mab;

		serviceDescriptionBean.setSdtTO(agreementTO.getServiceDescription());
		serviceDescriptionBean.setMab(this.mab);


		serviceReferenceBean.setSrTO(agreementTO.getServiceReference());
		serviceReferenceBean.setMab(this.mab);


		servicePropertiesBean.setSpTO(agreementTO.getServiceProperties());
		servicePropertiesBean.setMab(this.mab);

		initGuaranteesBean();
		reset();

	}

	public void initGuaranteesBean() throws LocalizedError {
		this.guaranteesBean = new ArrayList<GuaranteeBean>();
		for (GuaranteeTerms g : agreementTO.getGuaranteeTerms()) {
			this.guaranteesBean.add(new GuaranteeBean(g, this.mab));
		}
	}

	public Date getExpirationTime() {
		Date res = null;
		if (expirationTime != null) {
			res = (Date) expirationTime.clone();
		}
		return res;
	}

	public void setExpirationTime(Date expirationTime) {
		if (expirationTime != null) {
			this.expirationTime = (Date) expirationTime.clone();
		} else {
			this.expirationTime = null;
		}
	}

	// ==========================================================
	// === Control Methods
	// ==========================================================

	/**
	 * Saves the current techService with changes.
	 */
	public String save() throws LocalizedError {
		try { 
			agreementTO.setName(this.name);

			EndpointReferenceType eprClient = WSAddressing4AgreementFactory.newInstance().newEndpointReferenceType();
			eprClient.setAddress(this.clientEndpoint);
			agreementTO.getContext().setInitiator(eprClient);

			EndpointReferenceType eprProvider = WSAddressing4AgreementFactory.newInstance().newEndpointReferenceType();
			eprProvider.setAddress(this.providerEndpoint);
			agreementTO.getContext().setResponder(eprProvider);

			if(this.expirationTime != null) {
				agreementTO.getContext().setExpirationTime(expirationTime);
			}

			this.serviceDescriptionBean.save();
			this.serviceReferenceBean.save();
			this.servicePropertiesBean.save();

			for (GuaranteeBean g : this.guaranteesBean) {
				g.save();
			}
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
		}
		return "success";
	}

	/**
	 * Creates an object with current attributes
	 */
	public String create(String type, ManagedAgreementBean template, String agreementName, String clientName, String providerName,
			String clientEndpointName, Date expirationTime, TechnicalServiceBean techServiceBean,
			EndpointBean endpointBean) throws LocalizedError {
		try {
			if(type.equals("template")) {
				agreementTO = new AgreementServiceImpl().createProviderAgreementTemplate();
			} else if(type.equals("offer")) {
				if(template == null) {
					agreementTO = new AgreementServiceImpl().createClientAgreement();
				} 
			}

			if(template != null) {
				agreementTO = new AgreementServiceImpl().createClientAgreementFromProviderTemplate((AgreementTemplate) template.getAgreementBean().getAgreementTO());
			} else {
				agreementTO.setContext(agreementTO.newContext());
				agreementTO.setServiceDescription(agreementTO.newServiceDescription());
				agreementTO.setServiceReference(agreementTO.newServiceReference());
				agreementTO.setServiceProperties(agreementTO.newServiceProperties());
			}
			
			EndpointReferenceType eprClient = WSAddressing4AgreementFactory.newInstance().newEndpointReferenceType();
			agreementTO.getContext().setInitiator(eprClient);

			EndpointReferenceType eprProvider = WSAddressing4AgreementFactory.newInstance().newEndpointReferenceType();
			agreementTO.getContext().setResponder(eprProvider);
			
			reset();

			agreementTO.setName(agreementName);



			eprClient.setAddress(clientEndpointName);
			eprProvider.setAddress(endpointBean.getName());

			if(expirationTime != null) {
				agreementTO.getContext().setExpirationTime(expirationTime);
			}

			serviceDescriptionBean.setSdtTO(agreementTO.getServiceDescription());
			serviceDescriptionBean.setMab(this.mab);
			serviceDescriptionBean.setName(agreementName + "_ServiceDescription");
			serviceDescriptionBean.setProviderName(providerName);
			serviceDescriptionBean.setServiceName(techServiceBean.getName());
			serviceDescriptionBean.setWsdlURL(techServiceBean.getWsdlUrl());
			serviceDescriptionBean.save();

			serviceReferenceBean.setSrTO(agreementTO.getServiceReference());
			serviceReferenceBean.setMab(this.mab);
			serviceReferenceBean.setName(agreementName + "_ServiceReference");
			serviceReferenceBean.setServiceName(techServiceBean.getName());
			serviceReferenceBean.save();

			servicePropertiesBean.setSpTO(agreementTO.getServiceProperties());
			servicePropertiesBean.setMab(this.mab);
			servicePropertiesBean.setName(agreementName + "_ServiceProperties");
			servicePropertiesBean.setServiceName(techServiceBean.getName());
			servicePropertiesBean.save();

		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
		}
		return "success";
	}

	public String createClientEnpoint(String clientName, String endpointName) {
		String res = null;
		res = clientName + "_" + endpointName + "_clientEndpoint";
		return res;
	}

	/**
	 * Deletes the current endpoint from the database
	 */
	public String delete() throws LocalizedError {

		this.mab.delete();

		return "success";
	}

	/**
	 * Resets changes made to the current endpoint attributes.
	 * @throws LocalizedError 
	 */
	public void reset() throws LocalizedError {
		try {
			if (agreementTO != null) {
				this.name = agreementTO.getName();

				if(agreementTO instanceof AgreementTemplate) {
					this.templateId = ((AgreementTemplate)agreementTO).getTemplateId();
				}

				this.clientEndpoint = agreementTO.getContext().getInitiator().getAddress();
				this.providerEndpoint = agreementTO.getContext().getResponder().getAddress();
				this.expirationTime = agreementTO.getContext().getExpirationTime();
				this.serviceDescriptionBean.reset();
				this.serviceReferenceBean.reset();
				this.servicePropertiesBean.reset();
				for (GuaranteeBean g : this.guaranteesBean) {
					g.reset();
				}
			}
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
		}
	}

	/**
	 * @return the name
	 */
	public String getName() {
		return name;
	}

	/**
	 * @param name
	 *            the name to set
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * @return the initiator
	 */
	public String getClientEndpoint() {
		return this.clientEndpoint;
	}

	/**
	 * @param initiator
	 *            the initiator to set
	 */
	public void setClientEndpoint(String initiator) {
		this.clientEndpoint = initiator;
	}

	/**
	 * @return the responder
	 */
	public String getProviderEndpoint() {
		return this.providerEndpoint;
	}

	/**
	 * @param responder
	 *            the responder to set
	 */
	public void setProviderEndpoint(String responder) {
		this.providerEndpoint = responder;
	}

	public ServiceDescriptionTermBean getServiceDescriptionBean() {
		return serviceDescriptionBean;
	}

	public ServiceReferenceBean getServiceReferenceBean() {
		return serviceReferenceBean;
	}

	public ServicePropertiesBean getServicePropertiesBean() {
		return servicePropertiesBean;
	}

	/**
	 * @return the guaranteeBean
	 */
	public List<GuaranteeBean> getGuaranteesBean() {
		return guaranteesBean;
	}

	/**
	 * @param guaranteeBean
	 *            the guaranteeBean to set
	 */
	public void setGuaranteesBean(List<GuaranteeBean> guaranteeBean) {
		this.guaranteesBean = guaranteeBean;
	}

	/**
	 * @return the agreementTO
	 */
	public Agreement getAgreementTO() {
		return agreementTO;
	}

	/**
	 * @return the selectedProvider
	 */
	public String getSelectedProvider() {
		return selectedProvider;
	}

	/**
	 * @param selectedProvider
	 *            the selectedProvider to set
	 */
	public void setSelectedProvider(String selectedProvider) {
		this.selectedProvider = selectedProvider;
	}

	public List<String> getAllProviderOperations() throws LocalizedError {
		List<String> res = new ArrayList<String>();
		try {
			if (this.serviceReferenceBean != null) {
				EndpointTO ep = DragonServiceFactory.getInstance().getTechServiceManager()
				.getEndpoint(this.serviceReferenceBean.getServiceName(),
						this.agreementTO.getContext().getResponder().getAddress());
				if (ep != null) {
					res.add("ALL-OPERATIONS");
					List<BindingOperationTO> bops = ep.getBinding().getBindingOps();
					for (BindingOperationTO operationTO : bops) {
						String operationName = null;
						try {
							QName opq = QName.valueOf(operationTO.getSignature());
							operationName = opq.getLocalPart();
						} catch (IllegalArgumentException e) {
							operationName = operationTO.getSignature();
						}
						res.add(operationName);
					}
				}
			}
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
		}
		return res;
	}

	public String getTemplateId() {
		return templateId;
	}

	public void setTemplateId(String templateId) {
		this.templateId = templateId;
	}

	public String getSelectedTemplate() {
		return selectedTemplate;
	}

	public void setSelectedTemplate(String selectedTemplate) {
		this.selectedTemplate = selectedTemplate;
	}

}
