/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * GuaranteeBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.sla;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.apache.log4j.Logger;
import org.ow2.opensuit.core.error.LocalizedError;

import com.ebmwebsourcing.agreement.definition.api.GuaranteeTerms;
import com.ebmwebsourcing.agreement.definition.api.ServiceScope;
import com.ebmwebsourcing.wsstar.agreement.definition.org.ggf.schemas.graap.agreement.ServiceRoleType;

/**
 * 
 * @author nsaladge - eBM Websourcing
 * 
 */
public class GuaranteeBean {

	private Logger logger = Logger.getLogger(this.getClass());

	// ==========================================================
	// === Attributes
	// ==========================================================

	private static final List<String> listObligated = new ArrayList<String>(Arrays
			.asList(new String[] { ServiceRoleType.SERVICE_PROVIDER.value(), ServiceRoleType.SERVICE_CONSUMER.value() }));

	private GuaranteeTerms gTO;

	private ManagedAgreementBean mab;

	private String name;

	private String obligated;

	private ServiceLevelObjectiveBean sloBean = new ServiceLevelObjectiveBean();;

	private BusinessValueBean businessBean = new BusinessValueBean();

	private List<ServiceSelectorBean> serviceScopesBean = new ArrayList<ServiceSelectorBean>();

	private QualifyingConditionBean qualifyingConditionBean = new QualifyingConditionBean();

	private List<String> selectedOperations = new ArrayList<String>();

	/**
	 * @return the selectedOperations
	 */
	public List<String> getSelectedOperations() {
		if (selectedOperations == null) {
			this.selectedOperations = new ArrayList<String>();
		}
		return selectedOperations;
	}

	/**
	 * @param selectedOperations
	 *            the selectedOperations to set
	 */
	public void setSelectedOperations(List<String> selectedOperations) {
		this.selectedOperations = selectedOperations;
	}

	/**
	 * Default constructor
	 * 
	 */
	public GuaranteeBean() {
	}

	/**
	 * Constructor
	 * @throws LocalizedError 
	 */
	public GuaranteeBean(GuaranteeTerms gTO, ManagedAgreementBean mab) throws LocalizedError {
		this.gTO = gTO;
		this.mab = mab;

		this.businessBean.setBvTO(gTO.getBusinessValueList());
		this.businessBean.setMab(this.mab);
		this.businessBean.setGb(this);

		this.qualifyingConditionBean.setQcTO(gTO.getQualifyingCondition());
		this.qualifyingConditionBean.setMab(this.mab);
		this.qualifyingConditionBean.setGb(this);

		this.sloBean.setSloTO(gTO.getServiceLevelObjective());
		this.sloBean.setMab(this.mab);
		this.sloBean.setGb(this);

		initServiceScopeBean();

		reset();
	}

	private void initServiceScopeBean() {
		serviceScopesBean.clear();
		if(gTO != null) {
			for (ServiceScope sst : gTO.getServiceScopes()) {
				serviceScopesBean.add(new ServiceSelectorBean(sst, this, this.mab));
			}
		}
	}


	// ==========================================================
	// === Control Methods
	// ==========================================================

	/**
	 * Loads a technical service from the request parameter 'id'.
	 */
	public void init(String idGuarantee, ManagedAgreementBean mab) throws LocalizedError {


		this.mab = mab;
		if(!idGuarantee.equals("-1")) {
			try {
				if (idGuarantee != null) {
					if(this.mab.getAgreementBean().getGuaranteesBean() != null) {
						Iterator<GuaranteeBean> it = this.mab.getAgreementBean().getGuaranteesBean().iterator();
						while(it.hasNext()) {
							GuaranteeBean g = it.next();
							if(g.getName().equals(idGuarantee)) {
								this.gTO = g.getGTO();
								this.mab.getAgreementBean().getGuaranteesBean().remove(g);
								this.mab.getAgreementBean().getGuaranteesBean().add(this);
								break;
							}
						}
					}
				}


				if(this.gTO == null) {
					throw new LocalizedError("Impossible to find the guarantee named " + idGuarantee);
				}
			} catch (Exception e) {
				logger.error(e.getMessage(), e);
				throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
			}
			initServiceScopeBean();

			this.businessBean.setBvTO(gTO.getBusinessValueList());
			this.businessBean.setMab(this.mab);
			this.businessBean.setGb(this);

			this.qualifyingConditionBean.setQcTO(gTO.getQualifyingCondition());
			this.qualifyingConditionBean.setMab(this.mab);
			this.qualifyingConditionBean.setGb(this);

			this.sloBean.setSloTO(gTO.getServiceLevelObjective());
			this.sloBean.setMab(this.mab);
			this.sloBean.setGb(this);
		} else {
			this.gTO = null;
			this.businessBean.setBvTO(null);
			this.businessBean.setMab(this.mab);
			this.businessBean.setGb(this);

			this.qualifyingConditionBean.setQcTO(null);
			this.qualifyingConditionBean.setMab(this.mab);
			this.qualifyingConditionBean.setGb(this);

			this.sloBean.setSloTO(null);
			this.sloBean.setMab(this.mab);
			this.sloBean.setGb(this);

			initServiceScopeBean();

		}
		reset();
	}

	public String save() throws LocalizedError {
		gTO.setName(this.name);
		gTO.setObligated(ServiceRoleType.fromValue(this.obligated));


		this.qualifyingConditionBean.save();
		this.sloBean.save();
		this.businessBean.save();
		for (ServiceSelectorBean s : this.serviceScopesBean) {
			s.save();
		}

		return "success";
	}

	/**
	 * Deletes the current endpoint from the database
	 */
	public String delete(ManagedAgreementBean mab) throws LocalizedError {
		try {
			mab.getAgreementBean().getAgreementTO().getGuaranteeTerms().remove(this.gTO);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
		}
		return "success";
	}

	/**
	 * Deletes the current endpoint from the database
	 */
	public String deleteServiceScope(String name) throws LocalizedError {

		this.selectedOperations.remove(name);
		verifToAddOperationToGuarantee();

		return "success";
	}

	/**
	 * Creates an object with current attributes
	 */
	public String create(ManagedAgreementBean mab) throws LocalizedError {
		try {
			this.mab = mab;

			gTO = mab.getAgreementBean().getAgreementTO().newGuaranteeTerms();

			gTO.setBusinessValueList(gTO.newBusinessValueList());
			this.businessBean.setBvTO(gTO.getBusinessValueList());
			this.businessBean.setMab(this.mab);
			this.businessBean.setGb(this);

			gTO.setQualifyingCondition(gTO.newQualifyingCondition(null, null));
			this.qualifyingConditionBean.setQcTO(gTO.getQualifyingCondition());
			this.qualifyingConditionBean.setMab(this.mab);
			this.qualifyingConditionBean.setGb(this);

			gTO.setServiceLevelObjective(gTO.newServiceLevelObjective());
			gTO.getServiceLevelObjective().setKPITarget(gTO.getServiceLevelObjective().newKPITarget());
			gTO.getServiceLevelObjective().getKPITarget().setConstraint(gTO.getServiceLevelObjective().getKPITarget().newConstraint());
			gTO.getServiceLevelObjective().getKPITarget().getConstraint().setValue(gTO.getServiceLevelObjective().getKPITarget().getConstraint().newValue());
			gTO.getServiceLevelObjective().getKPITarget().getConstraint().getValue().setType(gTO.getServiceLevelObjective().getKPITarget().getConstraint().getValue().newType());
			gTO.getServiceLevelObjective().getKPITarget().getConstraint().getValue().setUnit(gTO.getServiceLevelObjective().getKPITarget().getConstraint().getValue().newUnit());
			this.sloBean.setSloTO(gTO.getServiceLevelObjective());
			this.sloBean.setMab(this.mab);
			this.sloBean.setGb(this);

			verifToAddOperationToGuarantee();
			mab.getAgreementBean().getAgreementTO().addGuarantee(gTO);
			mab.getAgreementBean().getGuaranteesBean().add(this);

			this.save();

			reset();


		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
		}
		return "success";
	}


	/**
	 * @param ag
	 * @throws LocalizedError 
	 */
	public String verifToAddOperationToGuarantee() throws LocalizedError {
		try {
			boolean find = false;

			// verif to add new scope
			for (String operation : this.getSelectedOperations()) {
				find = false;
				for(ServiceSelectorBean ssb: this.serviceScopesBean) {
					if((ssb.getValue() != null)&&(ssb.getValue().equals(operation))) {
						find = true;
					}
				}

				if(!find) {
					// add new scope
					ServiceScope ssTO = this.gTO.newServiceScope();
					ssTO.setScope(operation);
					ssTO.setServiceName(mab.getAgreementBean().getAgreementTO().getServiceReference().getServiceName());

					this.gTO.addServiceScope(ssTO);
					this.serviceScopesBean.add(new ServiceSelectorBean(ssTO, this, this.mab));
				}
			}

			// verif to delete old scope
			Iterator<ServiceSelectorBean> it = this.serviceScopesBean.iterator();
			while(it.hasNext()) {
				ServiceSelectorBean ssb = it.next(); 
				find = false;
				for (String operation : this.getSelectedOperations()) {
					if((ssb.getValue() != null)&&(ssb.getValue().equals(operation))) {
						find = true;
					}
				}

				if(!find) {
					// remove old scope
					this.gTO.removeServiceScope(ssb.getSsTO());
					this.serviceScopesBean.remove(ssb);

					it = this.serviceScopesBean.iterator();
				}
			}
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
		}

		return "success";
	}

	/**
	 * Resets changes made to the current endpoint attributes.
	 * @throws LocalizedError 
	 */
	public void reset() throws LocalizedError {
		if (gTO != null) {
			this.name = gTO.getName();
			if(gTO.getObligated() != null) {
				this.obligated = gTO.getObligated().value();
			}
		} else {
			this.name = null;
			this.obligated = ServiceRoleType.SERVICE_PROVIDER.value();
			this.selectedOperations.clear();
		}
		
		this.qualifyingConditionBean.reset();
		this.sloBean.reset();
		this.businessBean.reset();
		for (ServiceSelectorBean s : this.serviceScopesBean) {
			s.reset();
		}
	}

	public static List<String> getObligatedList() {
		return listObligated;
	}


	/**
	 * @return the name
	 */
	public String getName() {
		return name;
	}

	/**
	 * @param name
	 *            the name to set
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * @return the business
	 */
	public BusinessValueBean getBusinessBean() {
		return businessBean;
	}

	/**
	 * @param business
	 *            the business to set
	 */
	public void setBusinessBean(BusinessValueBean business) {
		this.businessBean = business;
	}

	/**
	 * @return the obligated
	 */
	public String getObligated() {
		return obligated;
	}

	/**
	 * @param obligated
	 *            the obligated to set
	 */
	public void setObligated(String obligated) {
		this.obligated = obligated;
	}

	/**
	 * @return the serviceScope
	 */
	public List<ServiceSelectorBean> getServiceScopesBean() {
		return serviceScopesBean;
	}

	/**
	 * @param serviceScope
	 *            the serviceScope to set
	 */
	public void setServiceScopesBean(List<ServiceSelectorBean> serviceScope) {
		this.serviceScopesBean = serviceScope;
	}

	/**
	 * @return the slo
	 */
	public ServiceLevelObjectiveBean getSloBean() {
		return sloBean;
	}

	/**
	 * @param slo
	 *            the slo to set
	 */
	public void setSloBean(ServiceLevelObjectiveBean slo) {
		this.sloBean = slo;
	}

	/**
	 * @return the gTO
	 */
	public GuaranteeTerms getGTO() {
		return gTO;
	}

	/**
	 * @return the qualifyingConditionBean
	 */
	public QualifyingConditionBean getQualifyingConditionBean() {
		return qualifyingConditionBean;
	}

	/**
	 * @param qualifyingConditionBean
	 *            the qualifyingConditionBean to set
	 */
	public void setQualifyingConditionBean(QualifyingConditionBean qualifyingConditionBean) {
		this.qualifyingConditionBean = qualifyingConditionBean;
	}

	public ManagedAgreementBean getMab() {
		return mab;
	}

	public void setMab(ManagedAgreementBean mab) {
		this.mab = mab;
	}

	public void setGTO(GuaranteeTerms gto) {
		gTO = gto;
	}

}
