/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * ManagedAgreementBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.sla;

import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.service.organization.OrganizationException;
import org.ow2.dragon.api.to.organization.OrganizationUnitTO;
import org.ow2.dragon.api.to.organization.PartyTO;
import org.ow2.dragon.api.to.organization.PersonTO;
import org.ow2.dragon.api.to.sla.ManagedAgreementTO;
import org.ow2.dragon.ui.businessdelegate.DragonDelegateException;
import org.ow2.dragon.ui.businessdelegate.factory.DragonServiceFactory;
import org.ow2.dragon.ui.uibeans.deployment.EndpointBean;
import org.ow2.dragon.ui.uibeans.deployment.TechnicalServiceBean;
import org.ow2.dragon.ui.uibeans.organization.OrganizationBean;
import org.ow2.dragon.ui.uibeans.organization.PartyBean;
import org.ow2.dragon.ui.uibeans.organization.PersonBean;
import org.ow2.opensuit.core.error.LocalizedError;

import com.ebmwebsourcing.agreement.definition.WSAgreementFactory;
import com.ebmwebsourcing.agreement.definition.api.AgreementLifeCycle;
import com.ebmwebsourcing.agreement.definition.api.AgreementLifeCycle.State;
import com.ebmwebsourcing.agreement.definition.api.WSAgreementException;
import com.ebmwebsourcing.easycommons.xml.XMLPrettyPrinter;

/**
 * 
 * @author nsaladge - eBM Websourcing
 * 
 */
public class ManagedAgreementBean {

    private Logger logger = Logger.getLogger(this.getClass());

    private static final List<String> listStates = new ArrayList<String>(Arrays
            .asList(new String[] { State.PENDING.value(), State.PENDING_AND_TERMINATING.value(),
                    State.OBSERVE.value(), State.OBSERVED_AND_TERMINATING.value(),
                    State.REJECTE.value(), State.COMPLETE.value(), State.TERMINATE.value(),
                    State.OFFER_RECEIVE.value() }));

    // ==========================================================
    // === Attributes
    // ==========================================================
    private ManagedAgreementTO managedAgreementTO;

    private String id = "-1";

    private AgreementBean agreementBean = new AgreementBean();;

    private String currentState;

    private Date signedByClient;

    private Date signedByProvider;

    private String slaDescription;

    private PartyBean clientBean;

    private PartyBean providerBean;

    private ManagedAgreementBean nill;

    public ManagedAgreementBean() {

    }

    public ManagedAgreementBean(ManagedAgreementTO magTO) throws LocalizedError,
            OrganizationException {
        try {
            this.managedAgreementTO = magTO;
            agreementBean = new AgreementBean(this.managedAgreementTO.getAgreement(), this);

            loadClientAndProvider();
        } catch (DragonDelegateException e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        reset();

        export();
    }

    // ==========================================================
    // === Control Methods
    // ==========================================================

    /**
     * Loads a technical service from the request parameter 'id'.
     */
    public String load(HttpServletRequest iRequest) throws LocalizedError {

        String idAgreement = iRequest.getParameter("agreementId");

        try {
            if (idAgreement != null) {
                managedAgreementTO = DragonServiceFactory.getInstance().getSLAManager()
                        .getManagedAgreement(idAgreement);

                agreementBean = new AgreementBean(managedAgreementTO.getAgreement(), this);

                
                loadClientAndProvider();

            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        reset();
        return "success";
    }

    /**
     * @throws DragonDelegateException
     * @throws LocalizedError
     * @throws OrganizationException
     */
    private void loadClientAndProvider() throws DragonDelegateException, LocalizedError,
            OrganizationException {
        if (managedAgreementTO.getIdClient() != null) {
            PartyTO clientTO = DragonServiceFactory.getInstance().getPartyManager().getParty(
                    managedAgreementTO.getIdClient());
            if (clientTO instanceof OrganizationUnitTO) {
                clientBean = new OrganizationBean((OrganizationUnitTO) clientTO);
            } else if (clientTO instanceof PersonTO) {
                clientBean = new PersonBean((PersonTO) clientTO);
            }
        }
        if (managedAgreementTO.getIdProvider() != null) {
            PartyTO providerTO = DragonServiceFactory.getInstance().getPartyManager().getParty(
                    managedAgreementTO.getIdProvider());
            if (providerTO instanceof OrganizationUnitTO) {
                providerBean = new OrganizationBean((OrganizationUnitTO) providerTO);
            } else if (providerTO instanceof PersonTO) {
                providerBean = new PersonBean((PersonTO) providerTO);
            }
        }
    }

    /**
     * Saves the current techService with changes.
     */
    public String save() throws LocalizedError {
        try {
            if (managedAgreementTO != null) {
                managedAgreementTO.setAgreement(this.getAgreementBean().getAgreementTO());
                managedAgreementTO.setState(this.currentState);
            }

            this.agreementBean.save();

            DragonServiceFactory.getInstance().getSLAManager().updateManagedAgreement(
                    managedAgreementTO);

           

            this.slaDescription = XMLPrettyPrinter.prettyPrint(WSAgreementFactory.newInstance().newWSAgreementWriter().getDocument(managedAgreementTO.getAgreement()));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }

        return "success";
    }

    public String export() throws LocalizedError {
        try {
            this.slaDescription = XMLPrettyPrinter.prettyPrint(WSAgreementFactory.newInstance().newWSAgreementWriter().getDocument(managedAgreementTO.getAgreement()));

        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    /**
     * Creates an object with current attributes
     */
    public String create(String type, ManagedAgreementBean template, String agreementName,
            PartyBean client, String providerIdOrName, String clientEndpointName,
            Date expirationTime, TechnicalServiceBean techServiceBean, EndpointBean endpointBean)
            throws LocalizedError {

        try {
            PartyTO clientTO = null;
            if (client instanceof OrganizationBean) {
                clientTO = DragonServiceFactory.getInstance().getOrganizationManager()
                        .getOrganization(client.getId());
            } else if (client instanceof PersonBean) {
                clientTO = DragonServiceFactory.getInstance().getPersonManager().getPerson(
                        client.getId());
            }

            String providerName = null;
            PartyTO providerTO = null;
            if (providerIdOrName != null) {
                PartyBean providerBean = null;
                providerTO = DragonServiceFactory.getInstance().getPartyManager().getParty(
                        providerIdOrName);
                if (providerTO instanceof OrganizationUnitTO) {
                    providerBean = new OrganizationBean((OrganizationUnitTO) providerTO);
                } else if (providerTO instanceof PersonTO) {
                    providerBean = new PersonBean((PersonTO) providerTO);
                }
                if (providerBean != null) {
                    providerName = providerBean.getGenericName();
                } else {
                    providerName = providerIdOrName;
                }
            }

            if (providerName == null) {
                providerName = "anonymous_provider";
            }

            managedAgreementTO = new ManagedAgreementTO();
            if (clientTO != null) {
                managedAgreementTO.setIdClient(clientTO.getId());
            }
            if (providerTO != null) {
                managedAgreementTO.setIdProvider(providerTO.getId());
            }
            this.agreementBean.create(type, template, agreementName, client.getGenericName(),
                    providerName, URLEncoder.encode(clientEndpointName, "UTF-8"), expirationTime,
                    techServiceBean, endpointBean);

            managedAgreementTO.setAgreement(this.getAgreementBean().getAgreementTO());

            String id = DragonServiceFactory.getInstance().getSLAManager().createManagedAgreement(
                    managedAgreementTO);

            managedAgreementTO.setId(id);

           
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        reset();
        return "success";
    }

    public String delete() throws LocalizedError {
        try {
            DragonServiceFactory.getInstance().getSLAManager().removeManagedAgreement(
                    managedAgreementTO.getId());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    /**
     * Deletes the current endpoint from the database
     */
    public String deleteMetric(String metricName) throws LocalizedError {
        try {
            for (MetricBean m : this.agreementBean.getServicePropertiesBean().getMetrics()) {
                if (m.getName().equals(metricName)) {
                    this.agreementBean.getAgreementTO().getServiceProperties().removeVariable(
                            m.getMTO());
                    this.agreementBean.getServicePropertiesBean().getMetrics().remove(m);
                    break;
                }
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    /**
     * Deletes the current endpoint from the database
     */
    public String deleteGuarantee(String guaranteeName) throws LocalizedError {
        try {
            for (GuaranteeBean gb : this.agreementBean.getGuaranteesBean()) {
                if (gb.getName().equals(guaranteeName)) {
                    this.agreementBean.getAgreementTO().removeGuarantee(gb.getGTO());
                    this.agreementBean.getGuaranteesBean().remove(gb);
                    break;
                }
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
        }
        return "success";
    }

    /**
     * Resets changes made to the current endpoint attributes.
     * 
     * @throws LocalizedError
     */
    public void reset() throws LocalizedError {
        if (managedAgreementTO != null) {
            this.id = managedAgreementTO.getId();
            this.setCurrentState(managedAgreementTO.getState());
            try {
                this.slaDescription = XMLPrettyPrinter.prettyPrint(WSAgreementFactory.newInstance().newWSAgreementWriter().getDocument(managedAgreementTO.getAgreement()));
            } catch (WSAgreementException e) {
                logger.error(e.getMessage(), e);
                throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
            }

            this.agreementBean.reset();
        }
    }

    // ==========================================================
    // === Attributes Getters & Setters
    // ==========================================================
    public String getId() {
        return id;
    }

    /**
     * @param id
     *            the id to set
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * @return the aBean
     */
    public AgreementBean getAgreementBean() {
        return agreementBean;
    }

    /**
     * @param bean
     *            the aBean to set
     */
    public void setAgreementBean(AgreementBean bean) {
        agreementBean = bean;
    }

    /**
     * @return the currentState
     */
    public String getCurrentState() {
        return currentState;
    }

    /**
     * @param currentState
     *            the currentState to set
     */
    public void setCurrentState(String currentState) {
        this.currentState = currentState;
    }

    /**
     * @return the managedAgreementTO
     */
    public ManagedAgreementTO getManagedAgreementTO() {
        return managedAgreementTO;
    }

    /**
     * @param managedAgreementTO
     *            the managedAgreementTO to set
     */
    public void setManagedAgreementTO(ManagedAgreementTO managedAgreementTO) {
        this.managedAgreementTO = managedAgreementTO;
    }

    /**
     * @return the signedByClient
     */
    public Date getSignedByClient() {
        Date res = null;
        if (signedByClient != null) {
            res = (Date) signedByClient.clone();
        }
        return res;
    }

    /**
     * @param signedByClient
     *            the signedByClient to set
     */
    public void setSignedByClient(Date signedByClient) {
        if (signedByClient != null) {
            this.signedByClient = (Date) signedByClient.clone();
        } else {
            this.signedByClient = null;
        }
    }

    /**
     * @return the signedByProvider
     */
    public Date getSignedByProvider() {
        Date res = null;
        if (signedByProvider != null) {
            res = (Date) signedByProvider.clone();
        }
        return res;
    }

    /**
     * @param signedByProvider
     *            the signedByProvider to set
     */
    public void setSignedByProvider(Date signedByProvider) {
        if (signedByProvider != null) {
            this.signedByProvider = (Date) signedByProvider.clone();
        } else {
            this.signedByProvider = null;
        }
    }

    /**
     * @return the slaDescription
     */
    public String getSlaDescription() {
        return slaDescription;
    }

    /**
     * @param slaDescription
     *            the slaDescription to set
     */
    public void setSlaDescription(String slaDescription) {
        this.slaDescription = slaDescription;
    }

    /**
     * @return the listStates
     */
    public static List<String> getListStates() {
        return listStates;
    }

    /**
     * @return the clientBean
     */
    public PartyBean getClientBean() {
        return clientBean;
    }

    /**
     * @param clientBean
     *            the clientBean to set
     */
    public void setClientBean(PartyBean clientBean) {
        this.clientBean = clientBean;
    }

    /**
     * @return the providerBean
     */
    public PartyBean getProviderBean() {
        return providerBean;
    }

    /**
     * @param providerBean
     *            the providerBean to set
     */
    public void setProviderBean(PartyBean providerBean) {
        this.providerBean = providerBean;
    }

    public String isOrganization(PartyBean pb) {
        String res = "person";
        if (pb instanceof OrganizationBean) {
            res = "organization";
        }
        return res;
    }

    public ManagedAgreementBean getNill() {
        return nill;
    }

    public void setNill(ManagedAgreementBean nill) {
        this.nill = nill;
    }

    public void setPending(HttpServletRequest iRequest, Object board) throws LocalizedError {
        this.load(iRequest);
        ((AgreementLifeCycle) this.getAgreementBean().getAgreementTO()).setState(State.PENDING);
        this.managedAgreementTO.setState(State.PENDING.value());
        this.save();

        if (board instanceof AgreementClientBoardBean) {
            ((AgreementClientBoardBean) board).load(iRequest);
        }
        if (board instanceof AgreementProviderBoardBean) {
            ((AgreementProviderBoardBean) board).load(iRequest);
        }
    }

    public void setObserve(HttpServletRequest iRequest, Object board) throws LocalizedError {
        this.load(iRequest);
        ((AgreementLifeCycle) this.getAgreementBean().getAgreementTO()).setState(State.OBSERVE);
        this.managedAgreementTO.setState(State.OBSERVE.value());
        this.save();

        if (board instanceof AgreementClientBoardBean) {
            ((AgreementClientBoardBean) board).load(iRequest);
        }
        if (board instanceof AgreementProviderBoardBean) {
            ((AgreementProviderBoardBean) board).load(iRequest);
        }
    }

    public void setReject(HttpServletRequest iRequest, Object board) throws LocalizedError {
        this.load(iRequest);
        ((AgreementLifeCycle) this.getAgreementBean().getAgreementTO()).setState(State.REJECTE);
        this.managedAgreementTO.setState(State.REJECTE.value());
        this.save();

        if (board instanceof AgreementClientBoardBean) {
            ((AgreementClientBoardBean) board).load(iRequest);
        }
        if (board instanceof AgreementProviderBoardBean) {
            ((AgreementProviderBoardBean) board).load(iRequest);
        }
    }
}
