/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * MetricBean.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.ui.uibeans.sla;

import java.net.URI;
import java.util.Iterator;
import java.util.Properties;

import org.apache.log4j.Logger;
import org.ow2.opensuit.core.error.LocalizedError;

import com.ebmwebsourcing.addressing.addressing4agreement.monitoring.impl.WSAddressing4AgreementFactory;
import com.ebmwebsourcing.agreement.definition.api.Variable;
import com.ebmwebsourcing.wsstar.qml.WSQMLFactory;

/**
 * 
 * @author nsaladge - eBM Websourcing
 * 
 */
public class MetricBean {

	private Logger logger = Logger.getLogger(this.getClass());

	// ==========================================================
	// === Attributes
	// ==========================================================
	private Variable mTO;

	private ManagedAgreementBean mab;

	private String name;

	private String metric;

	private String address;

	private String mode;

	private String operation;

	private String mbeanname;

	private String semanticConcept;

	private String ontology;

	private String dimension;

	private String type;

	private static String OPERATION = "operation";
	private static String DIMENSION = "dimension";
	private static String MODE = "mode";
	private static String MBEAN_NAME = "mbeanName";
	private static String TYPE = "type";
	
	/**
	 * Default constructor
	 * 
	 */
	public MetricBean() {
	}

	/**
	 * Constructor
	 * @throws LocalizedError 
	 */
	public MetricBean(Variable mTO, ManagedAgreementBean mab) throws LocalizedError {
		this.mTO = mTO;
		this.mab = mab;
		reset();
	}

	// ==========================================================
	// === Control Methods
	// ==========================================================

	/**
	 * Loads a technical service from the request parameter 'id'.
	 */
	public void init(String idMetrics, ManagedAgreementBean mab) throws LocalizedError {
		
		this.mab = mab;
		
		try {
			if(!idMetrics.equals("-1")) {
				if(this.mab.getAgreementBean().getServicePropertiesBean() != null) {
					Iterator<MetricBean> it = this.mab.getAgreementBean().getServicePropertiesBean().getMetrics().iterator();
					while(it.hasNext()) {
						MetricBean v = it.next();
						if(v.getName().equals(idMetrics)) {
							this.mTO = v.getMTO();
							this.mab.getAgreementBean().getServicePropertiesBean().getMetrics().remove(v);
							this.mab.getAgreementBean().getServicePropertiesBean().getMetrics().add(this);
							break;
						}
					}
				}
			} else {
				this.mTO = null;
			}
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
		}
		reset();
	}


	public String save() throws LocalizedError {
		try {
			if (mTO != null) {
				mTO.setName(this.name);
				mTO.setMetric(new URI(this.metric));
				mTO.getLocation().getQOSDimension().setSemanticConcept(this.semanticConcept);
				mTO.getLocation().getQOSDimension().setOntology(this.ontology);
				mTO.getLocation().getAddress().setAddress(this.address);

				if(mTO.getLocation().getAddress().getReferenceParameters().getMonitoringPropertiesType() == null) {
					mTO.getLocation().getAddress().getReferenceParameters().setMonitoringPropertiesType(mTO.getLocation().getAddress().getReferenceParameters().newMonitoringPropertiesType());
				}
				
				Properties properties = new Properties();
				properties.setProperty( TYPE, this.type);
				properties.setProperty( MODE, this.mode);
				properties.setProperty( OPERATION, this.operation);
				properties.setProperty( MBEAN_NAME, this.mbeanname);
				properties.setProperty( DIMENSION, this.dimension);
				mTO.getLocation().getAddress().getReferenceParameters().getMonitoringPropertiesType().setProperty(properties);
			}
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
		}

		return "success";
	}


	/**
	 * Creates an object with current attributes
	 */
	public String create(ManagedAgreementBean ag) throws LocalizedError {
		this.mab = ag;
		if (ag.getAgreementBean().getAgreementTO() != null) {
			try {
				mTO = ag.getAgreementBean().getAgreementTO().getServiceProperties().newVariable();
				ag.getAgreementBean().getAgreementTO().getServiceProperties().addVariable(mTO);
				mTO.setLocation(mTO.newLocation());
				mTO.getLocation().setAddress(WSAddressing4AgreementFactory.newInstance().newEndpointReferenceType());
				mTO.getLocation().setQOSDimension(WSQMLFactory.newInstance().newQOSDimension());

				this.save();

				ag.getAgreementBean().getServicePropertiesBean().getMetrics().add(new MetricBean(mTO, ag));
			} catch (Exception e) {
				logger.error(e.getMessage(), e);
				throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
			}
		}
		return "success";
	}

	/**
	 * Resets changes made to the current endpoint attributes.
	 * @throws LocalizedError 
	 */
	public void reset() throws LocalizedError {
		try {
			if (mTO != null) {
				this.name = mTO.getName();
				this.metric = mTO.getMetric().toString();
				if(mTO.getLocation() != null) {
					if(mTO.getLocation().getQOSDimension() != null) {
						this.semanticConcept = mTO.getLocation().getQOSDimension().getSemanticConcept();
						this.ontology = mTO.getLocation().getQOSDimension().getOntology();
					}
					if(mTO.getLocation().getAddress() != null) {
						this.address = mTO.getLocation().getAddress().getAddress();
						if(mTO.getLocation().getAddress().getReferenceParameters() != null) {
							if (mTO.getLocation().getAddress().getReferenceParameters().getMonitoringPropertiesType() != null) {
								
							}
							Properties properties = mTO.getLocation().getAddress().getReferenceParameters().getMonitoringPropertiesType().getProperties();
							this.mbeanname = properties.getProperty(MBEAN_NAME); 
							this.operation = properties.getProperty(OPERATION);
							this.mode = properties.getProperty(MODE);
							this.type = properties.getProperty(TYPE);
							this.dimension = properties.getProperty(DIMENSION);
						}
					}
				}
			} else {
				this.name = null;
				this.metric = null;
				this.semanticConcept = null;
				this.ontology = null;
				this.address = null;
				this.mbeanname = null;
				this.operation = null;
				this.mode = null;
				this.type = null;
				this.dimension = null;
			}
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
		}
	}

	/**
	 * Clears the agreement.
	 * @throws LocalizedError 
	 */
	public void clear() throws LocalizedError {
		try {
			mTO = this.mab.getAgreementBean().getAgreementTO().getServiceProperties().newVariable();
			this.mab.getAgreementBean().getAgreementTO().getServiceProperties().addVariable(mTO);
			mTO.setLocation(mTO.newLocation());
			mTO.getLocation().setAddress(WSAddressing4AgreementFactory.newInstance().newEndpointReferenceType());
			mTO.getLocation().setQOSDimension(WSQMLFactory.newInstance().newQOSDimension());
			reset();
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw new LocalizedError("database_technical", new Object[] { e.getMessage() });
		}
	}

	/**
	 * Validate the endpoint
	 */
	public void validate() {

	}

	/**
	 * @return the name
	 */
	public String getName() {
		return name;
	}

	/**
	 * @param name
	 *            the name to set
	 */
	public void setName(String name) {
		this.name = name;
	}

	public String getMetric() {
		return metric;
	}

	public String getAddress() {
		return address;
	}

	/**
	 * @param location
	 *            the location to set
	 */
	public void setAddress(String location) {
		this.address = location;
	}

	/**
	 * @param metric
	 *            the metric to set
	 */
	public void setMetric(String metric) {
		this.metric = metric;
	}

	public Variable getVariable() {
		return this.mTO;
	}

	public String getSemanticConcept() {
		return semanticConcept;
	}

	public void setSemanticConcept(String semanticConcept) {
		this.semanticConcept = semanticConcept;
	}

	public String getOntology() {
		return ontology;
	}

	public void setOntology(String ontology) {
		this.ontology = ontology;
	}

	public String getOperation() {
		return operation;
	}

	public void setOperation(String operation) {
		this.operation = operation;
	}

	public String getMbeanname() {
		return mbeanname;
	}

	public void setMbeanname(String mbeanname) {
		this.mbeanname = mbeanname;
	}

	public String getMode() {
		return mode;
	}

	public void setMode(String mode) {
		this.mode = mode;
	}

	public Variable getMTO() {
		return mTO;
	}

	public void setMTO(Variable mto) {
		mTO = mto;
	}

	public ManagedAgreementBean getMab() {
		return mab;
	}

	public void setMab(ManagedAgreementBean mab) {
		this.mab = mab;
	}

}
