/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * InputStreamUtil.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

import org.springframework.util.ResourceUtils;

/**
 * @author ofabre - eBM Websourcing
 * 
 */
public class InputStreamUtil {

    private static final String FILE_SCHEME = "file";

    /**
     * Default value is 2048.
     */
    public static final int DEFAULT_BUFFER_SIZE = 2048;

    public static InputStream getInputStream(String resourceLocation) throws IOException,
            URISyntaxException {
        return getInputStream(ResourceUtils.getURL(resourceLocation));
    }

    public static InputStream getInputStream(URI resourceLocation) throws IOException,
            URISyntaxException {
        return getInputStream(ResourceUtils.getURL(resourceLocation.toString()));
    }

    public static InputStream getInputStream(URL resourceLocation) throws IOException,
            URISyntaxException {
        InputStream result = null;
        if (FILE_SCHEME.equals(resourceLocation.getProtocol())) {
            final File f = new File(resourceLocation.toURI());
            result = new FileInputStream(f);
        } else {
            result = ResourceUtils.getURL(resourceLocation.toString()).openStream();
        }
        return result;
    }

    /**
     * Copies information from the input stream to the output stream using a
     * default buffer size of 2048 bytes.
     * 
     * @throws java.io.IOException
     */
    public static void copy(InputStream input, OutputStream output) throws IOException {
        copy(input, output, DEFAULT_BUFFER_SIZE);
    }

    /**
     * Copies information from the input stream to the output stream using the
     * specified buffer size
     * 
     * @throws java.io.IOException
     */
    public static void copy(InputStream input, OutputStream output, int bufferSize)
            throws IOException {
        byte[] buf = new byte[bufferSize];
        int bytesRead = input.read(buf);
        while (bytesRead != -1) {
            output.write(buf, 0, bytesRead);
            bytesRead = input.read(buf);
        }
        output.flush();
    }

    /**
     * Copies information between specified streams and then closes both of the
     * streams.
     * 
     * @throws java.io.IOException
     */
    public static void copyThenClose(InputStream input, OutputStream output) throws IOException {
        copy(input, output);
        input.close();
        output.close();
    }

    /**
     * @returns a byte[] containing the information contained in the specified
     *          InputStream.
     * @throws java.io.IOException
     */
    public static byte[] getBytes(InputStream input) throws IOException {
        ByteArrayOutputStream result = new ByteArrayOutputStream();
        copy(input, result);
        result.close();
        return result.toByteArray();
    }
}
