/***
 * OW2 FraSCAti Factory Maven plugin
 * Copyright (C) 2008-2009 INRIA, USTL
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: frascati@ow2.org 
 *
 * Author(s): Christophe Demarey
 * Contributor(s): Valerio Schiavoni 
 */

package org.ow2.frascati.factory;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.logging.LogManager;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.ow2.frascati.fscript.FrascatiScriptFactory;

/**
 * A maven plugin for SCA Factories.
 * 
 * @requiresDependencyResolution runtime
 * @goal exec
 */
public class FrascatiMojo extends AbstractMojo {
  /**
   * If the flag is true, activate the FrascatiScript presonnality.
   * 
   * @parameter expression="${useFscript}" default-value="false"
   */
  private boolean useFscript;
  /**
   * The path to the logging configuration file.
   * 
   * @parameter expression="${loggingConfFile}" default-value=""
   * @since 1.1
   */
  private String loggingConfFile;
  /**
   * The name of the SCA composite to execute.
   * 
   * @parameter expression="${compositeName}"
   * @required
   * @since 1.0
   */
  private String composite;
  /**
   * The name of the service to execute.
   * 
   * @parameter expression="${serviceName}" default-value=""
   * @since 1.0
   */
  private String service;
  /**
   * The name of the method to execute.
   * 
   * @parameter expression="${methodName}" default-value=""
   * @since 1.0
   */
  private String method;
  /**
   * Space-separated list of method parameters.
   * 
   * @parameter expression="${params}"
   * @since 1.1
   */
  private String methodParams;
  /**
   * List of method parameters with parameter children tags.
   * 
   * @parameter expression="${parameters}"
   * @since 1.1
   */
  private String[] methodParameters;

  /**
   * The Maven project reference.
   * 
   * @parameter expression="${project}"
   * @required
   * @since 1.0
   */
  private MavenProject project;
  
  
  /**
   * Build the classpath from project dependencies and sources.
   * @return A list of URL to load in the FraSCAti classloader.
   */
  private URL[] getClasspath() {
      List<URL> list = new ArrayList<URL>();
      File baseDir = project.getBasedir();
      
      // Add default maven directories to srcs and libs
      try {
          list.add( new File(baseDir.getAbsolutePath() + File.separator + "src/main/java").toURI().toURL() );
          list.add( new File(baseDir.getAbsolutePath() + File.separator + "src/main/resources").toURI().toURL() );
          list.add( new File( baseDir.getAbsolutePath() + File.separator + 
                              "target" + File.separator
                              + project.getArtifactId()+"-"+project.getVersion()+".jar")
                      .toURI().toURL() );
      } catch (MalformedURLException e1) {
          e1.printStackTrace();
      }

      // Add dependencies artifacts
      Set<Artifact> artifacts = project.getDependencyArtifacts();
      for (Artifact artifact : artifacts) {
        if (artifact.getScope().equalsIgnoreCase("test")) {
          continue;
        }
        try {
          list.add(artifact.getFile().toURI().toURL());
        } catch (MalformedURLException e) {
          getLog().warn("Could add dependency " + artifact.getId());
        }
      }

      return list.toArray(new URL[list.size()]);
  }
  
  /**
   * Tells if a MOJO parameter is empty or not.
   * 
   * @param s A string parameter.
   * @return True if the parameter is not filled.
   */
  private static boolean isEmpty(String s) {
      return ((s == null) || (s.trim().length() == 0));      
  }
  
  /**
   * Parse method parameters specified either in the params tag or in the parameters tag.
   * 
   * Return the list of parameters.
   */
  private String[] getMethodParams() throws MojoFailureException {
      if ( !isEmpty(methodParams) ) { // params tag is filled
           if (methodParameters.length > 0)           
               throw new MojoFailureException(
                   "You have to choose ONE (and only one) one way to specify parameters: "
                    + "either <params> or <parameters> tag.");
           else {
               return methodParams.split(" ");
           }
      } else {
          if (methodParameters.length > 0) {
              return methodParameters;
          } else // else no parameters nor params
              return new String[0];
      }
           
  }
  
  /**
   * Launch the Runtime Factory with specified parameters.
   */
  public void execute() throws MojoExecutionException, MojoFailureException {
	  if(loggingConfFile!=null) {
    try {
        LogManager.getLogManager().readConfiguration( new FileInputStream(loggingConfFile) );
    } catch (Exception e) {
        getLog().warn("Cannot load logging configuration file : "+loggingConfFile);
    }
	  }
    String[] methodParams = getMethodParams();
    
    try {
        Factory factory = null;
        Launcher launcher = null;
        
        if (useFscript)
            factory = new FrascatiScriptFactory();
        else
            factory = new Factory();
        launcher = new Launcher(composite, factory, getClasspath());
        
        if ( isEmpty(service) ) { // Run in a server mode
            getLog().info("FraSCAti is running in a server mode...");
            getLog().info("Press Ctrl+C to quit...");
            try {
                System.in.read();
            } catch (IOException e) {
                e.printStackTrace();
            }
        } else {
            getLog().info("Calling the '" + service + "' service: ");
            getLog().info("\tMethod '" + method + "'"
                    + ((methodParams.length == 0) ? "" : " with params: " + Arrays.toString(methodParams)));
            if (methodParams == null) methodParams = new String[0];
            Object result = launcher.call(service, method, Object.class, methodParams);
            getLog().info("Call done!");
            if (result != null) {
                getLog().info("Service response:");
                getLog().info(result.toString());
            }
        }
    } catch (FactoryInstantiationException e) {
        getLog().error("Cannot instantiate FraSCAti Factory!", e);
    }
  }
  
}
