/***
 * OW2 FraSCAti Factory Maven plugin
 * Copyright (C) 2008-2009 INRIA, USTL
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: frascati@ow2.org 
 *
 * Author(s): Damien Fournier
 *            Christophe Demarey
 */

package org.ow2.frascati.factory;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;

/**
 * A maven plugin for SCA Factories.
 * 
 * @execute phase="generate-sources"
 * @requiresDependencyResolution runtime
 * @goal compile
 */
public class GenerateFactoryMojo extends AbstractMojo {
  /**
   * The name of the SCA composite to execute.
   * 
   * @parameter expression="${compositeName}"
   * @required
   * @since 1.0
   */
  private String composite;
  /**
   * Space-separated list of source roots.
   * 
   * @parameter expression="${srcs}"
   * @since 1.0
   */
  private String[] srcs;
  /**
   * Space-separated list of lib paths.
   * 
   * @parameter expression="${libs}"
   * @since 1.0
   */
  private String[] libs;
  /**
   * true to obtain debug information about the code which is being generated.
   * 
   * @parameter default-value="false"
   * @since 1.0
   */
  private boolean debug;

  /**
   * The Maven project reference.
   * 
   * @parameter expression="${project}"
   * @required
   * @since 1.0
   */
  private MavenProject project;

  @SuppressWarnings( { "static-access" })
  public void execute() throws MojoExecutionException, MojoFailureException {

    // Add dependencies specified in the classpath of the project (pom.xml).
    File baseDir = project.getBasedir();
    
    // Add default maven directories to srcs and libs
    File fs = new File(baseDir.getAbsolutePath() + File.separator + "src/main/java");
    if (fs.exists()) srcs = addToArray(srcs, "src/main/java"); // Add only if exist (JuliacConfig throw IOException if not)
    libs = addToArray(libs, "src/main/resources");

    // List path of required libraries

    List<URL> list = new ArrayList<URL>();

    // Add dependencies artifacts
    Set<Artifact> artifacts = project.getDependencyArtifacts();
    for (Artifact artifact : artifacts) {
      try {
        list.add(artifact.getFile().toURI().toURL());
      } catch (MalformedURLException e) {
        getLog().warn("Could add dependency " + artifact.getId());
      }
    }

    // Add given library paths
    for (String lib : libs) {
      File f = null;
      try {
        // try to get file using project basedir
        f = new File(baseDir.getAbsolutePath() + File.separator + lib);
        if (!f.exists())
          throw new Exception();
      } catch (Exception e) {
        try {
          // try to get file using from root directory
          f = new File(lib);
          if (!f.exists())
            throw new Exception();
          System.out.println(lib);
        } catch (Exception e2) {
          getLog().error("Could not add library path : " + lib);
        }
      }

      try {
        list.add(f.toURI().toURL());
      } catch (MalformedURLException e) {
        getLog().error("Could not add library path : " + lib);
      }

    }

    URL[] urls = list.toArray(new URL[list.size()]);

    try {
        // Create a new instance of the generate factory
        // initialized with list of source path
        GenerateFactory generateFactory = new GenerateFactory(srcs);
        // Set the default directory for the generate factory
        generateFactory.setOutput(baseDir.getAbsolutePath());
    
        // Replace Generate Factory ClassLoader by Maven ClassLoader
        // ClassLoader parent = Thread.currentThread().getContextClassLoader();
        // URLClassLoader urlcl = new URLClassLoader(urls, parent);
        // generateFactory.setClassLoader(urlcl);
    
        // Set debug
        generateFactory.setDebug(debug);
    
        // generate the composite
        generateFactory.getComposite(composite, urls);
    } catch (FactoryInstantiationException e) {
        getLog().error("Cannot instantiate the FraSCAti factory!", e);
    }
  }
  
  // Array Add utility
  private String[] addToArray(String[] array, String s)
  {
     if (array == null | array.length == 0)return new String [] {s};
     String [] ans = new String[array.length+1];
     System.arraycopy(array, 0, ans, 0, array.length);
     ans[array.length] = s;
     return ans;
  }

}
