/**
 * Ishmael : An open source implementation of JSR-88
 * Copyright (C) 2006
 * Contact: ishmael-dev@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: BasicConnectedDeploymentManager.java 390 2009-01-28 15:57:07Z coqp $
 * --------------------------------------------------------------------------
 */
package org.ow2.ishmael.deploy.spi;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.Iterator;
import java.util.List;
import java.lang.reflect.Method;

import javax.enterprise.deploy.shared.ModuleType;
import javax.enterprise.deploy.spi.Target;
import javax.enterprise.deploy.spi.TargetModuleID;
import javax.enterprise.deploy.spi.exceptions.TargetException;
import javax.enterprise.deploy.spi.status.ProgressObject;
import javax.enterprise.deploy.spi.exceptions.InvalidModuleException;

import org.ow2.ishmael.deploy.spi.exceptions.IshmaelDeploymentManagerCreationException;
import org.ow2.ishmael.deploy.spi.impl.DeploymentSpecificEasyBeans;
import org.ow2.ishmael.deploy.spi.impl.DeploymentSpecificJOnAS;
import org.ow2.ishmael.deploy.spi.impl.IDeploymentSpecific;
import org.ow2.ishmael.deploy.spi.status.DistributeProgressObject;
import org.ow2.ishmael.deploy.spi.status.StartProgressObject;
import org.ow2.ishmael.deploy.spi.status.StopProgressObject;
import org.ow2.ishmael.deploy.spi.status.UnDeployProgressObject;
import org.ow2.ishmael.deploymentplan.JonasPlan;
import org.ow2.util.archive.api.IArchive;
import org.ow2.util.archive.impl.ArchiveManager;
import org.ow2.util.ee.deploy.api.deployable.CARDeployable;
import org.ow2.util.ee.deploy.api.deployable.EARDeployable;
import org.ow2.util.ee.deploy.api.deployable.EJBDeployable;
import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.deployable.RARDeployable;
import org.ow2.util.ee.deploy.api.deployable.WARDeployable;
import org.ow2.util.ee.deploy.api.deployer.DeployerException;
import org.ow2.util.ee.deploy.impl.helper.DeployableEntry;
import org.ow2.util.ee.deploy.impl.helper.DeployableHelper;
import org.ow2.util.ee.deploy.impl.helper.DeployableHelperException;
import org.ow2.util.ee.deploy.impl.helper.ModifyDeployableHelper;
import org.ow2.util.ee.deploy.impl.helper.PackDeployableHelper;
import org.ow2.util.ee.deploy.impl.helper.UnpackDeployableHelper;
import org.ow2.util.ee.metadata.war.impl.xml.parsing.WARDeploymentDescLoader;
import org.ow2.util.ee.metadata.common.impl.xml.parsing.ParsingException;
import org.ow2.util.file.FileUtils;
import org.ow2.util.url.URLUtils;
import org.w3c.dom.Document;


/**
 * BasicDeploymentManager.java This is the disconnected-mode JSR-88
 * DeploymentManager for JOnAS. Created: Fri Nov 22 15:36:32 2002
 *
 * @authors <a href="mailto:timh@mousetech.com">Tim Holloway </a><a
 * href="mailto:deanj@debian-sf.objectweb.org">Dean Jennings </a>
 */
public class BasicConnectedDeploymentManager extends BasicDeploymentManager {

    /**
     * Root temp folders.
     */
    private File tmpFolder = null;

    /**
     * folder for temporaries packed archives
     */
    private File tmpPackFolder = null;

    /**
     * folder for temporaries unpacked archives
     */
    private File tmpUnPackFolder = null;

    /**
     * Path of the application.xml deploymnet descriptor file
     */
    public static final String APPLICATION_FILE_NAME = "META-INF/application.xml";

    /**
     * Path of the jonas-application.xml deploymnet descriptor file
     */
    public static final String JONAS_APPLICATION_FILE_NAME = "META-INF/jonas-application.xml";

    /**
     * Path of the ra.xml deploymnet descriptor file
     */
    public static final String RA_FILE_NAME = "META-INF/ra.xml";

    /**
     * Path of the jonas-ra.xml deploymnet descriptor file
     */
    public static final String JONAS_RA_FILE_NAME = "META-INF/jonas-ra.xml";

    /**
     * The path to the application-client.xml file.
     */
    public static final String CLIENT_FILE_NAME = "META-INF/application-client.xml";

    /**
     * The path to the jonas-client.xml file.
     */
    public static final String JONAS_CLIENT_FILE_NAME = "META-INF/jonas-client.xml";

    /**
     * ejb-jar.xml filename
     */
    public static final String EJB_JAR_FILE_NAME = "META-INF/ejb-jar.xml";

    /**
     * jonas-ejb-jar.xml filename
     */
    public static final String JONAS_EJB_JAR_FILE_NAME = "META-INF/jonas-ejb-jar.xml";

    /**
     * The path to the web.xml file.
     */
    public static final String WEB_FILE_NAME = "WEB-INF/web.xml";

    /**
     * The path to the jonas-web.xml file.
     */
    public static final String JONAS_WEB_FILE_NAME = "WEB-INF/jonas-web.xml";

    /**
     * ejb-jar.xml filename
     */
    public static final String WS_EJBJAR_FILE_NAME = "META-INF/webservices.xml";

    /**
     * jonas-ejb-jar.xml filename
     */
    public static final String JONAS_WS_EJBJAR_FILE_NAME = "META-INF/jonas-webservices.xml";

    /**
     * ejb-jar.xml filename
     */
    public static final String WS_WEBAPP_FILE_NAME = "WEB-INF/webservices.xml";

    /**
     * jonas-ejb-jar.xml filename
     */
    public static final String JONAS_WS_WEBAPP_FILE_NAME = "WEB-INF/jonas-webservices.xml";

    // Management bean creation
    IDeploymentSpecific deployerImpl = null;

    /**
     * Boolean to indicate if the release mode was called. It implies that no
     * more methods can be called see javadoc of release method.
     */
    private boolean releasedMode = false;

    /**
     * Class of JOnAS WsGen generator
     */
    public static final String WSGENClASSNAME = "org.ow2.jonas.generators.wsgen.WsGen";



    public BasicConnectedDeploymentManager(String uri, String userName,
                                           String password) throws IshmaelDeploymentManagerCreationException {
        // Test URI
        if (uri.contains("syBeans")) {
            deployerImpl = new DeploymentSpecificEasyBeans(uri, userName,
                    password);
        } else {
            deployerImpl = new DeploymentSpecificJOnAS(uri, userName, password);
        }

    }

    /**
     * Start one or more targets.
     *
     * @throws java.lang.IllegalStateException
     *          thrown because we're in disconnected mode
     */
    @Override
    public ProgressObject start(TargetModuleID[] param1)
            throws IllegalStateException {
        checkNotReleased();
        StartProgressObject po = new StartProgressObject(deployerImpl, param1);
        new Thread(po).start();
        return po;
    }

    /**
     * Stop one or more targets.
     *
     * @throws java.lang.IllegalStateException
     *          thrown because we're in disconnected mode
     */
    @Override
    public ProgressObject stop(TargetModuleID[] param1)
            throws IllegalStateException {
        checkNotReleased();
        StopProgressObject po = new StopProgressObject(deployerImpl, param1);
        new Thread(po).start();
        return po;
    }

    /**
     * @throws java.lang.IllegalStateException
     *          thrown because we're in disconnected mode
     */
    @Override
    public Target[] getTargets() throws IllegalStateException {
        checkNotReleased();
        Target targets[] = new Target[1];
        targets[0] = new JOnASTarget();
        return targets;
    }

    /**
     *  Retrieve the list of J2EE application modules
     *  distributed to the identified targets
     *  and that are currently running on the associated server or servers
     * @param param1 module J2EE module type.
     * @param param2 target  A list of deployment Target designators
     * @return An array of TargetModuleID objects representing the running modules
     *         or 'null' if there are none
     * @throws javax.enterprise.deploy.spi.exceptions.TargetException
     *      An invalid Target designator encountered.
     * @throws java.lang.IllegalStateException
     *    is thrown when the method is called when running in disconnected mode
     */
    @Override
    public TargetModuleID[] getRunningModules(ModuleType param1, Target[] param2)
            throws TargetException, IllegalStateException {
        checkNotReleased();
        try {
            List l = deployerImpl.listModules(param1, 1);
            TargetModuleID[] ids = new TargetModuleID[l.size()];
            for (int i = 0; i < ids.length; i++) {
                String desc = (String) l.get(i);
                Target targetToDeploy = null;
                if (param2 != null)
                    targetToDeploy = param2[0];
                ids[i] = new TargetModuleIDImpl(desc, targetToDeploy);
            }
            return ids;
        } catch (Exception e) {
            e.printStackTrace();
            throw new TargetException(e.getMessage());
        }
    }

    /**
     * @param param1 <description>
     * @param param2 <description>
     * @return non running modules = availables modules
     * @throws javax.enterprise.deploy.spi.exceptions.TargetException
     *          <description>
     * @throws java.lang.IllegalStateException
     *          <description>
     */
    @Override
    public TargetModuleID[] getNonRunningModules(ModuleType param1,
                                                 Target[] param2) throws TargetException, IllegalStateException {
        checkNotReleased();
        try {
            List l = deployerImpl.listModules(param1, 0);
            TargetModuleID[] ids = new TargetModuleID[l.size()];
            for (int i = 0; i < ids.length; i++) {
                String desc = (String) l.get(i);
                Target targetToDeploy = null;
                if (param2 != null)
                    targetToDeploy = param2[0];
                ids[i] = new TargetModuleIDImpl(desc, targetToDeploy);
            }
            return ids;
        } catch (Exception e) {
            e.printStackTrace();
            throw new TargetException(e.getMessage());
        }
    }

    /**
     * Returns the list of all J2EE modules type param1 currently running on
     * target param2
     *
     * @param param1 ModuleType
     * @param param2 Target[]
     * @return TargetModuleID : available modules = non deployed| undeployed |stopped modules
     * @throws javax.enterprise.deploy.spi.exceptions.TargetException
     *          <description>
     * @throws java.lang.IllegalStateException
     *          <description>
     */
    @Override
    public TargetModuleID[] getAvailableModules(ModuleType param1,
                                                Target[] param2) throws TargetException, IllegalStateException {
        checkNotReleased();
        try {
            List l = deployerImpl.listModules(param1, 0);
            TargetModuleID[] ids = new TargetModuleID[l.size()];
            for (int i = 0; i < ids.length; i++) {
                String pathModule = (String) l.get(i);
                Target targetToDeploy = null;
                if (param2 != null)
                    targetToDeploy = param2[0];
                ids[i] = new TargetModuleIDImpl(pathModule, targetToDeploy);
            }
            return ids;
        } catch (Exception e) {
            e.printStackTrace();
            throw new TargetException(e.getMessage());
        }
    }

    /**
     * The distribute method performs three tasks; it validates the deployment
     * configuration data, generates all container specific classes and
     * interfaces, and moves the fully baked archive to the designated
     * deployment targets.
     *
     * @param param1 <description>
     * @param param2 <description>
     * @param param3 <description>
     * @return <description>
     * @throws java.lang.IllegalStateException
     *          <description>
     */
    @Override
    public ProgressObject distribute(Target[] targets, InputStream archive,
                                     InputStream plan) throws IllegalStateException {
        checkNotReleased();
        try {
            byte binarchive[] = new byte[archive.available()];
            archive.read(binarchive);
            byte binplan[] = null;
            if (plan != null) {
                binplan = new byte[plan.available()];
                plan.read(binplan);
            }
            String userName = System.getProperty("user.name");
            File archiveFile = File
                    .createTempFile(userName + "archive", ".tmp");
            FileOutputStream outarchive = new FileOutputStream(archiveFile);
            outarchive.write(binarchive);
            outarchive.close();

            // Rename the archive with the correct extention (suffix)
            ModuleType type = findTypeArchive(ArchiveManager.getInstance().getArchive(archiveFile));
            String pathArchiveGoodExtension = archiveFile
                    .getAbsolutePath()
                    .substring(0, archiveFile.getAbsolutePath().indexOf(".tmp"));
            pathArchiveGoodExtension = pathArchiveGoodExtension
                    + type.getModuleExtension();
            File tempFileReturn = new File(pathArchiveGoodExtension);

            copyStream(new FileInputStream(archiveFile), new FileOutputStream(
                    tempFileReturn));

            if (plan != null) {
                File planFile = File.createTempFile(userName + "plan", ".jar");
                FileOutputStream outplan = new FileOutputStream(planFile);
                outplan.write(binplan);
                outplan.close();
                tempFileReturn.deleteOnExit(); // delete the copy when it
                // possible
                return distribute(targets, tempFileReturn, planFile);
            }
            return distribute(targets, tempFileReturn, null);

        } catch (Exception ioe) {
            ioe.printStackTrace();
            throw new IllegalStateException(
                    "Problem deploying the application : " + ioe.toString());
        }
    }

    public ProgressObject distribute(Target[] arg0, ModuleType arg1,
                                     InputStream arg2, InputStream arg3) throws IllegalStateException {
        //throw new java.lang.UnsupportedOperationException("not implented");
        return distribute(arg0, arg2, arg3);
    }

    /**
     * The distribute method performs three tasks; it validates the deployment
     * configuration data, generates all container specific classes and
     * interfaces, and moves the fully baked archive to the designated
     * deployment targets.
     *
     * @param targets     to distribute
     * @param fileArchive initial archive
     * @param plan        jar file containing specific deployment descriptor to inject and a JOnAS Plan
     * @return ProgressObject
     * @throws java.lang.IllegalStateException
     *          <description>
     */
    @SuppressWarnings("unchecked")
    @Override
    public ProgressObject distribute(Target[] targets, File fileArchive,
                                     File plan) throws IllegalStateException {
        boolean isValidModule = true;
        File packedArchive = null;
        checkNotReleased();
        try {

            // Get Archive to distribute
            IArchive archive = ArchiveManager.getInstance().getArchive(fileArchive);
            String shorterArchiveName = URLUtils.shorterName(archive.getURL());

            IDeployable archivetoDeploy;
            try {
                archivetoDeploy = DeployableHelper.getDeployable(archive);
            } catch (DeployableHelperException e) {
                throw new Exception("Cannot get a deployable for the archive '"
                        + archive + "'", e);
            }
            // Detect type of archive
            ModuleType type = findTypeArchive(archive);

            if (plan == null) {
                FileInputStream in = new FileInputStream(fileArchive);
                byte fileToDeployed[] = new byte[in.available()];
                in.read(fileToDeployed);
                in.close();
                String[] genicArgsnull = {""};
                DistributeProgressObject po = new DistributeProgressObject(
                        deployerImpl, targets, fileToDeployed, new File(archive.getURL().getFile()).getName(), type, genicArgsnull, isValidModule);
                new Thread(po).start();
                return po;
            } else {
                JonasPlan jonasDeploymentPlan = new JonasPlan();
                jonasDeploymentPlan.createPlan(plan);
                /*
                System.out.println("+++++++++++++++++++");
                jonasDeploymentPlan.dumpMap();
                System.out.println("+++++++++++++++++++");
                */
                File tmpPackFolder = getTemporaryPackFolder();
                try {
                    if (type == ModuleType.EAR) {
                        packedArchive = updateEAR(archivetoDeploy, jonasDeploymentPlan);
                    } else if (type == ModuleType.RAR) {
                        shorterArchiveName = jonasDeploymentPlan.getArchiveName();
                        updateRAR(archivetoDeploy, jonasDeploymentPlan, shorterArchiveName);
                        packedArchive = PackDeployableHelper.pack(archivetoDeploy, tmpPackFolder, shorterArchiveName);
                    } else if (type == ModuleType.EJB) {
                        // ModuleType.EJB
                        // packedArchivetmp -> a new archive = initial archive updated
                        File packedArchivetmp = null;
                        updateJARWARCAR(archivetoDeploy, jonasDeploymentPlan, jonasDeploymentPlan.getArchiveName());
                        packedArchivetmp = PackDeployableHelper.pack(archivetoDeploy, tmpPackFolder, shorterArchiveName);
                        //run WSGen ifneeded
                        // Call the WSGen tool
                        String filename = packedArchivetmp.getAbsolutePath();
                        try {
                            ClassLoader clientClassloader = Thread.currentThread().getContextClassLoader();
                            Class<?> clazz = clientClassloader.loadClass(WSGENClASSNAME);
                            Method executeMethod = clazz.getDeclaredMethod("execute", new Class[]{String[].class});
                            Object wsgen = clazz.newInstance();
                            String[] args = new String[4];
                            args[0] = "-d";
                            args[1] = tmpPackFolder.getName();
                            args[2] = "-verbose";
                            args[3] = filename;
                            String nfile = (String) executeMethod.invoke(wsgen, new Object[]{args});
                            if (nfile.endsWith(".ear")) {
                                // archive has been changed.
                                packedArchive = new File(nfile);
                            } else {
                                packedArchive = packedArchivetmp;
                            }
                        } catch (Exception e) {
                            throw new DeployerException("Cannot execute WSGen on '" + filename + "'", e);
                        }
                    } else {
                        //ModuleType.WAR, ModuleType.CAR
                        // archivetoDeploy -> a new archive = initial archive updated
                        updateJARWARCAR(archivetoDeploy, jonasDeploymentPlan, jonasDeploymentPlan.getArchiveName());
                        packedArchive = PackDeployableHelper.pack(archivetoDeploy, tmpPackFolder, shorterArchiveName);
                    }
                } catch (InvalidModuleException ie) {
                    isValidModule = false;
                    packedArchive = fileArchive;
                }
                FileInputStream in = new FileInputStream(packedArchive);
                byte fileToDeployed[] = new byte[in.available()];
                in.read(fileToDeployed);
                in.close();
                DistributeProgressObject po;
                po = new DistributeProgressObject(deployerImpl, targets,
                        fileToDeployed, jonasDeploymentPlan.getArchiveName(),
                        type, null, isValidModule);
                new Thread(po).start();
                return po;
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new IllegalStateException("ISHMAEL :Exception", e);
        } finally {
            // delete temp files;
            //packedArchive.delete();
            //ishinstalldeleteTemporaries();
        }

    }

    /**
     * delete temporaries file used in distribute
     */
    private void deleteTemporaries() {
        FileUtils.delete(getTemporaryFolder());
    }


    private ModuleType findTypeArchive(IArchive archive) throws Exception {

        IDeployable deployable;
        try {
            deployable = DeployableHelper.getDeployable(archive);
        } catch (DeployableHelperException e) {
            throw new Exception("Cannot get a deployable for the archive '"
                    + archive + "'", e);
        }

        if (deployable instanceof CARDeployable) {
            return ModuleType.CAR;
        } else if (deployable instanceof EJBDeployable) {
            return ModuleType.EJB;
        } else if (deployable instanceof WARDeployable) {
            return ModuleType.WAR;
        } else if (deployable instanceof EARDeployable) {
            return ModuleType.EAR;
        } else if (deployable instanceof RARDeployable) {
            return ModuleType.RAR;
        } else {
            // Default is EAR;
            return ModuleType.EAR;
        }
    }

    private void copyStream(InputStream input, OutputStream output)
            throws IOException {
        BufferedInputStream reader = new BufferedInputStream(input);
        BufferedOutputStream writer = new BufferedOutputStream(output);
        byte[] cbuf = new byte[reader.available()];
        reader.read(cbuf);
        writer.write(cbuf);
        writer.flush();
        writer.close();
    }

    /**
     * Verify that the Deployment descriptor is valid
     *
     * @param deployable
     * @throws InvalidModuleException if standard descriptor is not valid
     */

    private void checkDescriptorValidity(IDeployable deployable) throws InvalidModuleException {
        IArchive archive = deployable.getArchive();
        try {
            ModuleType type = findTypeArchive(archive);
            if (type == ModuleType.WAR) {
                URL webXmlURL = archive.getResource("WEB-INF/web.xml");
                // Get document
                Document document = null;
                try {
                    WARDeploymentDescLoader.loadDeploymentDescriptor(webXmlURL, true);

                } catch (ParsingException e) {
                    throw new InvalidModuleException("Cannot parse the url");
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new InvalidModuleException("Invalid deploiement descriptor");
        }

    }

    /**
     * @param deployable         for archive  (jar/war/car)
     * @param plan               deployment plan
     * @param shorterArchiveName name of the embedded archive
     * @throws InvalidModuleException if standard descriptor is not valid
     * @throws DeployerException      if other deployer problems
     */

    @SuppressWarnings("unchecked")
    private void updateJARWARCAR(IDeployable deployable, JonasPlan plan, String shorterArchiveName)
            throws InvalidModuleException, DeployerException {

        //System.out.println(">>>> updateJARWARCAR deployable: "+deployable+" shorterArchiveName= "+plan.getArchiveName());
        // Get folder for packing/unpacking initial archive
        File tmpUnpackFolder = getTemporaryUnPackFolder();
        checkDescriptorValidity(deployable);    // may throw InvalidModuleException
        List<DeployableEntry> deployablesEntries = plan.getDeployableEntries(shorterArchiveName);
        ModifyDeployableHelper.addInDeployable(deployable, deployablesEntries, tmpUnpackFolder);

    }

    /**
     * @param deployable  for archive  (rar)
     * @param plan        deployment plan
     * @param archiveName archivename
     * @throws InvalidModuleException if standard descriptor is not valid
     * @throws DeployerException      if other deployer problems
     */
    @SuppressWarnings("unchecked")
    private void updateRAR(IDeployable deployable, JonasPlan plan, String archiveName)
            throws InvalidModuleException, DeployerException {
        //      Get folder for packing/unpacking initial archive
        File tmpUnpackFolder = getTemporaryUnPackFolder();
        checkDescriptorValidity(deployable);    // may throw InvalidModuleException
        List<DeployableEntry> deployablesEntries = plan.getDeployableEntries(archiveName);
        ModifyDeployableHelper.addInDeployable(deployable, deployablesEntries, tmpUnpackFolder);
    }

    /**
     * @return the folder root for temporary folders
     */
    private File getTemporaryFolder() {

        if (tmpFolder == null) {
            tmpFolder = new File(System.getProperty("java.io.tmpdir") + File.separator + System.getProperty("user.name", "default")
                    + "ishmael-tmp");
        }
        return tmpFolder;
    }

    /**
     * @return the folder for packed filesfor temporary folders
     */
    private File getTemporaryPackFolder() {
        if (tmpPackFolder == null) {
            //  Create folder for packing resulting archive
            tmpPackFolder = new File(getTemporaryFolder(), "pack");
            tmpPackFolder.mkdirs();
        }
        return tmpPackFolder;
    }

    /**
     * @return the folder for packed filesfor temporary folders
     */
    private File getTemporaryUnPackFolder() {
        if (tmpUnPackFolder == null) {
            // Create folder for unpacking initial archive
            tmpUnPackFolder = new File(getTemporaryFolder(), "unpack");
            tmpUnPackFolder.mkdirs();
        }
        return tmpUnPackFolder;
    }

    /**
     * updateEAR: update an ear archive by injecting specific deployment descriptors in each sub archives
     *
     * @param archive initial archive
     * @param plan    deployment plan
     * @return the deployable corresponding to the updated archive
     * @throws Exception
     */
    @SuppressWarnings("unchecked")
    private File updateEAR(IDeployable deployable, JonasPlan plan) throws Exception {

        // Get folders for packing/unpacking initial archive
        File tmpUnpackFolder = getTemporaryUnPackFolder();
        File tmpPackFolder = getTemporaryPackFolder();
        URL urlArchive = deployable.getArchive().getURL();
        String initialArchiveName = URLUtils.shorterName(urlArchive);
// unpack the initial ear
        EARDeployable unpackDeployable = UnpackDeployableHelper.unpack((EARDeployable) deployable, tmpUnpackFolder, initialArchiveName);
// update the global part of the ear
        List<DeployableEntry> deployablesEntries = plan.getDeployableEntries(initialArchiveName);
        ModifyDeployableHelper.addInDeployable(deployable, deployablesEntries, tmpUnpackFolder);
// update the unpacked ejbjars
        List<EJBDeployable<?>> lejbs = unpackDeployable.getEJBDeployables();
        Iterator<EJBDeployable<?>> itere = lejbs.iterator();
        while (itere.hasNext()) {
            EJBDeployable<?> ejbjar = itere.next();
            String archiveName = URLUtils.shorterName(ejbjar.getArchive().getURL());
            updateJARWARCAR(ejbjar, plan, archiveName);
        }
        // update the unpacked cars
        List<CARDeployable> lcars = unpackDeployable.getCARDeployables();
        Iterator<CARDeployable> iterc = lcars.iterator();
        while (iterc.hasNext()) {
            CARDeployable car = iterc.next();
            String archiveName = URLUtils.shorterName(car.getArchive().getURL());
            updateJARWARCAR(car, plan, archiveName);
        }

        // update the unpacked rars
        List<RARDeployable> lrars = unpackDeployable.getRARDeployables();
        Iterator<RARDeployable> iterr = lrars.iterator();
        while (iterr.hasNext()) {
            RARDeployable rar = iterr.next();
            String archiveName = URLUtils.shorterName(rar.getArchive().getURL());
            updateRAR(rar, plan, archiveName);
        }
        // update the unpacked wars
        List<WARDeployable> lwars = unpackDeployable.getWARDeployables();
        Iterator<WARDeployable> iterw = lwars.iterator();
        while (iterw.hasNext()) {
            WARDeployable war = iterw.next();
            String archiveName = URLUtils.shorterName(war.getArchive().getURL());
            updateJARWARCAR(war, plan, archiveName);
        }
        File packedArchive = PackDeployableHelper.pack(unpackDeployable, tmpPackFolder, "new-" + initialArchiveName);

        return packedArchive;
    }

    /**
     * Remove the application from the target server. Only the TargetModuleIDs
     * which represent a root module are valid for undeployment. A root
     * TargetModuleID has no parent. A TargetModuleID with a parent can not be
     * undeployed. A root TargetModuleID module and all its child modules will
     * be undeployed. The root TargetModuleID module and all its child modules
     * must stopped before they can be undeployed.
     *
     * @param param1 target to be undeploy
     * @return <description>
     * @throws java.lang.IllegalStateException
     *          <description>
     */
    @Override
    public ProgressObject undeploy(TargetModuleID[] param1)
            throws IllegalStateException {
        checkNotReleased();
        UnDeployProgressObject po = new UnDeployProgressObject(deployerImpl,
                param1);
        new Thread(po).start();
        return po;
    }

    /*
    * (non-Javadoc)
    *
    * @see javax.enterprise.deploy.spi.DeploymentManager#release()
    */
/**
 *
 */
    @Override
    public void release() {
        releasedMode = true;
// TODO : The connections to J2EE resources must be closed
    }

    /**
     * Check that the current mode is not the released mode. Else, throw an
     * IllegalStateException exception.
     */
    private void checkNotReleased() {
        if (releasedMode) {
            throw new IllegalStateException(
                    "The release() method was called, then no more methods can be called");
        }
    }


    protected class JOnASTarget implements Target {

        private String name = "Main";

        private String description = "The main jonas server";

        public String getName() {
            return name;
        }

        public String getDescription() {
            return description;
        }
    }


}
