/**
 * Ishmael : An open source implementation of JSR-88
 * Copyright (C) 2006-2007
 * Contact: ishmael-dev@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: AbsDeploymentSpecific.java 343 2007-10-02 13:18:58 +0000 (Tue, 02 Oct 2007) coqp $
 * --------------------------------------------------------------------------
 */
package org.ow2.ishmael.deploy.spi.impl;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import javax.enterprise.deploy.shared.ModuleType;
import javax.enterprise.deploy.spi.exceptions.DeploymentManagerCreationException;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;
import javax.naming.Context;

import org.ow2.ishmael.deploy.spi.exceptions.DeploymentException;
import org.ow2.ishmael.deploy.spi.exceptions.IshmaelDeploymentManagerCreationException;

/**
 * Abstract class for specific deployment.
 * @author Florent Benoit
 */
public abstract class AbsDeploymentSpecific implements IDeploymentSpecific {

    /**
     * JMXConnector used for all operations.
     */
    private JMXConnector jmxConnector = null;

    /**
     * Connection to MBean.
     */
    private MBeanServerConnection mBeanServerConnection = null;

    /**
     * ObjectName used for operations
     */
    private ObjectName objectName = null;

    /**
     * Prefix URL.
     */
    private static final String PREFIX_URL = "deployer:ishmael:";

    /**
     * URI used by deployer
     */
    private String uri = null;

    /**
     * State of the modules distributed on the Server but undeployed
     */
    private static final int AVAILABLE = 0;

    /**
     * State of the modules deployed on the Server
     */
    private static final int RUNNING = 1;


    /**
     * List of jar  Modules
     */
    private Map <Integer, List<String>> jarModules = null;

    /**
     * List of war  Modules
     */
    private Map <Integer, List<String>> warModules = null;

    /**
     * List of rar  Modules
     */
    private Map <Integer, List<String>>rarModules = null;

    /**
     * List of ear  Modules
     */
    private Map <Integer, List<String>> earModules = null;

    /**
     * Constructor.
     * Initialize connector to the remote system.
     * @param uri URI given by deployment tool
     */
    public AbsDeploymentSpecific(String uri) throws IshmaelDeploymentManagerCreationException {
        this.uri = uri;
        try {
            objectName = new ObjectName(getStringObjectName());
        } catch (MalformedObjectNameException e) {
            throw new IshmaelDeploymentManagerCreationException("Cannot create ObjectName", e);
        } catch (NullPointerException e) {
            throw new IshmaelDeploymentManagerCreationException("Cannot create ObjectName", e);
        }
        init();
    }




    protected ObjectName getObjectName() {
        return objectName;
    }



    protected MBeanServerConnection getMBeanServerConnection() {
        return mBeanServerConnection;
    }


    protected abstract String getStringObjectName();

    /**
     * Create JMX Connector with given uri
     */
    private void init() throws IshmaelDeploymentManagerCreationException {
        // Get JMX URL from this uri
        try {
            Map<String,String> env = new HashMap<String,String>();
            env.put(Context.INITIAL_CONTEXT_FACTORY, "org.objectweb.carol.jndi.spi.MultiOrbInitialContextFactory");
            jmxConnector = JMXConnectorFactory.connect(getJmxServiceUrl(), env);
        } catch (IOException e) {
            e.printStackTrace();
            throw new IshmaelDeploymentManagerCreationException("Cannot connect to the JMX connector", e);
        } catch (DeploymentManagerCreationException e) {
            e.printStackTrace();
            throw new IshmaelDeploymentManagerCreationException("Cannot connect to the JMX connector", e);
        }

        // Connect to the connector
        try {
            mBeanServerConnection = jmxConnector.getMBeanServerConnection();
        } catch (IOException e) {
            throw new IshmaelDeploymentManagerCreationException("Cannot get MBean Server connection", e);
        }

        jarModules = new HashMap<Integer, List<String>>();
        jarModules.put(AVAILABLE,new ArrayList<String>());
        jarModules.put(RUNNING,new ArrayList<String>());

        warModules = new HashMap<Integer, List<String>>();
        warModules.put(AVAILABLE,new ArrayList<String>());
        warModules.put(RUNNING,new ArrayList<String>());

        earModules = new HashMap<Integer, List<String>>();
        earModules.put(AVAILABLE,new ArrayList<String>());
        earModules.put(RUNNING,new ArrayList<String>());

        rarModules = new HashMap<Integer, List<String>>();
        rarModules.put(AVAILABLE,new ArrayList<String>());
        rarModules.put(RUNNING,new ArrayList<String>());
    }

    /**
     * Return JMX service URL from the uri.
     */
    private JMXServiceURL getJmxServiceUrl() throws DeploymentManagerCreationException {

        // Check URL
        if (!uri.startsWith(PREFIX_URL)) {
            throw new DeploymentManagerCreationException("Invalid URL '" + uri + "'. It doesn't start with " + PREFIX_URL);
        }

        // Now extract JMX connector URL
        String jmxConnectorURL = uri.substring(PREFIX_URL.length());
        System.out.println("Using JMX connector " + jmxConnectorURL);

        try {
            return new JMXServiceURL(jmxConnectorURL);
        } catch (MalformedURLException e) {
            DeploymentManagerCreationException newE = new DeploymentManagerCreationException("Cannot make url on '" + uri + "'.");
            newE.initCause(e);
            throw newE;
        }
    }


    /**
     * List modules type in state etat
     * @param state STATUS_RUNNING = 1, STATUS_STOPPED = 2, STATUS_ALL = 0
     * @param type TYPE_JAR = 1, TYPE_WAR = 2, TYPE_EAR = 3, TYPE_RAR = 4 public
     *        final int TYPE_CAR = 5;
     * Attention on ne sait toujours pas differentier l'etat Running/Stopping.
     * Lorsqu'une application est deployee , elle est automatiquement starter.
     * @return
     * @throws DeploymentException
     */
    @SuppressWarnings("unchecked")
    public List<String> listModules(ModuleType type, int state) throws DeploymentException {
        List<String> listReturned = new ArrayList<String>();
        try {
            if (type.equals(ModuleType.EAR))
                listReturned =  earModules.get(state);
            else if (type.equals(ModuleType.EJB))
                listReturned =  jarModules.get(state);
            else if (type.equals(ModuleType.RAR))
                listReturned =  rarModules.get(state);
            else if (type.equals(ModuleType.WAR))
                listReturned =  warModules.get(state);
        }catch (Exception e) {
            throw (new DeploymentException(e.getMessage()));
        }

        return listReturned;
    }

    /**
     * deploy application p
     * @param p
     * @throws DeploymentException
     */
    public void deployApplication(Properties p) throws DeploymentException {
        String file = p.getProperty("file");

        // No file
        if (file == null) {
            throw new DeploymentException("No file to deploy");
        }
        this.deploy(p);
        addRunning(file);
    }

    /**
     * Undeploy application p
     * @param p
     * @throws DeploymentException
     */
    public void undeployApplication(Properties p) throws DeploymentException {
        String fileName = p.getProperty("file");
        if (fileName == null) throw new DeploymentException("No file to undeploy");

        if (isDeployedApplication(p))
            throw new DeploymentException("Application must be stopped before they can be undeployed.");
        File app = new File(fileName);
        boolean isdelete = app.delete();
        System.out.println(">>>>>>>>>> in undeployApplication "+fileName+ " has been deleted");
        removeModule(fileName);
        if (!isdelete) throw new DeploymentException("Deleted application : " + fileName + " is not a success.");
    }

    /**
     * Stop application p
     * @param p
     * @throws DeploymentException
     */
    public void stopApplication(Properties p) throws DeploymentException {

        String file = p.getProperty("file");
        if (file == null) throw new DeploymentException("No file to stop");
        stop(file);
        removeRunning(file);
    }

    /**
     * @param p
     * @return true if p is deploy false otherwise
     * @throws DeploymentException
     */
    public boolean isDeployedApplication(Properties p) throws DeploymentException {
        String file = p.getProperty("file");
        if (file == null) throw new DeploymentException("Unknow file into isDeployedFile");
        HashMap<Integer, List<String>> moduleList = getModuleList(file);
        List <String> lmodules = moduleList.get(RUNNING);
        return(lmodules.contains(file));

    }

    /*
     * Add the module in the list of available modules
     * @param module  path to the module available
     */
    public void addAvailable(String module) {

        HashMap<Integer, List<String>> moduleList = getModuleList(module);
        List <String> lmodules = moduleList.get(AVAILABLE);
        if (!lmodules.contains(module)) {
            lmodules.add(module);
        }
    }

    /*
     * Add the module in the list of running modules
     * @param module  path to the module running
     */
    public void addRunning(String module) {
        HashMap<Integer, List<String>> moduleList = getModuleList(module);
        List <String> lavailables = moduleList.get(AVAILABLE);
        List <String> lrunning = moduleList.get(RUNNING);
        if (lavailables.contains(module)) {
            lavailables.remove(module);
        }
        if (!lrunning.contains(module)) {
            lrunning.add(module);
        }
    }

    /*
     * Remove the module from the list of running modules and add it in the available modules
     * @param module  path to the module running
     */
    public void removeRunning(String module) {
        HashMap<Integer, List<String>> moduleList = getModuleList(module);
        List <String> lavailables = moduleList.get(AVAILABLE);
        List <String> lrunning = moduleList.get(RUNNING);
        if (lrunning.contains(module)) {
            lrunning.remove(module);
        }
        if (!lavailables.contains(module)) {
            lavailables.add(module);
        }
    }

    /*
     * Remove the module from any list (the module has been deleted)
     * @param module  path to the module
     */
    public void removeModule(String module) {
        HashMap<Integer, List<String>> moduleList = getModuleList(module);
        List <String> lavailables = moduleList.get(AVAILABLE);
        List <String> lrunning = moduleList.get(RUNNING);
        if (lrunning.contains(module)) {
            lrunning.remove(module);
        }
        if (lavailables.contains(module)) {
            lavailables.remove(module);
        }
    }


    /*
     * Get the modules list corresponding to the Type of the given module
     * @param module  path to the module
     */
    @SuppressWarnings("unchecked")
    public HashMap<Integer, List<String>> getModuleList (String module) {
        ModuleType type = null;
        HashMap<Integer, List<String>> moduleList = null;
        try {
            type = findType(module);
            if (type.equals(ModuleType.EAR)) {
                moduleList = (HashMap<Integer, List<String>>)earModules;
            } else if (type.equals(ModuleType.EJB)) {
                moduleList = (HashMap<Integer, List<String>>)jarModules;
            } else if (type.equals(ModuleType.WAR)) {
                moduleList = (HashMap<Integer, List<String>>)warModules;
            } else if (type.equals(ModuleType.RAR)) {
                moduleList = (HashMap<Integer, List<String>>)rarModules;
            }
        } catch (DeploymentException e) {
        }
        return moduleList;
    }

    /**
     * deploy the file on the server : call genic to generate class for JOnAS
     * and copy the result file into JOnAS server directory for deployment
     * @param type
     * @param file
     * @param name
     * @throws DeploymentException
     */
    public String deployFile(ModuleType type, byte[] file, String name, String[] genicArgs) throws DeploymentException {
        Object param[] = {name, file};
        String signature[] = {"java.lang.String", "[B" };

        // invoke method on the MBean
        String nameapp = null;
        try {
            nameapp = (String) getMBeanServerConnection().invoke(getObjectName(), "dumpFile", param, signature);
            addAvailable(nameapp);
            return nameapp;


        } catch (Exception e) {
            e.printStackTrace(System.out);
            throw (new DeploymentException(e.getMessage()));
        }
    }


    /**
     * distribute (send)  the file on the server
     * and copy the result file into JOnAS server directory for deployment
     * @param file Byte Array representing the file to upload
     * @param name name of the file
     * @return path to the distributed file
     * @throws DeploymentException
     */
    public String sendFile(byte[] file, String name) throws DeploymentException {

        Object param[] = {name, file};
        String signature[] = {"java.lang.String", "[B" };

        // invoke method on the MBean
        String nameapp = null;
        try {

            System.out.println(">>>>>>>>>>> call to J2eeServer.distribute "+name);
            nameapp = (String) getMBeanServerConnection().invoke(getObjectName(), "distribute", param, signature);
            addAvailable(nameapp);
            return nameapp;


        } catch (Exception e) {
            e.printStackTrace(System.out);
            throw (new DeploymentException(e.getMessage()));
        }
    }

    /**
     * @param info property name file indicate the path and name ear file to
     *        deploy
     * @throws DeploymentException
     */
    protected void deploy(Properties info) throws DeploymentException {
        String earFile = info.getProperty("file");
        String param[] = {earFile};
        String signature[] = {"java.lang.String"};
        // invoke method on the MBean

        try {
            System.out.println(">>>>>>>>>>> call to J2eeServer.deploy "+earFile);
            getMBeanServerConnection().invoke(getObjectName(), "deploy", param, signature);
        } catch (Exception e) {

            throw (new DeploymentException("Error during EAR deployment" + e.getMessage()));
        }
    }

    /**
     * @param p String  name file to undeploy
     *
     * @throws DeploymentException
     */
    protected void stop(String file) throws DeploymentException {
        String param[] = {file};
        String signature[] = {"java.lang.String"};
        // invoke method on the MBean
        try {
            System.out.println(">>>>>>>>>>> (stop) call to J2eeServer.undeploy "+file);
            getMBeanServerConnection().invoke(getObjectName(), "undeploy", param, signature);
        } catch (Exception e) {
            throw (new DeploymentException(e.getMessage()));
        }
    }




    /**
     * @param p property name file indicate the path and name ear file
     * @return true if p is an ear application deployed false otherwise
     * @throws DeploymentException
     */
    protected boolean isEarDeployed(Properties p) throws DeploymentException {
        String ear_file = p.getProperty("file");
        String param[] = {ear_file};
        String signature[] = {"java.lang.String"};
        // invoke method on the MBean
        Object deployed = null;
        try {
            deployed = getMBeanServerConnection().invoke(getObjectName(), "isEarDeployed", param, signature);
        } catch (Exception e) {
            throw (new DeploymentException(e.getMessage()));
        }
        return Boolean.parseBoolean(deployed.toString());
    }


    protected boolean isJarDeployed(Properties p) throws DeploymentException {
        String ejb_file = p.getProperty("file");
        String param[] = {ejb_file};
        String signature[] = {"java.lang.String"};
        // invoke method on the MBean
        Object deployed = null;
        try {
            deployed = getMBeanServerConnection().invoke(getObjectName(), "isJarDeployed", param, signature);
        } catch (Exception e) {
            throw (new DeploymentException(e.getMessage()));
        }
        return Boolean.parseBoolean(deployed.toString());
    }



    protected ModuleType findType(String path) throws DeploymentException {
        if (path.endsWith(ModuleType.EJB.getModuleExtension())) {
            return ModuleType.EJB;
        } else if (path.endsWith(ModuleType.EAR.getModuleExtension())) {
            return ModuleType.EAR;
        } else if (path.endsWith(ModuleType.WAR.getModuleExtension())) {
            return ModuleType.WAR;
        } else if (path.endsWith(ModuleType.RAR.getModuleExtension())) {
            return ModuleType.RAR;
        }
        throw new DeploymentException("No module type found for path '" + path + "'.");
    }

}