/*
 * Ishmael : An open source implementation of JSR-88
 * Contact: ishmael-dev@lists.debian-sf.objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 *
 * Initial developer: Camilleri J?r?me
 */
package org.ow2.ishmael.deploymentplan;

import java.io.File;
import java.net.URL;
import java.net.URLDecoder;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;
import java.text.MessageFormat;

import org.ow2.util.archive.api.ArchiveException;
import org.ow2.util.archive.api.IArchive;
import org.ow2.util.archive.impl.ArchiveManager;
import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.deployer.DeployerException;
import org.ow2.util.ee.deploy.impl.helper.DeployableEntry;
import org.ow2.util.ee.deploy.impl.helper.DeployableHelper;
import org.ow2.util.ee.deploy.impl.helper.DeployableHelperException;
import org.ow2.util.ee.deploy.impl.helper.UnpackDeployableHelper;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.url.URLUtils;
import org.ow2.util.xml.DocumentParser;
import org.ow2.util.xml.DocumentParserException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;


/**
 * @author Florent Benoit
 */

public class JonasPlan  {


    /**
     * Name of the jonas deplyment plan descriptor
     */
    private static final String JDEPLOYMENTPLAN = "jonas-deployment-plan.xml";

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(JonasPlan.class);
    /**
     * Archive Name element.
     * null means JonasPlan class is not initialized
     */
    private String archiveName = null;

    /**
     * File of the plan
     */
    private  File file = null;

    /**
     * map that links DD and archives name
     *      the keys are sub archive names
     *      the values are ArrayList<Entry>
     */
    private Hashtable planMap = null;

    /**
     * @author coqp
     *
     * util class that knows the naming convention for entries in JOnASdeplyment plan
     * the entry name has the following syntax :
     * <dir>_<archivename>.<descriptorname>
     * with dir = meta-inf or web-inf
     * archivename = xx.jar/xx.war/xx.ear/xx.rar
     * descriptorname = xx.xml
     *
     * An Entry has
     *  - an entryName: name of the entry in the archive that will be updated
     *  - an urlEntry: URL towards the deaplyment descriptor file to inject
     *  - an archiveName: name of the sub archive in which the entry must be added
     */
    public class Entry {

        public String archiveName = null;
        public String entryName = null;
        public URL urlEntry = null;
        public Entry(String ddentryName) {
            StringTokenizer st = new StringTokenizer(ddentryName);
            String dir = st.nextToken().toUpperCase();
            String following = st.nextToken();
            StringTokenizer stf = new StringTokenizer(following,".");
            String baseArchiveName = stf.nextToken();
            String extArchiveName = stf.nextToken();
            String baseDDName = stf.nextToken();
            String extDDName = stf.nextToken();
            this.archiveName = baseArchiveName + "."+ extArchiveName;
            this.entryName = dir+"/"+ baseDDName + "."+ extDDName;
        }

        /**
         * @return the archiveName
         */
        public String getArchiveName() {
            return archiveName;
        }

        /**
         * @param archiveName the archiveName to set
         */
        public void setArchiveName(String archiveName) {
            this.archiveName = archiveName;
        }

        /**
         * @return the entryName
         */
        public String getEntryName() {
            return entryName;
        }

        /**
         * @param entryName the entryName to set
         */
        public void setEntryName(String entryName) {
            this.entryName = entryName;
        }


        /**
         * @return the url Entry
         */
        public URL getUrlEntry() {
            return urlEntry;
        }


        /**
         * @param fileEntry the fileEntry to set
         */
        public void setUrlEntry(URL urlEntry) {
            this.urlEntry = urlEntry;
        }

    }
    /**
     * Constructor.
     */
    public JonasPlan() {
        planMap = new Hashtable();
    }



    /**
     * createPlan : extract information from the JonAS deployment plan
     *
     * @param file archive to read
     *      a JOnAS deployment plan is a jar file that contains:
     *      - a jonas-deployment-plan.xml with the name of the archive (temporary?)
     *      - one entry for each deployment descriptor to inject in the archive to update
     *      - the entry name has the following syntax :
     *          <dir>_<archivename>.<descriptorname>
     *          with dir = meta-inf or web-inf
     *          archivename = xx.jar/xx.war/xx.ear/xx.rar  (sub archive name)
     *          descriptorname = xx.xml
     */
    public void createPlan(File file) throws Exception {
        //      Get archive
        IArchive deploymentPlanArchive = ArchiveManager.getInstance().getArchive(file);
        IDeployable deployable;
        try {
            deployable = DeployableHelper.getDeployable(deploymentPlanArchive);
        } catch (DeployableHelperException e) {
            throw new Exception("Cannot get a deployable for the archive '"
                    + deploymentPlanArchive + "'", e);
        }
        // unpack the JonAS deployment plan under temporary directory
        IDeployable unpackedPlan = UnpackDeployableHelper.unpack(deployable, "jonas-plan");

        URL urlarchive = URLUtils.fileToURL2(file);
        String shorterArchiveName = URLUtils.shorterName(urlarchive);
        Iterator<URL> itResouces = null;
        try {
            itResouces = unpackedPlan.getArchive().getResources();
        } catch (ArchiveException e) {
            throw new DeployerException("Cannot get the resources on the archive '" + deploymentPlanArchive + "'.", e);
        }
        // read each entry in the JonAS deployment plan and update the Map if needed
        while (itResouces.hasNext()) {
            URL url = itResouces.next();
            String path = URLDecoder.decode(url.getPath(), "UTF-8");

            String entryName = path.substring(path.lastIndexOf(shorterArchiveName)+shorterArchiveName.length()+1);
            updateMap(entryName, url);
            if (entryName.equals("jonas-deployment-plan.xml")) {
                Document document = null;
                try {
                    document = DocumentParser.getDocument(url, false, null);
                } catch (DocumentParserException e) {
                    throw new DeployerException("Cannot parse the url '"+ url + "'", e);
                }
                //  Root element = <jonas-deployment-plan>
                Element planRootElement = document.getDocumentElement();
                //  Set archive name
                NodeList moduleList = planRootElement.getElementsByTagName("archive-name");
                if (moduleList.getLength() > 0) {
                    Node node = moduleList.item(0).getFirstChild();
                    if (node != null) {
                        this.setArchiveName(node.getNodeValue());
                    }
                }
            }
        }

        if (deploymentPlanArchive != null) {
            deploymentPlanArchive.close();
        }
        this.setFile(file);
        //dumpMap();
    }

    /**
     * updateMap : update the map if the entry correspond to a deployment descriptor
     *
     * @param entryName name of the entry in the archive for the deployment plan
     * @param URL of the corresponding  URL File for the entry
     */
    private void updateMap(String entryName, URL urlEntry) {
        ArrayList<Entry> alf = null;
        if (validentry(entryName)) {
            Entry entry = new Entry(entryName);
            entry.setUrlEntry(urlEntry);
            String key = entry.getArchiveName();
            if (!planMap.containsKey(key)) {
                alf = new ArrayList<Entry>();
                planMap.put(key, alf);
            } else {
                alf = (ArrayList<Entry>)planMap.get(key);
            }
            alf.add(entry);
        }

    }

    /**
     * dumpMap : for debuuging purpose
     */
    public void  dumpMap() {
        System.out.println(">>>>> archivename: "+this.getArchiveName());
        Collection keys = planMap.keySet();
        Iterator it = keys.iterator();
        while ( it.hasNext()) {
            String key = (String) it.next();
            ArrayList<Entry> alf = null;
            alf = (ArrayList<Entry>) planMap.get(key);
            System.out.println(" Key : "+key);
            for (int i = 0; i< alf.size(); i++) {
                Entry en = alf.get(i);
                System.out.println(">>>>> entry: "+en.getEntryName());
                System.out.println(">>>>> url: "+en.getUrlEntry().toString());
            }

        }
    }



    /**
     * validentry
     * @param entryName entry beginning with META-INF or jonas-deployment-plan must be ignored
     * @return true if this entry correspond to a deployment descriptor to inject in an archive
     */
    private boolean validentry(String entryName) {
        if (entryName.toLowerCase().startsWith("meta-inf/"))
            return false;
        if (entryName.equalsIgnoreCase(JDEPLOYMENTPLAN) ){
            return false;
        }
        return true;
    }



    /**
     * Set the host element of this object
     * @param host host element of this object
     */
    public void setArchiveName(String archiveName) {
        this.archiveName = archiveName;
    }

    /**
     * @return the archiveName element
     */
    public String getArchiveName() {
        return archiveName;
    }



    /**
     * @return the file
     */
    public File getFile() {
        return file;
    }



    /**
     * @param file the file to set
     */
    public void setFile(File file) {
        this.file = file;
    }

    /**
     * @param name of the resource (DD)
     * @return the corresponding file
     */
    File getResource(String name) {
        File res = (File)planMap.get(name);
        return res;
    }

    /**
     * @param key key in the Map
     * @return a list of DeployableEntry
     */
    public List<DeployableEntry>  getDeployableEntries (String key) {
        List<DeployableEntry> deployablesEntries = new ArrayList<DeployableEntry>();
        if (planMap.containsKey(key)) {
            ArrayList<Entry> alf = (ArrayList<Entry>)planMap.get(key);
            for (int i = 0; i< alf.size(); i++) {
                Entry en = alf.get(i);
                DeployableEntry de = new DeployableEntry();
                de.setFile(URLUtils.urlToFile(en.getUrlEntry()));
                de.setName(en.getEntryName());
                deployablesEntries.add(de);
            }
            return deployablesEntries;

        } else {
        return null;
        }
    }
}
