/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DiscoveryManagerSLSB.java 6918 2010-09-22 12:37:25Z veyjul $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.agent.server.discovery;

import java.util.List;
import java.util.UUID;

import javax.ejb.Remote;
import javax.ejb.Stateless;
import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.Message;
import javax.jms.MessageProducer;
import javax.jms.Session;
import javax.jms.Topic;
import javax.naming.Context;
import javax.naming.InitialContext;

import org.apache.felix.ipojo.annotations.Bind;
import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Unbind;
import org.osgi.framework.ServiceReference;
import org.ow2.jasmine.agent.common.discovery.AgentDiscoveryService;
import org.ow2.jasmine.agent.common.discovery.SystemProperties;
import org.ow2.jasmine.agent.server.Agent;
import org.ow2.jasmine.agent.server.AgentState;
import org.ow2.jasmine.agent.server.event.AgentEvent;
import org.ow2.jasmine.agent.server.event.AgentEventType;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Stateless Session Bean to control the agents and the discovery
 * 
 * @author Julien Vey
 */
@Stateless(mappedName = "agent/discoverymanager")
@Remote(DiscoveryManager.class)
@Component(name = "JASMINe Discovery Manager")
public class DiscoveryManagerSLSB implements DiscoveryManager {

    /**
     * Logger
     */
    private static Log logger = LogFactory.getLog(DiscoveryManagerSLSB.class);

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.agent.server.discovery.DiscoveryManager#getAgentList()
     */
    public List<Agent> getAgentList() {
        return DiscoveryManagerImpl.getInstance().getAgentList();
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.agent.server.discovery.DiscoveryManager#setAgentList(java.util.List)
     */
    public void setAgentList(List<Agent> list) {
        DiscoveryManagerImpl.getInstance().setAgentList(list);
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.agent.server.discovery.DiscoveryManager#launchDiscoveryOnAgent(java.lang.String)
     */
    public void launchDiscoveryOnAgent(String idAgent) {
        DiscoveryManagerImpl.getInstance().launchDiscoveryOnAgent(idAgent);

    }

    /**
     * Bind any agent discovery service
     * 
     * @param service
     *            the service binded
     * @param reference
     *            the reference of the service
     */
    @Bind(optional = true, aggregate = true)
    public void bindAgentDiscoveryService(AgentDiscoveryService service, ServiceReference reference) {
        Agent a = DiscoveryManagerImpl.getInstance().getAgentFromUUID(UUID.fromString(service.getId()));
        if (a != null) {
            a.setSystemProperties(service.getSystemProperties());
            a.setState(AgentState.RUNNING);
            a.setIsDiscoveryRunning(false);
            logger.info("Agent {0} re-registered with the following OS properties", a.getAgentId());
            UUID id = UUID.fromString(service.getId());
            DiscoveryManagerImpl.getInstance().getServiceIdagentIdMatch().put(
                    (Long) reference.getProperty("service.id"), id);
            a.getSystemProperties().printSystemConfiguration();
            updateAgentListOnTopic(agentToEvent(a, AgentEventType.AGENT_UPDATE));
        } else {
            Agent agent = new Agent();
            UUID id = UUID.fromString(service.getId());
            SystemProperties props = service.getSystemProperties();
            agent.setId(id);
            agent.setSystemProperties(props);
            DiscoveryManagerImpl.getInstance().getServiceIdagentIdMatch().put(
                    (Long) reference.getProperty("service.id"), id);
            DiscoveryManagerImpl.getInstance().getAgentList().add(agent);
            System.out.println();
            logger.info("Agent {0} registered with the following OS properties", id);
            props.printSystemConfiguration();
            updateAgentListOnTopic(agentToEvent(agent, AgentEventType.AGENT_ADD));
        }

    }

    /**
     * @param service
     *            the service unbinded
     * @param reference
     *            the reference of the service
     */
    @Unbind(optional = true, aggregate = true)
    public void unbindAgentDiscoveryService(AgentDiscoveryService service, ServiceReference reference) {
        UUID id = DiscoveryManagerImpl.getInstance().getServiceIdagentIdMatch()
                .get(reference.getProperty("service.id"));
        Agent a = DiscoveryManagerImpl.getInstance().getAgentFromUUID(id);
        a.setState(AgentState.STOPPED);
        a.setIsDiscoveryRunning(false);
        DiscoveryManagerImpl.getInstance().getServiceIdagentIdMatch().remove(reference.getProperty("service.id"));
        a.removeAllRegisteredServices();
        logger.info("Agent {0} unregistered", id);
        updateAgentListOnTopic(agentToEvent(a, AgentEventType.AGENT_UPDATE));
    }

    /**
     * Convert an Agent to an AgentEvent
     * 
     * @param a
     *            the agent to convert
     * @param type
     *            the type of the event
     * @return the AgentEvent created
     */
    public AgentEvent agentToEvent(Agent a, AgentEventType type) {
        AgentEvent event = new AgentEvent();
        event.setAgent(new Agent(a));
        event.setType(type.ordinal());
        return event;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.agent.server.discovery.DiscoveryManager#updateAgent(java.lang.String)
     */
    public void updateAgent(String id) {
        Agent a = DiscoveryManagerImpl.getInstance().getAgentFromUUID(UUID.fromString(id));
        updateAgentListOnTopic(agentToEvent(new Agent(a), AgentEventType.AGENT_UPDATE));
    }

    /**
     * Send an update on the JMS Topic about the agent
     * 
     * @param event
     */
    public void updateAgentListOnTopic(AgentEvent event) {
        ClassLoader old = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(DiscoveryManagerSLSB.class.getClassLoader());
        try {
            Context ctx = new InitialContext();
            ConnectionFactory factory = (ConnectionFactory) ctx.lookup("JTCF");
            Topic topic = (Topic) ctx.lookup("jasmineAgent");

            Connection connection = factory.createConnection();
            Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
            Message message = session.createObjectMessage(event);
            MessageProducer producer = session.createProducer(topic);
            producer.send(message);
            // Close JMS objects
            session.close();
            connection.close();
            ctx.close();
        } catch (Exception e) {
            logger.error("Error : {0}", e.getMessage());
            e.printStackTrace(System.err);
        } finally {
            Thread.currentThread().setContextClassLoader(old);
        }
    }

}
