/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Utilities.java 9171 2011-10-13 14:30:56Z richardd $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.agent.common.utils;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.Socket;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

/**
 * Utilities method
 * 
 * @author Julien Vey
 */
public class Utilities {

    /**
     * Get a line in a file
     * 
     * @param search
     * @param file
     * @return the line corresponding to the search
     */
    public static String getStringInFile(String search, File file) {

        // lecture du fichier texte
        try {
            InputStream ips = new FileInputStream(file);
            InputStreamReader ipsr = new InputStreamReader(ips);
            BufferedReader br = new BufferedReader(ipsr);
            String ligne;
            while ((ligne = br.readLine()) != null) {
                if (ligne.contains(search) && !ligne.startsWith("#")) {
                    br.close();
                    return ligne;
                }
            }
            br.close();
            return null;

        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;

    }

    /**
     * Convert an input stream into a file
     * 
     * @param inputStream
     *            the input stream to convert
     * @return the file correspondign to the input stream
     */
    public static File convertStreamToFile(InputStream inputStream) {
        try {
            File f = new File("outFile");
            OutputStream out = new FileOutputStream(f);

            byte buf[] = new byte[1024];

            int len;
            while ((len = inputStream.read(buf)) > 0) {
                out.write(buf, 0, len);
            }
            out.close();
            inputStream.close();
            return f;
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * Read a file and return the line it contains as a list of String
     * 
     * @param f
     *            the file to read
     * @return the list of the lines in the file
     */
    public static List<String> readFile(File f) {
        List<String> list = new ArrayList<String>();
        try {
            InputStream ips = new FileInputStream(f);
            InputStreamReader ipsr = new InputStreamReader(ips);
            BufferedReader br = new BufferedReader(ipsr);
            String ligne;
            while ((ligne = br.readLine()) != null) {
                list.add(ligne);
            }
            br.close();

        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
        return list;
    }

    /**
     * Scan a port on a host to check if it's currently accepting connections.
     * 
     * @param port
     *            the port to scan
     * @param host
     *            the host to scan
     * @return true if the port on the host is open. False otherwise
     */
    public static boolean scan(int port, InetAddress host) {
        try {
            Socket s = new Socket(host, port);
            s.close();
            return true;
        } catch (Exception e) {
            // The remote host is not listening on this port
            return false;
        }
    }

    /**
     * Write a String in a file
     * 
     * @param s
     *            the String to write
     * @param f
     *            the file in which the string will be written
     */
    public static void writeInFile(String s, File f) {
        try {
            FileWriter fw = new FileWriter(f);
            BufferedWriter bw = new BufferedWriter(fw);
            PrintWriter outputFile = new PrintWriter(bw);
            outputFile.println(s);
            outputFile.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    // Utility methods to get the local address even on a linux host

    /**
     * Returns an InetAddress representing the address of the localhost. Every attempt is made to find an address for
     * this host that is not the loopback address. If no other address can be found, the loopback will be returned.
     * 
     * @return InetAddress - the address of localhost
     * @throws UnknownHostException
     *             - if there is a problem determing the address
     */
    public static InetAddress getLocalHost() throws UnknownHostException {
        InetAddress localHost = InetAddress.getLocalHost();
        if (!localHost.isLoopbackAddress())
            return localHost;
        InetAddress[] addrs = getAllLocalUsingNetworkInterface();
        for (int i = 0; i < addrs.length; i++) {
            if (!addrs[i].isLoopbackAddress() && !addrs[i].getHostAddress().contains(":"))
                return addrs[i];
        }
        return localHost;
    }

    /**
     * This method attempts to find all InetAddresses for this machine in a conventional way (via InetAddress). If only
     * one address is found and it is the loopback, an attempt is made to determine the addresses for this machine using
     * NetworkInterface.
     * 
     * @return InetAddress[] - all addresses assigned to the local machine
     * @throws UnknownHostException
     *             - if there is a problem determining addresses
     */
    public static InetAddress[] getAllLocal() throws UnknownHostException {
        InetAddress[] iAddresses = InetAddress.getAllByName("127.0.0.1");
        if (iAddresses.length != 1)
            return iAddresses;
        if (!iAddresses[0].isLoopbackAddress())
            return iAddresses;
        return getAllLocalUsingNetworkInterface();

    }

    /**
     * Utility method that delegates to the methods of NetworkInterface to determine addresses for this machine.
     * 
     * @return InetAddress[] - all addresses found from the NetworkInterfaces
     * @throws UnknownHostException
     *             - if there is a problem determining addresses
     */
    private static InetAddress[] getAllLocalUsingNetworkInterface() throws UnknownHostException {
        ArrayList<InetAddress> addresses = new ArrayList<InetAddress>();
        Enumeration<NetworkInterface> e = null;
        try {
            e = NetworkInterface.getNetworkInterfaces();
        } catch (SocketException ex) {
            throw new UnknownHostException("127.0.0.1");
        }
        while (e.hasMoreElements()) {
            NetworkInterface ni = (NetworkInterface) e.nextElement();
            for (Enumeration<InetAddress> e2 = ni.getInetAddresses(); e2.hasMoreElements();) {
                addresses.add(e2.nextElement());
            }
        }
        InetAddress[] iAddresses = new InetAddress[addresses.size()];
        for (int i = 0; i < iAddresses.length; i++) {
            iAddresses[i] = (InetAddress) addresses.get(i);
        }
        return iAddresses;
    }
}
