/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Apache2Discovery.java 9171 2011-10-13 14:30:56Z richardd $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.agent.remote.discovery.application.apache2;

import java.io.File;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.ow2.jasmine.agent.common.discovery.Application;
import org.ow2.jasmine.agent.common.discovery.ApplicationDiscoveryService;
import org.ow2.jasmine.agent.common.discovery.ApplicationState;
import org.ow2.jasmine.agent.common.utils.Utilities;
import org.ow2.jasmine.agent.remote.RemoteService;
import org.ow2.jasmine.agent.utils.filesystem.discovery.FileSystemDiscovery;
import org.ow2.jasmine.agent.utils.pattern.DiscoveryPattern;
import org.ow2.jasmine.agent.utils.pattern.PatternUtility;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Apache2 Discovery Service implementation
 * 
 * @author Julien Vey
 */
@Component(name = "JASMINe Agent Apache2 Discovery", propagation = true)
@Provides
public class Apache2Discovery extends RemoteService implements ApplicationDiscoveryService {

    /**
     * Logger
     */
    private static Log logger = LogFactory.getLog(Apache2Discovery.class);

    /**
     * Type of the discovered managed elements
     */
    private static final String TYPE = "Apache 2";

    /**
     * Apache2 pattern file v1
     */
    private static final String APACHE2_PATTERN_FILE_V1 = "/apache2v1.xml";

    /**
     * Apache2 pattern file v2
     */
    private static final String APACHE2_PATTERN_FILE_V2 = "/apache2v2.xml";

    /**
     * the FileSystemDiscoveryService used in this class.
     */
    @Requires
    private FileSystemDiscovery fsService;

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.agent.common.discovery.ApplicationDiscoveryService#discoverApplications()
     */
    public synchronized List<Application> discoverApplications() {

        return discoverApplications_dynamic(discoverApplications_static());
    }

    /**
     * discover applications in a dynamic way
     * 
     * @param discoverApplicationsStatic
     *            the applications already discovered in a static way
     * @return the application discovered dynamically
     */
    private List<Application> discoverApplications_dynamic(List<Application> discoverApplicationsStatic) {

        for (Application application : discoverApplicationsStatic) {
            try {
                boolean portIsOpen = Utilities.scan(Integer.parseInt(application
                        .getProperty(Apache2ApplicationPropertiesMap.LISTENING_PORT_KEY)), InetAddress.getLocalHost());

                if (portIsOpen) {
                    application.setState(ApplicationState.MAYBE_RUNNING);
                } else {
                    application.setState(ApplicationState.STOPPED);
                }
            } catch (Exception e) {
                System.out.println("Exception");
                application.setState(ApplicationState.STOPPED);
            }
        }
        return discoverApplicationsStatic;
    }

    /**
     * discover applications in a static way
     * 
     * @return
     */
    private List<Application> discoverApplications_static() {

        List<Application> applicationList = new ArrayList<Application>();

        DiscoveryPattern apache2Patternv1 = PatternUtility.convertFileToPattern(getClass().getResourceAsStream(
                APACHE2_PATTERN_FILE_V1));
        DiscoveryPattern apache2Patternv2 = PatternUtility.convertFileToPattern(getClass().getResourceAsStream(
                APACHE2_PATTERN_FILE_V2));

        logger.info("Starting discovering Apache2");

        List<File> fileList = fsService.searchPattern(apache2Patternv1);
        for (File file : fileList) {
            Application application = new Application();
            application.setRootLocation(file.getAbsolutePath());
            application.setState(ApplicationState.UNKNOWN);
            application.setApplicationType(TYPE);

            Apache2ApplicationPropertiesMap properties = new Apache2ApplicationPropertiesMap();
            File conf;
            conf = new File(application.getRootLocation() + "/ports.conf");
            String listenConf = Utilities.getStringInFile("Listen", conf);
            listenConf = listenConf.replaceAll("Listen", "");
            String ip;
            String port;
            if (listenConf.contains(":")) { /* ip + port */
                StringTokenizer tokenizer = new StringTokenizer(listenConf.replace(" ", ""), ":");
                ip = tokenizer.nextToken();
                port = tokenizer.nextToken();
            } else { /* only port */
                port = listenConf.replaceAll(" ", "");
                ip = "";
            }
            properties.setListeningAddress(ip);
            properties.setListeningPort(Integer.parseInt(port));
            application.setApplicationProperties(properties.getPropertiesList());
            applicationList.add(application);
        }

        List<File> fileList2 = fsService.searchPattern(apache2Patternv2);
        for (File file : fileList2) {
            Application application = new Application();
            application.setRootLocation(file.getAbsolutePath());
            application.setState(ApplicationState.UNKNOWN);
            application.setApplicationType(TYPE);

            Apache2ApplicationPropertiesMap properties = new Apache2ApplicationPropertiesMap();
            File conf;
            conf = new File(application.getRootLocation() + "/conf/httpd.conf");
            String listenConf = Utilities.getStringInFile("Listen", conf);
            listenConf = listenConf.replaceAll("Listen", "");
            String ip = null;
            String port;
            if (listenConf.contains(":")) { /* ip + port */
                StringTokenizer tokenizer = new StringTokenizer(listenConf.replace(" ", ""), ":");
                ip = tokenizer.nextToken();
                port = tokenizer.nextToken();
            } else { /* only port */
                port = listenConf.replaceAll(" ", "");
            }
            if (ip != null) {
                properties.setListeningAddress(ip);
            }
            properties.setListeningPort(Integer.parseInt(port));
            application.setApplicationProperties(properties.getPropertiesList());
            applicationList.add(application);

        }

        return applicationList;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.agent.common.discovery.ApplicationDiscoveryService#getType()
     */
    public String getType() {
        return TYPE;
    }
}
