/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JonasDiscovery.java 9171 2011-10-13 14:30:56Z richardd $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.agent.remote.discovery.application.jonas;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.InvocationHandler;
import java.net.URL;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;

import javax.management.MBeanServerConnection;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.jgroups.Channel;
import org.jgroups.conf.ConfiguratorFactory;
import org.jgroups.conf.ProtocolStackConfigurator;
import org.jgroups.stack.IpAddress;
import org.ow2.jasmine.agent.common.discovery.ApplicationDiscoveryService;
import org.ow2.jasmine.agent.common.discovery.Application;
import org.ow2.jasmine.agent.common.discovery.ApplicationState;
import org.ow2.jasmine.agent.common.utils.AgentFileFilter;
import org.ow2.jasmine.agent.common.utils.PropertyManager;
import org.ow2.jasmine.agent.common.utils.Utilities;
import org.ow2.jasmine.agent.remote.RemoteService;
import org.ow2.jasmine.agent.utils.filesystem.discovery.FileSystemDiscovery;
import org.ow2.jasmine.agent.utils.pattern.DiscoveryPattern;
import org.ow2.jasmine.agent.utils.pattern.PatternUtility;
import org.ow2.jonas.discovery.DiscoveryState;
import org.ow2.jonas.discovery.base.comm.DiscEvent;
import org.ow2.jonas.discovery.jgroups.utils.IDiscoveryChannel;
import org.ow2.jonas.discovery.jgroups.utils.JGroupsDiscoveryUtils;
import org.ow2.util.cluster.jgroups.ConnectionManager;
import org.ow2.util.cluster.jgroups.IChannel;
import org.ow2.util.cluster.jgroups.JChannelWrapper;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Implementation of the ApplicationDiscoveryService for JOnAS
 * 
 * @author Julien Vey
 */
@Component(name = "JASMINe Agent Jonas Discovery", propagation = true)
@Provides
public class JonasDiscovery extends RemoteService implements ApplicationDiscoveryService {

    /**
     * Logger
     */
    private static Log logger = LogFactory.getLog(JonasDiscovery.class);

    /**
     * Type of the discovered managed elements
     */
    private static final String TYPE = "JOnAS";

    /**
     * Resource file describing a jonas_base pattern
     */
    private static final String JONAS_BASE_PATTERN_FILE = "/jonasbase.xml";

    /**
     * Resource file describing a micro jonas pattern
     */
    private static final String MICRO_JONAS_PATTERN_FILE = "/microjonas.xml";

    /**
     * Resource file describing a jonas_root pattern
     */
    private static final String JONAS_ROOT_PATTERN_FILE = "/jonasroot.xml";

    /**
     * Resource file describing a jonas that has already been started at least once
     */
    private static final String ALREADY_STARTED_JONAS_PATTERN_FILE = "/alreadystartedjonas.xml";

    /**
     * version properties file name
     */
    private static final String VERSION_PROPERTY_FILE_NAME = "versions.properties";

    /**
     * jonas version property key
     */
    private static final String JONAS_VERSION_PROPERTY_NAME = "org.ow2.jonas";

    /**
     * Carol properties file name
     */
    private static final String CAROL_FILE_NAME = "carol.properties";

    /**
     * conf directory name
     */
    private static final String CONF_DIRECTORY_NAME = "conf";

    /**
     * carol protocol property key
     */
    private static final String CAROL_PROTOCOLS_KEY = "carol.protocols";

    /**
     * jonas_base mbean attribute name
     */
    private static final String JONAS_BASE_MBEAN_ATTRIBUTE_NAME = "jonasBase";

    /**
     * jonas_root mbean attribute name
     */
    private static final String JONAS_ROOT_MBEAN_ATTRIBUTE_NAME = "jonasRoot";

    /**
     * jonas J2ee server objectname pattern
     */
    private static final String JONAS_J2EE_SERVER_OBJECTNAME_PATTERN = "*:j2eeType=J2EEServer,*";

    /**
     * jonas services key
     */
    private static final String JONAS_SERVICES_KEY = "jonas.services";

    /**
     * discovery service name
     */
    private static final String DISCOVERY_SERVICE_NAME = "discovery";

    /**
     * jgroups configuration file key
     */
    private static final String JGROUPS_CONFIGURATION_FILE_KEY = "jonas.service.discovery.jgroups.conf";

    /**
     * jgroups name key
     */
    private static final String JGROUPS_GROUP_NAME_KEY = "jonas.service.discovery.group.name";

    /**
     * jgroups timeout key
     */
    private static final String JGROUPS_TIMEOUT_KEY = "jonas.service.discovery.reconnection.timeout";

    /**
     * jonas name key
     */
    private static final String JONAS_NAME_KEY = "jonas.name";

    /**
     * jonas domain key
     */
    private static final String JONAS_DOMAIN_KEY = "domain.name";

    /**
     * name of the jonas property file
     */
    private static final String JONAS_PROPS_FILE_NAME = "jonas.properties";

    /**
     * local server name by default
     */
    private static final String LOCAL_SERVER_NAME = "jasmineagent";

    /**
     * default waiting time to listen for jgroups event
     */
    private static final int DEFAULT_WAITING_TIME_FOR_JGROUPS_EVENT = 2000;

    /**
     * the FileSystemDiscoveryService used in this class.
     */
    @Requires
    private FileSystemDiscovery fsService;

    /**
     * discover the JOnAS instances installed or running on the system
     * 
     * @return the list of JOnAS instances found
     * @see org.ow2.jasmine.agent.common.discovery.ApplicationDiscoveryService#discoverApplications()
     */
    public synchronized List<Application> discoverApplications() {
        return discoverApplications_dynamic(discoverApplications_static());
    }

    /**
     * discover the JOnAS instances in a dynamic way. This means it will try to define if the instances that have been
     * found by the static method are running or not
     * 
     * @param discoverApplicationsStatic
     *            the list of application discovered in a static way
     * @return the list of application discovered in a dynamic way
     */
    @SuppressWarnings({ "deprecation", "static-access" })
    private List<Application> discoverApplications_dynamic(List<Application> discoverApplicationsStatic) {

        File f = null;
        if (!System.getProperty("os.name").startsWith("Win")) {
            try {
                Process p = Runtime.getRuntime().exec("/bin/ps aux");
                f = Utilities.convertStreamToFile(p.getInputStream());
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        for (Application application : discoverApplicationsStatic) {

            // try to connect to a default url
            String jmxUrl = getPossibleUrlFromCarol(getPropertyFile(CAROL_FILE_NAME, application.getRootLocation()),
                    getPropertyFile(JONAS_PROPS_FILE_NAME, application.getRootLocation()));
            if (tryConnectToUrl(jmxUrl)) {
                String jonasRoot = getJonasRoot(jmxUrl);
                String jonasBase = getJonasBase(jmxUrl);

                if (jonasBase.equalsIgnoreCase(application.getRootLocation())) {
                    JonasApplicationPropertiesMap jonasProps = new JonasApplicationPropertiesMap(application
                            .getApplicationProperties());
                    jonasProps.setJmxConnector(jmxUrl);
                    jonasProps.setConfirmedJonasBase(jonasBase);
                    jonasProps.setConfirmedJonasRoot(jonasRoot);
                    application.setState(ApplicationState.RUNNING);
                    application.setApplicationProperties(jonasProps.getPropertiesList());
                }

            }

            // trying using process table (only on Unix-based systems)
            if (f != null) {
                if (!application.stateAsApplicationState().equals(ApplicationState.RUNNING)
                        && !application.getProperty(JonasApplicationPropertiesMap.JONAS_TYPE_KEY).equals(
                                JonasType.JONAS_ROOT.toString())) {
                    String res = getStringInFile("-Djonas.base=" + application.getRootLocation(), f);
                    if (res != null) {
                        if (res.contains("-Djonas.name")) {
                            StringTokenizer tokenizer = new StringTokenizer(res, " ");
                            boolean keepon = true;
                            while (keepon && tokenizer.hasMoreTokens()) {
                                String next = tokenizer.nextToken();
                                if (next.contains("-Djonas.name")) {
                                    keepon = false;
                                    next = next.replaceAll("-Djonas.name=", "");
                                    String jmxNewUrl = getPossibleUrlFromCarol(getPropertyFile(CAROL_FILE_NAME,
                                            application.getRootLocation()), getPropertyFile(JONAS_PROPS_FILE_NAME,
                                            application.getRootLocation()), next);
                                    if (tryConnectToUrl(jmxNewUrl)) {
                                        String jonasRoot = getJonasRoot(jmxNewUrl);
                                        String jonasBase = getJonasBase(jmxNewUrl);

                                        if (jonasBase.equalsIgnoreCase(application.getRootLocation())) {
                                            JonasApplicationPropertiesMap jonasProps = new JonasApplicationPropertiesMap(
                                                    application.getApplicationProperties());
                                            jonasProps.setJmxConnector(jmxNewUrl);
                                            jonasProps.setConfirmedJonasBase(jonasBase);
                                            jonasProps.setConfirmedJonasRoot(jonasRoot);
                                            application.setState(ApplicationState.RUNNING);
                                            application.setApplicationProperties(jonasProps.getPropertiesList());
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }

            // trying using Jonas internal discovery service
            if (!application.stateAsApplicationState().equals(ApplicationState.RUNNING)) {
                try {
                    File conf = new File(application.getRootLocation(), "conf");
                    File confFile = new File(conf, JONAS_PROPS_FILE_NAME);
                    PropertyManager prop = new PropertyManager(confFile.getAbsolutePath());
                    String runningServices = prop.getProperty(JONAS_SERVICES_KEY);
                    if (runningServices.contains(DISCOVERY_SERVICE_NAME)) { /* Discovey Service is started */
                        URL confile;
                        IChannel channel = null;

                        confile = new File(conf, prop.getProperty(JGROUPS_CONFIGURATION_FILE_KEY)).toURL();
                        ProtocolStackConfigurator configurator = ConfiguratorFactory.getStackConfigurator(confile);
                        channel = new JChannelWrapper(configurator.getProtocolStackString());

                        // Avoid our own messages
                        channel.setOpt(Channel.LOCAL, false);
                        // Set the auto-reconnect option for enabling a node to leave and re-join the cluster
                        channel.setOpt(Channel.AUTO_RECONNECT, true);
                        InvocationHandler invocationHandler = new ConnectionManager(Integer.parseInt(prop
                                .getProperty(JGROUPS_TIMEOUT_KEY)), channel, IDiscoveryChannel.class);
                        channel.addChannelListener((ConnectionManager) invocationHandler);
                        // IChannel chan = (IChannel) Proxy.newProxyInstance(IChannel.class.getClassLoader(),
                        // new Class[] { IChannel.class }, invocationHandler);

                        channel.connect(prop.getProperty(JGROUPS_GROUP_NAME_KEY));

                        String sourceAddr = ((IpAddress) channel.getLocalAddress()).getIpAddress().toString();
                        int sourcePort = ((IpAddress) channel.getLocalAddress()).getPort();
                        String serverName = LOCAL_SERVER_NAME;
                        String domainName = prop.getProperty(JONAS_DOMAIN_KEY);
                        String serverId = LOCAL_SERVER_NAME;
                        String[] connectorURLs = null;
                        boolean isMaster = true;

                        DiscEvent event = new DiscEvent(sourceAddr, sourcePort, serverName, domainName, serverId,
                                connectorURLs, isMaster);
                        event.setState(DiscoveryState.STARTUP);
                        // We have created the message, we can now send it

                        JGroupsHandler comHandler = new JGroupsHandler();
                        channel.setReceiver(comHandler);

                        channel.send(null, (IpAddress) channel.getLocalAddress(), JGroupsDiscoveryUtils
                                .objectToBytes(event));
                        
                        Thread.currentThread().sleep(DEFAULT_WAITING_TIME_FOR_JGROUPS_EVENT);
                                                
                        channel.close();
                        List<String> urlConnectors = comHandler.getUrlConnectors();
                        for (String connector : urlConnectors) {
                            if (tryConnectToUrl(connector)) {
                                // logger.info("Success");
                                String jonasRoot = getJonasRoot(connector);
                                String jonasBase = getJonasBase(connector);

                                if (jonasBase.equalsIgnoreCase(application.getRootLocation())) {
                                    JonasApplicationPropertiesMap jonasProps = new JonasApplicationPropertiesMap(
                                            application.getApplicationProperties());
                                    jonasProps.setJmxConnector(connector);
                                    jonasProps.setConfirmedJonasBase(jonasBase);
                                    jonasProps.setConfirmedJonasRoot(jonasRoot);
                                    application.setState(ApplicationState.RUNNING);
                                    application.setApplicationProperties(jonasProps.getPropertiesList());
                                    break;
                                }
                            }
                        }
                    }
                } catch (Exception e) {
                    logger
                            .error(
                                    "An error occured while trying to use Jonas Internal Discovery Service to discover the application, exception is {0}",
                                    e.getMessage());
                }
            }

            if (!application.stateAsApplicationState().equals(ApplicationState.RUNNING)) {
                // we have tried everything, the instance must be down
                application.setState(ApplicationState.STOPPED);
            }
        }
        if (f != null) {
            f.delete();
        }

        return discoverApplicationsStatic;
    }

    /**
     * get the jonas base path for a specified server identified by its jmx connector
     * 
     * @param jmxUrl
     *            the jmx connector url
     * @return the location of the jonas_base of the application
     */
    private String getJonasBase(String jmxUrl) {
        return getAttributeFromMBeanServer(jmxUrl, JONAS_J2EE_SERVER_OBJECTNAME_PATTERN,
                JONAS_BASE_MBEAN_ATTRIBUTE_NAME);
    }

    /**
     * get the jonas root path for a specified server identified by its jmx connector
     * 
     * @param jmxUrl
     *            the jmx connector url
     * @return the location of the jonas_root of the application
     */
    private String getJonasRoot(String jmxUrl) {
        return getAttributeFromMBeanServer(jmxUrl, JONAS_J2EE_SERVER_OBJECTNAME_PATTERN,
                JONAS_ROOT_MBEAN_ATTRIBUTE_NAME);
    }

    /**
     * get an attribute with a mbean pattern from a jmx connector
     * 
     * @param jmxUrl
     *            the jmx connector url
     * @param mbeanPattern
     *            the mbean pattern
     * @param attributeName
     *            the name of the attribute
     * @return the value of the attribute
     */
    private String getAttributeFromMBeanServer(String jmxUrl, String mbeanPattern, String attributeName) {
        ClassLoader old = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(JonasDiscovery.class.getClassLoader());
        try {
            JMXServiceURL url = new JMXServiceURL(jmxUrl);
            JMXConnector jmxc = JMXConnectorFactory.connect(url, null);
            MBeanServerConnection mbsc = jmxc.getMBeanServerConnection();

            ObjectName myObjectName = new ObjectName(mbeanPattern);

            Set<ObjectInstance> mbeanSet = mbsc.queryMBeans(myObjectName, null);
            ObjectInstance mbeanObjectInstance = (ObjectInstance) mbeanSet.toArray()[0];
            ObjectName mbeanObjectName = mbeanObjectInstance.getObjectName();
            String attributeValue = (String) mbsc.getAttribute(mbeanObjectName, attributeName);

            return attributeValue;

        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            Thread.currentThread().setContextClassLoader(old);
        }

        return null;
    }

    /**
     * will try to connect to a specified jmx connector
     * 
     * @param jmxUrl
     *            the jmx connector url
     * @return true if the connection succeeded, false if it did not
     */
    private Boolean tryConnectToUrl(String jmxUrl) {
        ClassLoader old = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(JonasDiscovery.class.getClassLoader());
        try {
            JMXServiceURL url = new JMXServiceURL(jmxUrl);
            JMXConnector jmxc = JMXConnectorFactory.connect(url, null);
            jmxc.close();
            return true;
        } catch (Exception e) {
            return false;
        } finally {
            Thread.currentThread().setContextClassLoader(old);
        }
    }

    /**
     * get a default url using properties from the carol property file
     * 
     * @param carolPropertyFile
     *            the path of the carol property file
     * @param jonasPropertyFile
     *            the path of the jonas property file
     * @return the possible url for the jmx connector
     */
    private String getPossibleUrlFromCarol(File carolPropertyFile, File jonasPropertyFile) {
        PropertyManager props = new PropertyManager(carolPropertyFile.getAbsolutePath());
        String carolProtocol = props.getProperty(CAROL_PROTOCOLS_KEY);
        String carolUrl = props.getProperty("carol." + carolProtocol + ".url");

        StringTokenizer tokenizer = new StringTokenizer(carolUrl, ":");
        String protocol = tokenizer.nextToken();
        String host = tokenizer.nextToken();
        String port = tokenizer.nextToken();

        props = new PropertyManager(jonasPropertyFile.getAbsolutePath());
        String jonasName = props.getProperty(JONAS_NAME_KEY);

        return "service:jmx:" + protocol + ":" + host + "/jndi/" + protocol + ":" + host + ":" + port + "/"
                + carolProtocol + "connector_" + jonasName;

    }

    /**
     * get a default url using properties from the carol property file and a provided name
     * 
     * @param carolPropertyFile
     *            the path of the carol property file
     * @param jonasPropertyFile
     *            the path of the jonas property file
     * @param name
     *            the name of the jonas instance
     * @return the possible url for the jmx connector
     */
    private String getPossibleUrlFromCarol(File carolPropertyFile, File jonasPropertyFile, String name) {
        PropertyManager props = new PropertyManager(carolPropertyFile.getAbsolutePath());
        String carolProtocol = props.getProperty(CAROL_PROTOCOLS_KEY);
        String carolUrl = props.getProperty("carol." + carolProtocol + ".url");

        StringTokenizer tokenizer = new StringTokenizer(carolUrl, ":");
        String protocol = tokenizer.nextToken();
        String host = tokenizer.nextToken();
        String port = tokenizer.nextToken();

        return "service:jmx:" + protocol + ":" + host + "/jndi/" + protocol + ":" + host + ":" + port + "/"
                + carolProtocol + "connector_" + name;
    }

    /**
     * get the property file from the conf directory in the root location given
     * 
     * @param fileName
     *            the name of the property file
     * @param rootLocation
     *            the root location of the application
     * @return the File corresponding to the query
     */
    private File getPropertyFile(String fileName, String rootLocation) {
        File root = new File(rootLocation);
        File[] subRoot = root.listFiles(new AgentFileFilter(CONF_DIRECTORY_NAME));
        File[] confFiles = subRoot[0].listFiles(new AgentFileFilter(fileName));
        return confFiles[0];

    }

    /**
     * discover the JOnAS instances installed or running on the system in a static way
     * 
     * @return the list of JOnAS instances found in a static way
     */
    public List<Application> discoverApplications_static() {

        List<Application> applicationList = new LinkedList<Application>();

        DiscoveryPattern jonasBasePattern = PatternUtility.convertFileToPattern(getClass().getResourceAsStream(
                JONAS_BASE_PATTERN_FILE));
        DiscoveryPattern jonasRootPattern = PatternUtility.convertFileToPattern(getClass().getResourceAsStream(
                JONAS_ROOT_PATTERN_FILE));
        DiscoveryPattern jonasStartedOncePattern = PatternUtility.convertFileToPattern(getClass().getResourceAsStream(
                ALREADY_STARTED_JONAS_PATTERN_FILE));
        DiscoveryPattern microJonasPattern = PatternUtility.convertFileToPattern(getClass().getResourceAsStream(
                MICRO_JONAS_PATTERN_FILE));

        // System.out.println(PatternUtility.matchPattern(new File("/home/veyj/Programmation/micro-jonas-5.2.0-M1"),
        // jonasBasePattern));

        logger.info("Starting discovering JOnAS");
        List<File> fileList = fsService.searchPattern(jonasBasePattern);
        for (File base : fileList) {
            Application application = new Application();
            application.setApplicationType(TYPE);
            application.setRootLocation(base.getAbsolutePath());
            application.setState(ApplicationState.UNKNOWN);
            Boolean isJonasBase = true;
            Boolean isJonasRoot = PatternUtility.matchPattern(base, jonasRootPattern);
            Boolean hasAlreadyBeenStartedOnce = PatternUtility.matchPattern(base, jonasStartedOncePattern);
            Boolean isMicroJonas = PatternUtility.matchPattern(base, microJonasPattern);

            Boolean wellFormedDirectory = true;

            JonasApplicationPropertiesMap jonasProps = new JonasApplicationPropertiesMap();
            if ((isJonasBase && !isJonasRoot) && !isMicroJonas) {
                jonasProps.setJonasType(JonasType.JONAS_BASE);
            } else if (isJonasRoot && !hasAlreadyBeenStartedOnce) {
                jonasProps.setJonasType(JonasType.JONAS_ROOT);
            } else if (isJonasRoot && hasAlreadyBeenStartedOnce) {
                jonasProps.setJonasType(JonasType.JONAS_ROOT_AND_BASE);
            } else if (!isJonasRoot && isMicroJonas) {
                jonasProps.setJonasType(JonasType.MICRO_JONAS);
            }

            else {
                wellFormedDirectory = false;
            }

            if (wellFormedDirectory) {
                if (jonasProps.getJonasType().equals(JonasType.JONAS_ROOT)
                        || jonasProps.getJonasType().equals(JonasType.JONAS_ROOT_AND_BASE)
                        || jonasProps.getJonasType().equals(JonasType.MICRO_JONAS)) {
                    PropertyManager props = new PropertyManager(base.getAbsolutePath() + "/"
                            + VERSION_PROPERTY_FILE_NAME);
                    String jonasVersion = props.getProperty(JONAS_VERSION_PROPERTY_NAME);
                    jonasProps.setJonasVersion(jonasVersion);
                }

                application.setApplicationProperties(jonasProps.getPropertiesList());

                applicationList.add(application);
            }

        }

        return applicationList;
    }

    /**
     * Get a line in file which contains a given string
     * 
     * @param search
     *            the string to search
     * @param file
     *            the file in which search
     * @return the line containing the search
     */
    private String getStringInFile(String search, File file) {

        // lecture du fichier texte
        try {
            InputStream ips = new FileInputStream(file);
            InputStreamReader ipsr = new InputStreamReader(ips);
            BufferedReader br = new BufferedReader(ipsr);
            String ligne;
            while ((ligne = br.readLine()) != null) {
                if (ligne.contains(search)) {
                    br.close();
                    return ligne;
                }
            }
            br.close();
            return null;

        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;

    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.agent.common.discovery.ApplicationDiscoveryService#getType()
     */
    public String getType() {
        return TYPE;
    }

}
