/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: SystemDiscovery.java 9171 2011-10-13 14:30:56Z richardd $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.agent.remote.discovery.system;

import java.io.File;
import java.io.IOException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.StringTokenizer;

import org.ow2.jasmine.agent.common.discovery.Interface;
import org.ow2.jasmine.agent.common.discovery.SystemProperties;
import org.ow2.jasmine.agent.common.utils.Utilities;
import org.ow2.jonas.lib.bootstrap.JProp;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * This class enables the discovery of the OS properties
 * 
 * @author Julien Vey
 */
public class SystemDiscovery {

    /**
     * Logger
     */
    private static Log logger = LogFactory.getLog(SystemDiscovery.class);

    /**
     * get the system configuration
     * 
     * @return an instance of SystemProperties containing the system configuration
     */
    public SystemProperties getSystemProperties() {
        SystemProperties props = new SystemProperties(System.getProperty("os.name", ""), System.getProperty(
                "os.version", ""), System.getProperty("os.arch", ""), Runtime.getRuntime().availableProcessors(),
                getLinuxDistribution(), getIP(), getNetworkInterfaces(), getHostname(), getInstanceName());
        return props;

    }

    /**
     * get the name of the jonas instance
     * 
     * @return the name of the jonas instance
     */
    private String getInstanceName() {
        return JProp.getInstance().getServerName();
    }

    /**
     * get the hostname
     * 
     * @return the hostname
     */
    private String getHostname() {
        try {
            return InetAddress.getLocalHost().getHostName();
        } catch (Exception e) {
            logger.error("Error while getting the hostname");
            return "";
        }
    }

    /**
     * get the ip address of the host
     * 
     * @return the ip address
     */
    private String getIP() {
        try {
            return getLocalHost().getHostAddress();
        } catch (UnknownHostException e) {
            logger.error("Error while getting the ip address");
            return "";
        }
    }

    /**
     * get the list of all network interfaces of the host
     * 
     * @return the mac address
     */
    private List<Interface> getNetworkInterfaces() {
        List<Interface> interfaces = new ArrayList<Interface>();
        Enumeration<NetworkInterface> e = null;

        try {
            e = NetworkInterface.getNetworkInterfaces();
        } catch (SocketException ex) {
            logger.error("Error while getting network interfaces");
            return new ArrayList<Interface>();
        }
        while (e.hasMoreElements()) {
            NetworkInterface ni = (NetworkInterface) e.nextElement();
            Interface n = new Interface();
            n.setIpAddress(ni.getInetAddresses().nextElement().getHostAddress());
            n.setName(ni.getName());
            // TODO : Uncomment this part when the agent will be build using jdk1.6
            // try {
            // byte[] mac = ni.getHardwareAddress();
            // StringBuffer buffer = new StringBuffer();
            // System.out.println();
            // for(byte b : mac) {
            // String next = Integer.toHexString(b);
            // if(next.length()==1){
            // next = "0".concat(next);
            // } else if (next.length()>2){
            // next = next.substring(6, 8);
            // }
            // buffer.append(next);
            // buffer.append(":");
            // }
            // n.setMacAddress(buffer.toString().toUpperCase().substring(0,17));
            // } catch (Exception e1) {
            // n.setMacAddress("");
            // }
            n.setMacAddress("");

            interfaces.add(n);
        }
        return interfaces;
    }

    /**
     * get the distribution if not on a windows host
     * 
     * @return the distribution
     */
    public String getLinuxDistribution() {
        String res = "";
        if (!System.getProperty("os.name").startsWith("Win")) {
            File f;
            try {
                Process p = Runtime.getRuntime().exec("cat /etc/issue");
                f = Utilities.convertStreamToFile(p.getInputStream());
                List<String> list = Utilities.readFile(f);
                res = list.get(0);
                if (!(res != null && res.length() > 0)) {
                    res = "";
                } else {
                    String res2 = "";
                    StringTokenizer tokenizer = new StringTokenizer(res, "\\");
                    res2 = tokenizer.nextToken();
                    res = res2;
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return res;
    }

    // Utility methods to get the local address even on a linux host

    /**
     * Returns an InetAddress representing the address of the localhost
     * 
     * @return InetAddress - the address of localhost
     * @throws UnknownHostException
     *             - if there is a problem determing the address
     */
    public static InetAddress getLocalHost() throws UnknownHostException {
        InetAddress localHost = InetAddress.getLocalHost();
        if (!localHost.isLoopbackAddress())
            return localHost;
        InetAddress[] addrs = getAllLocalUsingNetworkInterface();
        for (int i = 0; i < addrs.length; i++) {
            if (!addrs[i].isLoopbackAddress() && !addrs[i].getHostAddress().contains(":"))
                return addrs[i];
        }
        return localHost;
    }

    /**
     * Utility method that delegates to the methods of NetworkInterface to determine addresses for this machine.
     * 
     * @return InetAddress[] - all addresses found from the NetworkInterfaces
     * @throws UnknownHostException
     *             - if there is a problem determining addresses
     */
    private static InetAddress[] getAllLocalUsingNetworkInterface() throws UnknownHostException {
        ArrayList<InetAddress> addresses = new ArrayList<InetAddress>();
        Enumeration<NetworkInterface> e = null;
        try {
            e = NetworkInterface.getNetworkInterfaces();
        } catch (SocketException ex) {
            throw new UnknownHostException("127.0.0.1");
        }
        while (e.hasMoreElements()) {
            NetworkInterface ni = e.nextElement();
            for (Enumeration<InetAddress> e2 = ni.getInetAddresses(); e2.hasMoreElements();) {
                addresses.add(e2.nextElement());
            }
        }
        InetAddress[] iAddresses = new InetAddress[addresses.size()];
        for (int i = 0; i < iAddresses.length; i++) {
            iAddresses[i] =  addresses.get(i);
        }
        return iAddresses;
    }
}
