/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Agent.java 9171 2011-10-13 14:30:56Z richardd $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.agent.server;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import javax.naming.Context;
import javax.naming.InitialContext;

import org.ow2.jasmine.agent.common.discovery.Application;
import org.ow2.jasmine.agent.common.discovery.ApplicationDiscoveryService;
import org.ow2.jasmine.agent.common.discovery.SystemProperties;
import org.ow2.jasmine.agent.common.jk.JkManagerService;
import org.ow2.jasmine.agent.server.discovery.DiscoveryManager;
import org.ow2.jasmine.agent.server.discovery.DiscoveryManagerImpl;
import org.ow2.jasmine.agent.server.discovery.DiscoveryServiceRunner;

/**
 * Class representing an agent
 * 
 * @author Julien Vey
 */
public class Agent implements Serializable {

    /**
     * serial version UID
     */
    private static final long serialVersionUID = 3670722052592266373L;

    /**
     * The OS properties of the agent.
     */
    private SystemProperties systemProperties;

    /**
     * The map of all known application discovery services with the managed element type as key
     */
    private List<ApplicationDiscoveryService> applicationDiscoveryServiceList = new ArrayList<ApplicationDiscoveryService>();

    /**
     * The jkmanager service of the agent if available
     */
    private JkManagerService jkManagerService;

    /**
     * The Agent State
     */
    private AgentState state = AgentState.RUNNING;

    /**
     * The date when the last discovery has been launched on this agent
     */
    private Date lastDiscovery = null;

    /**
     * The list of all the applications currently discovered
     */
    private List<Application> applicationList = new ArrayList<Application>();

    /**
     * A temporary list to avoid re-initialize the true application list when a new discovery is launched
     */
    private List<Application> temp_applicationList = new ArrayList<Application>();

    /**
     * A boolean to know when a discovery is currently running on an agent.
     */
    private Boolean isDiscoveryRunning = false;

    /**
     * the id of the agent
     */
    private String agentId;

    /**
     * Default constructor of the agent
     */
    public Agent() {
        systemProperties = new SystemProperties();
    }

    /**
     * Agent constructor with SystemProperties
     * 
     * @param props
     */
    public Agent(SystemProperties props) {
        systemProperties = props;
    }

    /**
     * Clone an agent into this one, excepted the applicationDiscoveryService List
     * 
     * @param clone
     */
    public Agent(Agent clone) {
        this.applicationDiscoveryServiceList = new ArrayList<ApplicationDiscoveryService>();
        this.applicationList = clone.applicationList;
        this.agentId = clone.agentId;
        this.isDiscoveryRunning = clone.isDiscoveryRunning;
        this.lastDiscovery = clone.lastDiscovery;
        this.state = clone.state;
        this.systemProperties = clone.systemProperties;
    }

    /**
     * @return the systemProperties
     */
    public SystemProperties getSystemProperties() {
        return systemProperties;
    }

    /**
     * @param systemProperties
     *            the systemProperties to set
     */
    public void setSystemProperties(SystemProperties systemProperties) {
        this.systemProperties = systemProperties;
    }

    /**
     * @return the jkManagerService
     */
    public JkManagerService getJkManagerService() {
        return jkManagerService;
    }

    /**
     * @param jkManagerService
     *            the jkManagerService to set
     */
    public void setJkManagerService(JkManagerService jkManagerService) {
        this.jkManagerService = jkManagerService;
    }

    /**
     * Add an ApplicationDiscoveryService to the list
     * 
     * @param service
     *            the service to add
     */
    public void addApplicationDiscoveryService(ApplicationDiscoveryService service) {
        applicationDiscoveryServiceList.add(service);
    }

    /**
     * Get the ID of the Agent
     * 
     * @return the id of the agent
     */
    public String getAgentId() {
        return agentId;
    }

    /**
     * Set the ID of the agent
     * 
     * @param id
     *            the id to set
     */
    public void setAgentId(String id) {
        this.agentId = id;
    }

    /**
     * Get the ID of the agent as an UUID
     * 
     * @return the id as an UUID
     */
    public UUID getIdAsUUID() {
        return UUID.fromString(agentId);
    }

    /**
     * Set the ID of the agent from an UUID
     * 
     * @param id2
     *            the id to set
     */
    public void setId(UUID id2) {
        this.agentId = id2.toString();
    }

    /**
     * @return the applicationDiscoveryServiceList
     */
    public List<ApplicationDiscoveryService> getApplicationDiscoveryServiceList() {
        return applicationDiscoveryServiceList;
    }

    /**
     * @param applicationDiscoveryServiceList
     *            the applicationDiscoveryServiceList to set
     */
    public void setApplicationDiscoveryServiceList(List<ApplicationDiscoveryService> applicationDiscoveryServiceList) {
        this.applicationDiscoveryServiceList = applicationDiscoveryServiceList;
    }

    /**
     * @return the state
     */
    public int getState() {
        return state.ordinal();
    }

    /**
     * Set the state from an int value
     * 
     * @param state
     *            the state to set
     */
    public void setState(int state) {
        this.state = AgentState.values()[state];
    }

    /**
     * Set the state from an AgentState
     * 
     * @param state
     */
    public void setState(AgentState state) {
        this.state = state;
    }

    /**
     * Get the LastDiscovery Date in a String format
     * 
     * @return the last discovery date in a String format
     */
    public String getLastDiscovery() {
        if (lastDiscovery != null) {
            return lastDiscovery.toString();
        }
        return null;
    }

    /**
     * @param lastDiscovery
     */
    public void setLastDiscovery(String lastDiscovery) {

    }

    /**
     * Remove all registered Services
     */
    public void removeAllRegisteredServices() {
        jkManagerService = null;
        applicationDiscoveryServiceList = new ArrayList<ApplicationDiscoveryService>();

    }

    /**
     * @return the applicationList
     */
    public List<Application> getApplicationList() {
        return applicationList;
    }

    /**
     * @param applicationList
     *            the applicationList to set
     */
    public void setApplicationList(List<Application> applicationList) {
        this.applicationList = applicationList;
    }

    /**
     * Add discovered applications to the agent application list
     * 
     * @param appList
     *            the list of discovered applications to add
     */
    public synchronized void addDiscoveredApplication(List<Application> appList) {
        temp_applicationList.addAll(appList);
        for (Application discoveredApp : appList) {
            Boolean found = false;
            for (Application agentApp : applicationList) {
                if (agentApp.getApplicationType().equals(discoveredApp.getApplicationType())
                        && agentApp.getRootLocation().equals(discoveredApp.getRootLocation())) {
                    agentApp.setApplicationProperties(discoveredApp.getApplicationProperties());
                    agentApp.setState(discoveredApp.getState());
                    found = true;
                    break;
                }
            }
            if (!found) {
                applicationList.add(discoveredApp);
            }
        }
    }

    /**
     * Check the application list to see if a application has been removed or not since the last discovery
     */
    public synchronized void checkAppList() {
        List<Application> toRemove = new ArrayList<Application>();
        for (Application agentApp : applicationList) {
            Boolean found = false;
            for (Application tempApp : temp_applicationList) {
                if (agentApp.getApplicationType().equals(tempApp.getApplicationType())
                        && agentApp.getRootLocation().equals(tempApp.getRootLocation())) {
                    found = true;
                    break;
                }
            }
            if (!found) {
                toRemove.add(agentApp);
            }
        }

        for (Application agentAppR : toRemove) {
            applicationList.remove(agentAppR);
        }
    }

    /**
     * Launch Application discovery
     */
    public synchronized void discoverApplications() {
        temp_applicationList = new ArrayList<Application>();
        lastDiscovery = new Date();
        DiscoveryServiceRunner runner = new DiscoveryServiceRunner(applicationDiscoveryServiceList, this);
        isDiscoveryRunning = true;
        runner.start();
        sendUpdate();
    }

    /**
     * @return the isDiscoveryRunning
     */
    public synchronized Boolean getIsDiscoveryRunning() {
        return isDiscoveryRunning;
    }

    /**
     * @param isDiscoveryRunning
     *            the isDiscoveryRunning to set
     */
    public synchronized void setIsDiscoveryRunning(Boolean isDiscoveryRunning) {
        this.isDiscoveryRunning = isDiscoveryRunning;
    }

    /**
     * Send an update to the DiscoveryManager that the agent has been updated
     */
    public synchronized void sendUpdate() {
        ClassLoader old = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(DiscoveryManagerImpl.class.getClassLoader());
        try {
            Context ctx = new InitialContext();
            DiscoveryManager managerBean = (DiscoveryManager) ctx.lookup("agent/discoverymanager");
            managerBean.updateAgent(this.agentId);
            ctx.close();
        } catch (Exception e) {
            e.printStackTrace(System.err);
        } finally {
            Thread.currentThread().setContextClassLoader(old);
        }
    }

    /**
     * Method called at the end of a discovery
     */
    public synchronized void endDiscovery() {
        setIsDiscoveryRunning(false);
        checkAppList();
        sendUpdate();
    }
}
