/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: GroupsManagerSLSB.java 9171 2011-10-13 14:30:56Z richardd $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.agent.server.groups;

import java.util.List;
import java.util.UUID;

import javax.ejb.Remote;
import javax.ejb.Stateless;
import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.Message;
import javax.jms.MessageProducer;
import javax.jms.Session;
import javax.jms.Topic;
import javax.naming.Context;
import javax.naming.InitialContext;

import org.ow2.jasmine.agent.server.groups.event.GroupEvent;
import org.ow2.jasmine.agent.server.groups.event.GroupEventType;

/**
 * Stateless Session Bean which enables group management
 * 
 * @author Julien Vey
 */
@Stateless(mappedName = "agent/groupsmanager")
@Remote(GroupsManager.class)
public class GroupsManagerSLSB implements GroupsManager {

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.agent.server.groups.GroupsManager#addGroup(org.ow2.jasmine.agent.server.groups.Group)
     */
    public void addGroup(Group g) {
        GroupsManagerImpl.getInstance().addGroup(g);
        updateGroupListOnTopic(groupToEvent(g, GroupEventType.GROUP_ADD));
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.agent.server.groups.GroupsManager#getGroupList()
     */
    public List<Group> getGroupList() {
        return GroupsManagerImpl.getInstance().getGroupList();
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.agent.server.groups.GroupsManager#removeGroup(org.ow2.jasmine.agent.server.groups.Group)
     */
    public void removeGroup(Group g) {
        GroupsManagerImpl.getInstance().removeGroup(g);
        updateGroupListOnTopic(groupToEvent(g, GroupEventType.GROUP_REMOVE));
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.agent.server.groups.GroupsManager#updateGroup(org.ow2.jasmine.agent.server.groups.Group)
     */
    public void updateGroup(Group g) {
        GroupsManagerImpl.getInstance().updateGroup(g);
        updateGroupListOnTopic(groupToEvent(g, GroupEventType.GROUP_UPDATE));
    }

    /**
     * Create a GroupEvent from a group and a GroupEventType
     * 
     * @param g
     *            the group
     * @param type
     *            the GroupEvent type
     * @return the GroupEvent created
     */
    public GroupEvent groupToEvent(Group g, GroupEventType type) {
        GroupEvent event = new GroupEvent();
        event.setGroup(g);
        event.setType(type.ordinal());
        return event;
    }

    /**
     * Send an update to the JMS Topic
     * 
     * @param event
     */
    public void updateGroupListOnTopic(GroupEvent event) {
        ClassLoader old = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(GroupsManagerSLSB.class.getClassLoader());
        try {
            Context ctx = new InitialContext();
            ConnectionFactory factory = (ConnectionFactory) ctx.lookup("JTCF");
            Topic topic = (Topic) ctx.lookup("jasmineAgent");

            Connection connection = factory.createConnection();
            Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
            Message message = session.createObjectMessage(event);
            MessageProducer producer = session.createProducer(topic);
            producer.send(message);
            // Close JMS objects
            session.close();
            connection.close();
            ctx.close();
        } catch (Exception e) {
            e.printStackTrace(System.err);
        } finally {
            Thread.currentThread().setContextClassLoader(old);
        }
    }
}
