/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: PatternUtility.java 9171 2011-10-13 14:30:56Z richardd $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.agent.utils.pattern;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.ow2.jasmine.agent.common.utils.Utilities;
import org.ow2.jasmine.agent.utils.filesystem.discovery.impl.CustomFileFilter;

/**
 * Multiple methods used to manage DiscoveryPattern
 * 
 * @author Julien Vey
 */
public class PatternUtility {




    /**
     * Convert a file to a DiscoveryPattern
     * 
     * @param patternFile
     * @return
     */
    public static DiscoveryPattern convertFileToPattern(File patternFile) {

        SAXBuilder sxb = new SAXBuilder();
        Document document = null;
        try {
            document = sxb.build(patternFile);
        } catch (JDOMException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        if (document != null) {
            Element racine = document.getRootElement();
            return new DiscoveryPattern(getChildren(racine));
        }
        return null;

       

    }

    /**
     * Method used by the xml parser. Get the child list corresponding to a DOM element
     * 
     * @param el
     *            the dom element
     * @return the child list of this element
     */
    public static List<Child> getChildren(Element el) {

        List<Child> childList = new LinkedList<Child>();

        for (Object obj : el.getChildren()) {
            Element element = (Element) obj;
            Child c = new Child(element.getAttributeValue("name"), convertStringToChildType(element.getName()));
            c.setChildren(getChildren(element));
            childList.add(c);
        }
        return childList;
    }

    /**
     * Converts a String to a ChildType
     * 
     * @param type
     *            the type to convert
     * @return the ChildType corresponding to the type
     */
    public static ChildType convertStringToChildType(String type) {
        if (type.equalsIgnoreCase("file")) {
            return ChildType.FILE;
        } else if (type.equalsIgnoreCase("dir")) {
            return ChildType.DIRECTORY;
        }
        return null;
    }

    /**
     * Match a list of child with a directory
     * 
     * @param directory
     *            the directory to match
     * @param childList
     *            the child list to match
     * @return true if the pattern is included in the directory, false if it is not
     */
    public static boolean matchPattern(File directory, List<Child> childList) {

        if (childList.size() == 0) {
            return true;
        }

        boolean result = true;
        Iterator<Child> iterator = childList.iterator();

        while (iterator.hasNext() && result) {

            Child c = iterator.next();
            File[] childFiles = directory.listFiles(new CustomFileFilter(c));
            if (childFiles.length == 1) {
                result = matchPattern(childFiles[0], c.getChildren());
            } else {
                result = false;
            }
        }

        if (iterator.hasNext()) {
            return false;
        } else {
            return result;
        }
    }

    /**
     * Match a pattern with a directory
     * 
     * @param directory
     *            the directory to match
     * @param pattern
     *            the pattern to match
     * @return true if there is a match, false if there is not
     */
    public static boolean matchPattern(File directory, DiscoveryPattern pattern) {
        return matchPattern(directory, pattern.getChildList());
    }

    /**
     * Convert an input stream in a Pattern
     * 
     * @param resourceAsStream
     *            the stream to convert
     * @return the pattern
     */
    public static DiscoveryPattern convertFileToPattern(InputStream resourceAsStream) {

        File f = Utilities.convertStreamToFile(resourceAsStream);
        DiscoveryPattern p = convertFileToPattern(f);
        f.delete();
        return p;

    }

}
