/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Application.java 9171 2011-10-13 14:30:56Z richardd $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.agent.common.discovery;

import java.io.Serializable;
import java.util.List;
import java.util.UUID;

/**
 * An application is a managed element to be discovered, such as Apache Http, JOnAS, PostGreSQL... It consists of a
 * state ,a rootLocation on the file system, an id and a list of properties
 * 
 * @author Julien Vey
 */
public class Application implements Serializable {

    /**
     * serial version UID
     */
    private static final long serialVersionUID = -9182558372622788839L;

    /**
     * The state of the application
     */
    private ApplicationState state;

    /**
     * the root location of the application on the drive
     */
    private String rootLocation;

    /**
     * The map of the application properties. Can be a JMX URL, a listening port. Anything needed for the application.
     */
    private List<ApplicationProperty> applicationProperties;

    /**
     * id of the application
     */
    private String id;

    /**
     * Default constructor of the application
     */
    public Application() {
        id = UUID.randomUUID().toString();
    }

    /**
     * The type of the application
     */
    private String applicationType;

    /**
     * Constructor with fields
     * 
     * @param state
     *            the state of the application
     * @param rootLocation
     *            the rootLocation of the application
     * @param applicationProperties
     *            the properties of the application
     */
    public Application(ApplicationState state, String rootLocation, List<ApplicationProperty> applicationProperties,
            String type) {
        id = UUID.randomUUID().toString();
        this.state = state;
        this.rootLocation = rootLocation;
        this.applicationProperties = applicationProperties;
        this.applicationType = type;
    }

    // Getters and Setters
    /**
     * @return the state as an int
     */
    public int getState() {
        return state.ordinal();
    }

    /**
     * @return the state as an ApplicationState
     */
    public ApplicationState stateAsApplicationState() {
        return state;
    }

    /**
     * @param state
     *            the state to set
     */
    public void setState(int state) {
        this.state = ApplicationState.values()[state];
    }

    /**
     * set the state
     * 
     * @param state
     *            the state to set
     */
    public void setState(ApplicationState state) {
        this.state = state;
    }

    /**
     * @return the rootLocation
     */
    public String getRootLocation() {
        return rootLocation;
    }

    /**
     * @param rootLocation
     *            the rootLocation to set
     */
    public void setRootLocation(String rootLocation) {
        this.rootLocation = rootLocation;
    }

    /**
     * @return the applicationProperties
     */
    public List<ApplicationProperty> getApplicationProperties() {
        return applicationProperties;
    }

    /**
     * @param applicationProperties
     *            the applicationProperties to set
     */
    public void setApplicationProperties(List<ApplicationProperty> applicationProperties) {
        this.applicationProperties = applicationProperties;
    }

    /**
     * @param key
     * @return
     */
    public String getProperty(String key) {
        for (ApplicationProperty prop : applicationProperties) {
            if (prop.getPropertyName().equals(key)) {
                return prop.getPropertyValue();
            }
        }
        return null;
    }

    /**
     * @return the applicationType
     */
    public String getApplicationType() {
        return applicationType;
    }

    /**
     * @param applicationType
     *            the applicationType to set
     */
    public void setApplicationType(String applicationType) {
        this.applicationType = applicationType;
    }

    /**
     * @return the id
     */
    public String getId() {
        return id;
    }

    /**
     * @param id
     *            the id to set
     */
    public void setId(String id) {
        this.id = id;
    }
}
