/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: SystemProperties.java 9171 2011-10-13 14:30:56Z richardd $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.agent.common.discovery;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Class that contains the properties of the Operating System
 * 
 * @author Julien Vey
 */
public class SystemProperties implements Serializable {

    /**
     * serial version UID
     */
    private static final long serialVersionUID = 8881901462673499785L;

    /**
     * Logger
     */
    private static Log logger = LogFactory.getLog(SystemProperties.class);

    /**
     * The name of the operating system
     */
    String osName;
    /**
     * The version of the operating system
     */
    String osVersion;
    /**
     * the architecture of the operating system
     */
    String osArchitecture;

    /**
     * Number of processors
     */
    int nbProc;

    /**
     * Distribution name for a linux host
     */
    String distribution;

    /**
     * IP address
     */
    String ip;

    /**
     * hostname
     */
    String hostname;

    /**
     * agent instance name
     */
    String instanceName;

    /**
     * list of network interfaces
     */
    List<Interface> networkInterfaces;

    /**
     * Default Constructor with no parameters
     */
    public SystemProperties() {
        osName = "";
        osVersion = "";
        osArchitecture = "";
        nbProc = 0;
        distribution = "";
        ip = "";
        networkInterfaces = new ArrayList<Interface>();
        hostname = "";
        instanceName = "";
    }

    /**
     * Constructor with the properties
     * 
     * @param name
     *            the name of the OS
     * @param version
     *            the version of the OS
     * @param architecture
     *            the architecture of the OS
     * @param nbProc
     *            the number of processors
     * @param distribution
     *            the distribution of the OS
     * @param ip
     *            the IP address of the node
     * @param networkInterfaces
     *            the list of network interfaces
     * @param hostname
     *            the hostname
     * @param instanceName
     *            the instance name
     */
    public SystemProperties(String name, String version, String architecture, int nbProc, String distribution,
            String ip, List<Interface> networkInterfaces, String hostname, String instanceName) {
        this.osName = name;
        this.osVersion = version;
        this.osArchitecture = architecture;
        this.nbProc = nbProc;
        this.distribution = distribution;
        this.ip = ip;
        this.networkInterfaces = networkInterfaces;
        this.hostname = hostname;
        this.instanceName = instanceName;
    }

    /**
     * Print the system configuration
     */
    public void printSystemConfiguration() {
        logger.info("OS Name : {0}", osName);
        logger.info("OS Architecture : {0}", osArchitecture);
        logger.info("OS Version : {0}", osVersion);
        logger.info("Number of processors : {0}", nbProc);
        logger.info("Distribution : {0}", distribution);
        logger.info("IP Address : {0}", ip);
        logger.info("Hostname : {0}", hostname);
        logger.info("Logical Name : {0}", instanceName);
        logger.info("Network interfaces :");
        for (Interface n : networkInterfaces) {
            logger.info("\t Interface {0}", n.getName());
            logger.info("\t\t IP Address : {0}", n.getIpAddress());
            logger.info("\t\t MAC Address : {0}", n.getMacAddress());
        }
    }

    // Getters and Setters
    /**
     * @return the osName
     */
    public String getOsName() {
        return osName;
    }

    /**
     * @param osName
     *            the osName to set
     */
    public void setOsName(String osName) {
        this.osName = osName;
    }

    /**
     * @return the osVersion
     */
    public String getOsVersion() {
        return osVersion;
    }

    /**
     * @param osVersion
     *            the osVersion to set
     */
    public void setOsVersion(String osVersion) {
        this.osVersion = osVersion;
    }

    /**
     * @return the osArchitecture
     */
    public String getOsArchitecture() {
        return osArchitecture;
    }

    /**
     * @param osArchitecture
     *            the osArchitecture to set
     */
    public void setOsArchitecture(String osArchitecture) {
        this.osArchitecture = osArchitecture;
    }

    /**
     * @return the nbProc
     */
    public int getNbProc() {
        return nbProc;
    }

    /**
     * @param nbProc
     *            the nbProc to set
     */
    public void setNbProc(int nbProc) {
        this.nbProc = nbProc;
    }

    /**
     * @return the distribution
     */
    public String getDistribution() {
        return distribution;
    }

    /**
     * @param distribution
     *            the distribution to set
     */
    public void setDistribution(String distribution) {
        this.distribution = distribution;
    }

    /**
     * @return the logger
     */
    public static Log getLogger() {
        return logger;
    }

    /**
     * @param logger
     *            the logger to set
     */
    public static void setLogger(Log logger) {
        SystemProperties.logger = logger;
    }

    /**
     * @return the ip
     */
    public String getIp() {
        return ip;
    }

    /**
     * @param ip
     *            the ip to set
     */
    public void setIp(String ip) {
        this.ip = ip;
    }

    /**
     * @return the networkInterfaces
     */
    public List<Interface> getNetworkInterfaces() {
        return networkInterfaces;
    }

    /**
     * @param networkInterfaces
     *            the networkInterfaces to set
     */
    public void setNetworkInterfaces(List<Interface> networkInterfaces) {
        this.networkInterfaces = networkInterfaces;
    }

    /**
     * @return the hostname
     */
    public String getHostname() {
        return hostname;
    }

    /**
     * @param hostname
     *            the hostname to set
     */
    public void setHostname(String hostname) {
        this.hostname = hostname;
    }

    /**
     * @return the instanceName
     */
    public String getInstanceName() {
        return instanceName;
    }

    /**
     * @param instanceName
     *            the instanceName to set
     */
    public void setInstanceName(String instanceName) {
        this.instanceName = instanceName;
    }

}
