/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JonasApplicationPropertiesMap.java 9171 2011-10-13 14:30:56Z richardd $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.agent.remote.discovery.application.jonas;

import java.util.ArrayList;
import java.util.List;

import org.ow2.jasmine.agent.common.discovery.ApplicationProperty;

/**
 * Virtual map of jonas properties
 * 
 * @author Julien Vey
 */
public class JonasApplicationPropertiesMap {

    /**
     * JMX connector key
     */
    public static final String JMX_CONNECTOR_KEY = "JMX Connector";

    /**
     * JOnAS type key
     */
    public static final String JONAS_TYPE_KEY = "Installation Type";

    /**
     * JOnAS version key
     */
    public static final String JONAS_VERSION_KEY = "Version";

    /**
     * JOnAS base key
     */
    public static final String JONAS_BASE_KEY = "JOnAS Base";

    /**
     * JOnAS Root Key
     */
    public static final String JONAS_ROOT_KEY = "JOnAS Root";

    /**
     * jmx connector
     */
    private String jmxConnector;

    /**
     * jonas type
     */
    private JonasType jonasType;

    /**
     * jonas version
     */
    private String jonasVersion;

    /**
     * confirmed Jonas BASE for a running application
     */
    private String confirmedJonasBase;

    /**
     * confirmed jonas root for a running application
     */
    private String confirmedJonasRoot;

    /**
     * default constructor with no parameters
     */
    public JonasApplicationPropertiesMap() {
        jmxConnector = null;
    }

    /**
     * Constructor using an already existing map
     * 
     * @param properties
     *            the map to convert to jonas properties map
     */
    public JonasApplicationPropertiesMap(List<ApplicationProperty> properties) {
        setJmxConnector(getPropFromList(JMX_CONNECTOR_KEY, properties));
        setJonasType(stringToJonasType(getPropFromList(JONAS_TYPE_KEY, properties)));
        setJonasVersion(getPropFromList(JONAS_VERSION_KEY, properties));
        setConfirmedJonasBase(getPropFromList(JONAS_BASE_KEY, properties));
        setConfirmedJonasRoot(getPropFromList(JONAS_ROOT_KEY, properties));
    }

    /**
     * Get a property from the list of properties
     * 
     * @param key
     *            the key of the property
     * @param props
     *            the list of properties
     * @return the property value found
     */
    private String getPropFromList(String key, List<ApplicationProperty> props) {
        for (ApplicationProperty prop : props) {
            if (prop.getPropertyName().equals(key)) {
                return prop.getPropertyValue();
            }
        }
        return null;
    }

    /**
     * convert a string to a jonas TYPE
     * 
     * @param string
     *            the string to convert
     * @return the type corresponding to the string
     */
    private JonasType stringToJonasType(String string) {
        if (string.equalsIgnoreCase(JonasType.JONAS_BASE.toString())) {
            return JonasType.JONAS_BASE;
        }
        if (string.equalsIgnoreCase(JonasType.JONAS_ROOT.toString())) {
            return JonasType.JONAS_ROOT;
        }
        if (string.equalsIgnoreCase(JonasType.JONAS_ROOT_AND_BASE.toString())) {
            return JonasType.JONAS_ROOT_AND_BASE;
        }
        if (string.equalsIgnoreCase(JonasType.MICRO_JONAS.toString())) {
            return JonasType.MICRO_JONAS;
        }
        return null;
    }

    /**
     * constructor with a jmx connector
     * 
     * @param jmxConnector
     */
    public JonasApplicationPropertiesMap(String jmxConnector) {
        this.jmxConnector = jmxConnector;
    }

    /**
     * get the object as a property map
     * 
     * @return the map of properties
     */
    public List<ApplicationProperty> getPropertiesList() {
        List<ApplicationProperty> list = new ArrayList<ApplicationProperty>();
        list.add(new ApplicationProperty(JMX_CONNECTOR_KEY, jmxConnector));
        list.add(new ApplicationProperty(JONAS_TYPE_KEY, jonasType.toString()));
        list.add(new ApplicationProperty(JONAS_VERSION_KEY, jonasVersion));
        list.add(new ApplicationProperty(JONAS_BASE_KEY, confirmedJonasBase));
        list.add(new ApplicationProperty(JONAS_ROOT_KEY, confirmedJonasRoot));

        return list;
    }

    /**
     * @return the jmxConnector
     */
    public String getJmxConnector() {
        return jmxConnector;
    }

    /**
     * @param jmxConnector
     *            the jmxConnector to set
     */
    public void setJmxConnector(String jmxConnector) {
        this.jmxConnector = jmxConnector;
    }

    /**
     * @return the jonasType
     */
    public JonasType getJonasType() {
        return jonasType;
    }

    /**
     * @param jonasType
     *            the jonasType to set
     */
    public void setJonasType(JonasType jonasType) {
        this.jonasType = jonasType;
    }

    /**
     * @return the jonasVersion
     */
    public String getJonasVersion() {
        return jonasVersion;
    }

    /**
     * @param jonasVersion
     *            the jonasVersion to set
     */
    public void setJonasVersion(String jonasVersion) {
        this.jonasVersion = jonasVersion;
    }

    /**
     * @return the confirmedJonasBase
     */
    public String getConfirmedJonasBase() {
        return confirmedJonasBase;
    }

    /**
     * @param confirmedJonasBase
     *            the confirmedJonasBase to set
     */
    public void setConfirmedJonasBase(String confirmedJonasBase) {
        this.confirmedJonasBase = confirmedJonasBase;
    }

    /**
     * @return the confirmedJonasRoot
     */
    public String getConfirmedJonasRoot() {
        return confirmedJonasRoot;
    }

    /**
     * @param confirmedJonasRoot
     *            the confirmedJonasRoot to set
     */
    public void setConfirmedJonasRoot(String confirmedJonasRoot) {
        this.confirmedJonasRoot = confirmedJonasRoot;
    }
}
