/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: PostgreSQLDiscovery.java 9171 2011-10-13 14:30:56Z richardd $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.agent.remote.discovery.application.postgresql;

import java.io.File;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.ow2.jasmine.agent.common.discovery.Application;
import org.ow2.jasmine.agent.common.discovery.ApplicationDiscoveryService;
import org.ow2.jasmine.agent.common.discovery.ApplicationState;
import org.ow2.jasmine.agent.common.utils.AgentFileFilter;
import org.ow2.jasmine.agent.common.utils.Utilities;
import org.ow2.jasmine.agent.remote.RemoteService;
import org.ow2.jasmine.agent.utils.filesystem.discovery.FileSystemDiscovery;
import org.ow2.jasmine.agent.utils.pattern.DiscoveryPattern;
import org.ow2.jasmine.agent.utils.pattern.PatternUtility;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Apache2 Discovery Service implementation
 * 
 * @author Julien Vey
 */
@Component(name = "JASMINe Agent PostgreSQL Discovery", propagation = true)
@Provides
public class PostgreSQLDiscovery extends RemoteService implements ApplicationDiscoveryService {

    /**
     * Postgre pattern file
     */
    public static final String POSTGRE_PATTERN_FILE = "/postgresql.xml";

    public static final String POSTGRE_PATTERN_FILE_2 = "/postgresqlv2.xml";

    /**
     * Type of the discovered managed elements
     */
    public static final String TYPE = "PostgreSQL";
    /**
     * Postgre configuration port key
     */
    public static final String POSTGRE_CONF_PORT_KEY = "port";
        
    /**
     * Postgre configuration listening addresses key
     */
    public static final String POSTGRE_CONF_LISTEN_ADDRESSES_KEY = "listen_addresses";

    /**
     * Postgre configuration maximum connections key
     */
    public static final String POSTGRE_CONF_MAX_CONNECTIONS_KEY = "max_connections";

    /**
     * Logger
     */
    private static Log logger = LogFactory.getLog(PostgreSQLDiscovery.class);

    /**
     * the FileSystemDiscoveryService used in this class.
     */
    @Requires
    private FileSystemDiscovery fsService;

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.agent.common.discovery.ApplicationDiscoveryService#discoverApplications()
     */
    public synchronized List<Application> discoverApplications() {

        return discoverApplications_dynamic(discoverApplications_static());
    }

    /**
     * dynamic discovery
     * 
     * @param discoveredApplicationsStatic
     *            applications discovered by the static method
     * @return the applications discovered dynamically
     */
    private List<Application> discoverApplications_dynamic(List<Application> discoveredApplicationsStatic) {

        for (Application application : discoveredApplicationsStatic) {
            PostgreSQLApplicationPropertiesMap map = new PostgreSQLApplicationPropertiesMap(application
                    .getApplicationProperties());
            int port = map.getPort();
            boolean isListening = false;
            try {
                isListening = Utilities.scan(port, InetAddress.getLocalHost());
            } catch (UnknownHostException e) {
                e.printStackTrace();
            }
            if (isListening) {
                application.setState(ApplicationState.MAYBE_RUNNING);
            }
        }
        return discoveredApplicationsStatic;
    }

    /**
     * static discovery
     * 
     * @return the applications discovered statically
     */
    private List<Application> discoverApplications_static() {
        List<Application> applicationList = new ArrayList<Application>();

        DiscoveryPattern postgrePattern = PatternUtility.convertFileToPattern(getClass().getResourceAsStream(
                POSTGRE_PATTERN_FILE));
        DiscoveryPattern postgrePattern2 = PatternUtility.convertFileToPattern(getClass().getResourceAsStream(
                POSTGRE_PATTERN_FILE_2));

        logger.info("Starting discovering Postgresql");

        List<File> fileList = fsService.searchPattern(postgrePattern);
        fileList.addAll(fsService.searchPattern(postgrePattern2));

        for (File file : fileList) {
            Application application = new Application();
            application.setApplicationType(TYPE);
            application.setRootLocation(file.getAbsolutePath());
            application.setState(ApplicationState.UNKNOWN);

            try {
                File confdir;
                try {
                    confdir = (file.listFiles(new AgentFileFilter("data"))[0]);
                } catch (Exception e) {
                    confdir = (file.listFiles(new AgentFileFilter("main"))[0]);
                }
                PostgreSQLApplicationPropertiesMap map = new PostgreSQLApplicationPropertiesMap();
                if (!confdir.canRead()) {
                    logger.warn("Unable to get PostgreSQL configuration at {0}, you do not have enough rights",
                            file.getAbsolutePath());
                } else {
                    File conf = confdir.listFiles(new AgentFileFilter("postgresql.conf"))[0];

                    String portLine = Utilities.getStringInFile(POSTGRE_CONF_PORT_KEY, conf);
                    String listenLine = Utilities.getStringInFile(POSTGRE_CONF_LISTEN_ADDRESSES_KEY, conf);
                    String maxLine = Utilities.getStringInFile(POSTGRE_CONF_MAX_CONNECTIONS_KEY, conf);
                    
                    int port = 0;
                    String listenAddresses = null;
                    int maxConnections = 0;

                    boolean keepon = true;

                    StringTokenizer tokenizer;
                    String token;

                    if (portLine != null) {
                        tokenizer = new StringTokenizer(portLine, " ");
                        while (keepon && tokenizer.hasMoreTokens()) {
                            token = tokenizer.nextToken();
                            if (!token.equals(POSTGRE_CONF_PORT_KEY)) {
                                keepon = false;
                            } else {
                                token = tokenizer.nextToken();
                                if (token.equals("=")) {
                                    port = Integer.parseInt((new StringTokenizer(tokenizer.nextToken(), "\t"))
                                            .nextToken());
                                }
                                keepon = false;
                            }

                        }
                    }

                    if (listenLine != null) {
                        tokenizer = new StringTokenizer(listenLine, " ");
                        keepon = true;
                        while (keepon && tokenizer.hasMoreTokens()) {
                            token = tokenizer.nextToken();
                            if (!token.equals(POSTGRE_CONF_LISTEN_ADDRESSES_KEY)) {
                                keepon = false;
                            } else {
                                token = tokenizer.nextToken();
                                if (token.equals("=")) {
                                    listenAddresses = (new StringTokenizer(tokenizer.nextToken(), "\t")).nextToken();
                                }
                                keepon = false;
                            }
                        }
                    }

                    if (maxLine != null) {
                        tokenizer = new StringTokenizer(maxLine, " ");
                        keepon = true;
                        while (keepon && tokenizer.hasMoreTokens()) {
                            token = tokenizer.nextToken();
                            if (!token.equals(POSTGRE_CONF_MAX_CONNECTIONS_KEY)) {
                                keepon = false;
                            } else {
                                token = tokenizer.nextToken();
                                if (token.equals("=")) {
                                    maxConnections = Integer
                                            .parseInt((new StringTokenizer(tokenizer.nextToken(), "\t")).nextToken());
                                }
                                keepon = false;
                            }
                        }
                    }

                    map.setListeningAddresses(listenAddresses);
                    map.setMaxConnections(maxConnections);
                    map.setPort(port);
                }
                application.setApplicationProperties(map.getPropertiesList());

            } catch (Exception e) {
                logger.warn("Unable to get PostgreSQL configuration at {0}, exception is : {1}",
                        file.getAbsolutePath(), e.getMessage());
                e.printStackTrace();
            }
            applicationList.add(application);
        }

        return applicationList;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.agent.common.discovery.ApplicationDiscoveryService#getType()
     */
    public String getType() {
        return TYPE;
    }
}
