/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: FileSystemDiscoveryImpl.java 9171 2011-10-13 14:30:56Z richardd $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.agent.utils.filesystem.discovery.impl;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Provides;
import org.ow2.jasmine.agent.utils.filesystem.discovery.FileSystemDiscovery;
import org.ow2.jasmine.agent.utils.pattern.DiscoveryPattern;
import org.ow2.jasmine.agent.utils.pattern.PatternUtility;
import org.ow2.jonas.lib.bootstrap.JProp;

/**
 * Implementation of the FileSystemDiscovery service
 * 
 * @author Julien Vey
 */
@Component(name = "JASMINe Agent FileSystem Discovery")
@Provides
public class FileSystemDiscoveryImpl implements FileSystemDiscovery {

    /**
     * Name of the agent properties file
     */
    private static final String AGENT_PROPERTIES_FILE_NAME = "agent.properties";
    /**
     * key of the blacklist in the properties file
     */
    private static final String BLACKLIST_KEY = "discovery.filesystem.blacklist";

    /**
     * The blacklisted prefixes
     */
    private List<String> blackList;

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.agent.utils.filesystem.discovery.FileSystemDiscovery#searchPattern(org.ow2.jasmine.agent.utils
     * .pattern.DiscoveryPattern)
     */
    public List<File> searchPattern(DiscoveryPattern pattern) {
        List<File> result = new ArrayList<File>();
        blackList = getBlackList();
        for (File root : File.listRoots()) {
            result.addAll(scanDirectory(root, pattern));
        }
        return result;
    }

    /**
     * scan a given directory and search for a given pattern
     * 
     * @param directory
     *            the directory to scan
     * @param pattern
     *            the pattern to match
     * @return the list of sub-directories matching the pattern
     */
    public List<File> scanDirectory(File directory, DiscoveryPattern pattern) {
        List<File> result = new ArrayList<File>();
        // We add the result of all sub dirs
        try {
            for (File dir : directory.listFiles(DirectoryFilter.getInstance())) {
                if (!isBlackListed(dir)) {
                    result.addAll(scanDirectory(dir, pattern));
                }
            }
            if (PatternUtility.matchPattern(directory, pattern.getChildList())) {
                result.add(directory);
            }
        } catch (Exception e) {

        }

        return result;
    }

    /**
     * Tells if a directory is blacklisted
     * 
     * @param dir
     *            the directory to test
     * @return true is the directory is blacklisted
     */
    private boolean isBlackListed(File dir) {
        String name = dir.getAbsolutePath();
        for (String black : blackList) {
            if (name.startsWith(black)) {
                return true;
            }
        }
        return false;
    }

    /**
     * retrieve the blacklist from the properties file
     * 
     * @return the blacklist
     */
    @SuppressWarnings("static-access")
    private List<String> getBlackList() {
        List<String> blacklist = new ArrayList<String>();
        try {
            JProp props = new JProp().getInstance(AGENT_PROPERTIES_FILE_NAME);
            String list = props.getValue(BLACKLIST_KEY);
            StringTokenizer tokenizer = new StringTokenizer(list, ",");
            while (tokenizer.hasMoreTokens()) {
                blacklist.add(tokenizer.nextToken());
            }
            return blacklist;
        } catch (Exception e) {
            return new ArrayList<String>();
        }

    }

}
