/**
 * JASMINe Deploy ME [Managed Element]
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XmlLoader.java 6391 2010-04-27 09:24:33Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.deployme.api;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLConnection;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import org.ow2.jasmine.deployme.generated.Topology;
import org.xml.sax.SAXException;

/**
 * Load the xml topology
 * 
 * @author Remy Bresson
 */
public class XmlLoader {

    /**
     * The loaded topology
     */
    Topology topology;

    /**
     * Constructor. Loads the topology
     * 
     * @param topologyUrl path to the xml topology file
     * @throws SAXException
     * @throws JAXBException
     * @throws IOException
     */
    public XmlLoader(final URL topologyUrl) throws SAXException, JAXBException, IOException {
        this.topology = XmlLoader
            .loadSchemaAndFile(DeployMEPropertiesManager.getXsdTopologyPath(), DeployMEPropertiesManager.getTopologyXMLNS(),
                DeployMEPropertiesManager.getTopologySchemaLocation(), "topology", Topology.class, topologyUrl);
    }

    /**
     * Constructor. Loads the topology
     * 
     * @param xml XML to load.
     * @throws SAXException
     * @throws JAXBException
     * @throws IOException
     */
    public XmlLoader(final String xml) throws SAXException, JAXBException, IOException {
        this.topology = XmlLoader.loadSchemaAndFile(DeployMEPropertiesManager.getXsdTopologyPath(), DeployMEPropertiesManager
            .getTopologyXMLNS(), DeployMEPropertiesManager.getTopologySchemaLocation(), "topology", Topology.class, xml);
    }

    /**
     * Verifies an XML file against an XSD and instantiates it using a given
     * class.
     * 
     * @param xsdPath XSD file path.
     * @param xmlns XML namespace, will be added if non found in the XML
     * @param schemaLocation XML schema location, will be added if non found in
     *        the XML
     * @param xmlRoot Root element of the XML (for completing the XML if XSD
     *        xmlns is missing)
     * @param rootClass Root class used for instantiating JAXB.
     * @param urlXML XML to load.
     * @return XML loaded using JAXB and the rootClass.
     * @throws SAXException If creating the schema fails.
     * @throws IOException If fetching the XML fails.
     * @throws JAXBException If parsing the XML fails.
     */
    public static <T> T loadSchemaAndFile(final String xsdPath, final String xmlns, final String schemaLocation,
        final String xmlRoot, final Class<T> rootClass, final URL urlXML) throws SAXException, JAXBException, IOException {
        String xml = readURL(urlXML);
        return XmlLoader.loadSchemaAndFile(xsdPath, xmlns, schemaLocation, xmlRoot, rootClass, xml);
    }

    /**
     * Verifies an XML file against an XSD and instantiates it using a given
     * class.
     * 
     * @param xsdPath XSD file path.
     * @param xmlns XML namespace, will be added if non found in the XML
     * @param schemaLocation XML schema location, will be added if non found in
     *        the XML
     * @param xmlRoot Root element of the XML (for completing the XML if XSD
     *        xmlns is missing)
     * @param rootClass Root class used for instantiating JAXB.
     * @param xml XML to load.
     * @return XML loaded using JAXB and the rootClass.
     * @throws SAXException If creating the schema fails.
     * @throws IOException If fetching the XML fails.
     * @throws JAXBException If parsing the XML fails.
     */
    @SuppressWarnings("unchecked")
    public static <T> T loadSchemaAndFile(final String xsdPath, final String xmlns, final String schemaLocation,
        final String xmlRoot, final Class<T> rootClass, String xml) throws SAXException, JAXBException, IOException {
        JAXBContext jc = JAXBContext.newInstance(rootClass.getPackage().getName());
        Unmarshaller unMarshaller = jc.createUnmarshaller();

        URL xsdURL = rootClass.getClassLoader().getResource(xsdPath);
        SchemaFactory schemaFactory = SchemaFactory.newInstance("http://www.w3.org/2001/XMLSchema");
        Schema schema = schemaFactory.newSchema(xsdURL);
        unMarshaller.setSchema(schema);

        final int xmlRootStart = xml.indexOf("<" + xmlRoot);
        if (xmlRootStart != -1) {
            final int xmlRootEnd = xml.indexOf(">", xmlRootStart);

            if (xmlRootEnd != -1) {
                final int xmlnsIndex = xml.indexOf("xmlns", xmlRootStart);

                if (xmlnsIndex == -1) {
                    xml = xml.substring(0, xmlRootStart) + "<" + xmlRoot + " xmlns=\"" + xmlns
                        + "\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"" + xmlns + " "
                        + schemaLocation + "\"" + xml.substring(xmlRootStart + xmlRoot.length() + 1);
                } else {
                    // If old version, migrate
                    if (xml.indexOf("/deployme-1.1") != -1) {
                        xml = xml.replace("/deployme-1.1", "/deployme-1.2");

                        // JDK has been renamed to JAVA_HOME
                        xml = xml.replace("<jdk>", "<java_home>");
                        xml = xml.replace("</jdk>", "</java_home>");

                        // JDBC supports multiple elements
                        xml = xml.replace("<jdbc>", "<jdbc><jdbc_ra>");
                        xml = xml.replace("</jdbc>", "</jdbc_ra></jdbc>");
                    }
                }
            }
        }
        InputStream xmlInputStream = new ByteArrayInputStream(xml.getBytes());

        return (T) unMarshaller.unmarshal(xmlInputStream);
    }

    private static String readURL(final URL url) throws IOException {
        URLConnection connection = url.openConnection();
        BufferedReader in = new BufferedReader(new InputStreamReader(connection.getInputStream()));

        try {
            StringBuilder sb = new StringBuilder();
            String inputLine;
            while ((inputLine = in.readLine()) != null) {
                if (sb.length() > 0) {
                    sb.append('\n');
                }
                sb.append(inputLine);
            }

            return sb.toString();
        } finally {
            in.close();
        }
    }

    /**
     * Return the loaded topology
     * 
     * @return the topology
     */
    public Topology getTopology() {
        return this.topology;
    }

}
