/**
 * JASMINe Deploy ME [Managed Element]
 * Copyright (C) 2012 Bull S.A.S.
 * Copyright (C) 2012 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DeployMeCommand.java 10090 2012-06-27 20:01:28Z cazauxj $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.shelbie.command;

import org.apache.felix.gogo.commands.Action;
import org.apache.felix.gogo.commands.Argument;
import org.apache.felix.gogo.commands.Command;
import org.apache.felix.gogo.commands.Option;
import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.HandlerDeclaration;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.service.command.CommandSession;
import org.ow2.jasmine.deployme.api.ICommandLine;
import org.ow2.jasmine.deployme.api.IDeployMeService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.io.File;
import java.net.URL;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * DeployME Shelbie Command
 * @author Jeremy Cazaux
 */
@Component
@Command(name="deploy", scope="deployme", description="DeployME Command")
@HandlerDeclaration("<sh:command xmlns:sh='org.ow2.shelbie'/>")
public class DeployMeCommand implements Action {

    /**
     * Dash
     */
    public static final String DASH = "-";

    /**
     * Topology file option
     */
    public static final String TOPOLOGY_FILE_OPTION = DASH + ICommandLine.COMMAND_OPTION_TOPOLOGY_FILE;

    /**
     * Machine name option
     */
    public static final String MACHINE_NAME_OPTION = DASH + ICommandLine.COMMAND_OPTION_MACHINE_NAME;

    /**
     * Domain name option
     */
    public static final String DOMAIN_NAME_OPTION = DASH + ICommandLine.COMMAND_OPTION_DOMAIN_NAME;

    /**
     * Server name option
     */
    public static final String SERVER_NAME_OPTION = DASH + ICommandLine.COMMAND_OPTION_SERVER_NAME;

    /**
     * Cluster name option
     */
    public static final String CLUSTER_NAME_OPTION = DASH + ICommandLine.COMMAND_OPTION_CLUSTER_NAME;

    /**
     * Help option
     */
    public static final String HELP_OPTION = DASH + "help";

    /**
     * {@link IDeployMeService}
     */
    @Requires(optional = false)
    private IDeployMeService deployme;

    /**
     * The path to the topology file
     */
    @Option(name = TOPOLOGY_FILE_OPTION, required = true)
    private String topologyFilePath;

    /**
     * Machine name
     */
    @Option(name = MACHINE_NAME_OPTION, required = false)
    private String machineName;

    /**
     * Domain name
     */
    @Option(name = DOMAIN_NAME_OPTION, required = false)
    private String domainName;

    /**
     * Server name
     */
    @Option(name = SERVER_NAME_OPTION, required = false)
    private String serverName;

    /**
     * Cluster name
     */
    @Option(name = CLUSTER_NAME_OPTION, required = false)
    private String clusterName;

    /**
     * Help option
     */
    @Option(name = HELP_OPTION, required = false)
    private String help;

    /**
     * Other arguments
     */
    @Argument(multiValued=true)
    private List<String> arguments;

    /**
     * Language
     */
    @Option(name="-", aliases="--lang", required=false)
    private String lang = "en";

    /**
     * The logger
     */
    private static Log logger = LogFactory.getLog(DeployMeCommand.class);

    /**
     * {@inheritDoc}
     */
    public Object execute(final CommandSession commandSession) throws Exception {

        //TODO check if option without value is null or not
        if (this.help() != null) {
           // return help();
        }

        /**
         * Configure the {@link ICommandLine}
         */
        ICommandLine commandLine = new CommandLine();
        commandLine.addValue(ICommandLine.COMMAND_OPTION_TOPOLOGY_FILE, this.topologyFilePath);
        commandLine.addValue(ICommandLine.COMMAND_OPTION_MACHINE_NAME, this.machineName);
        commandLine.addValue(ICommandLine.COMMAND_OPTION_DOMAIN_NAME, this.domainName);
        commandLine.addValue(ICommandLine.COMMAND_OPTION_SERVER_NAME, this.serverName);
        commandLine.addValue(ICommandLine.COMMAND_OPTION_CLUSTER_NAME, this.clusterName);

        //check command options
        commandLine.checkCommandLine();

        if (commandLine.getVerbose()) {
            Logger.getLogger("").setLevel(Level.ALL);
        }

        //get the URL of the topology
        URL urlXML = null;
        File topologyFile = new File(commandLine.getTopologyFile());
        if (topologyFile != null && topologyFile.exists()) {
            urlXML = topologyFile.toURI().toURL();
        }
        if (urlXML == null) {
            logger.info("The " + urlXML + " is not a local file or resource.");
            logger.info("JASMINe Deploy ME will download it using Java URLConnections.");
            urlXML = new URL(commandLine.getTopologyFile());
        }
        if (urlXML == null) {
            throw new IllegalStateException("Cannot find a valid URL for the topology '" + this.topologyFilePath + "'");
        }

        //deploy a topology configuration
        this.deployme.deploy(urlXML, commandLine.getMachineName(), commandLine.getDomainName(),
                commandLine.getServerName(), commandLine.getClusterName());


        //print nothing in the shell
        return null;
    }

    /**
     * @return a help message
     */
    private String help() {
        StringBuffer buffer = new StringBuffer();

        buffer.append("\n");
        buffer.append("Usage : deployme:deploy -topologyFile=topology_xml (-machineName=host | -domainName=domain [-serverName=server | -clusterName=cluster]) [-verbose]");
        buffer.append("\n\n");
        buffer.append(" -topologyFile=topology_xml : ");
        buffer.append("\n\t");
        buffer.append("This file describes your server infrastructure. You have two ways to create it : ");
        buffer.append("\n\t");
        buffer.append("1. Export a topology file from JASMINe Design. This is the recommended method.");
        buffer.append("\n\t");
        buffer.append("2. Manually create your topology file. Please refer to the JASMINe Deploy ME XSD file and/or the JASMINe Deploy ME samples distribution for details.");
        buffer.append("\n\n");
        buffer.append("-machineName=host : name of a machine in the topology.xml file");
        buffer.append("\n\t");
        buffer.append("Deploy all servers on a given machine");
        buffer.append("\n\n");
        buffer.append("-domainName=domain : A domain name of the topology.xml file.");
        buffer.append("\n");
        buffer.append("-serverName=server : A server name of the topology.xml file.");
        buffer.append("\n\t");
        buffer.append("Deploy the server of the the specified domain.");
        buffer.append("\n");
        buffer.append("-clusterName=cluster : A cluster daemon name of the topology.xml file.");
        buffer.append("\n\t");
        buffer.append("Deploy the cluster daemon of the specified domain.");

        return buffer.toString();
    }

    /**
     * @param topologyFilePath The topology file path to set
     */
    public void setTopologyFilePath(final String topologyFilePath) {
        this.topologyFilePath = topologyFilePath;
    }

    /**
     * @param machineName The machine name to set
     */
    public void setMachineName(final String machineName) {
        this.machineName = machineName;
    }

    /**
     * @param domainName The domain name to set
     */
    public void setDomainName(final String domainName) {
        this.domainName = domainName;
    }

    /**
     * @param deploymeService The {@link IDeployMeService} to set
     */
    public void setDeploymeManager(final IDeployMeService deploymeService) {
        this.deployme = deploymeService;
    }

    /**
     * @param serverName The server name to set
     */
    public void setServerName(final String serverName) {
        this.serverName = serverName;
    }

    /**
     * @param clusterName The clustername to set
     */
    public void setClusterName(final String clusterName) {
        this.clusterName = clusterName;
    }
}
