/**
 * JASMINe Deploy ME [Managed Element]
 * Copyright (C) 2008-2012 Bull S.A.S.
 * Copyright (C) 2008-2012 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DeployMeService.java 10003 2012-05-02 15:51:24Z cazauxj $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.deployme;

import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.xml.bind.JAXBException;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.ow2.jasmine.deployme.api.IDeployMeService;
import org.ow2.jasmine.deployme.api.IServer;
import org.ow2.jasmine.deployme.api.extensions.IDeploymeExtension;
import org.ow2.jasmine.deployme.api.modules.IDeploymeModule;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import org.ow2.jasmine.deployme.api.DeployMEPropertiesManager;
import org.ow2.jasmine.deployme.api.DeploymeVersion;
import org.ow2.jasmine.deployme.api.IDeployme;
import org.ow2.jasmine.deployme.api.XmlLoader;
import org.ow2.jasmine.deployme.v1.DeploymeV1;
import org.ow2.jasmine.deployme.v1.generated.Topology;
import org.ow2.jasmine.deployme.v2.DeploymeV2;
import org.ow2.jasmine.deployme.v2.generated.TopologyType;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Manage deployment of  topology
 * 
 * @author Remy Bresson
 * @author Jeremy Cazaux (Conversion into an OSGi service)
 */
public class DeployMeService implements IDeployMeService {

    /**
     * The loaded topology
     */
    protected Object topology;

    /**
     * Namespaces of the XML document
     */
    protected Map<String, Node> namespaces;

    /**
     * The logger
     */
    protected static Log logger = LogFactory.getLog(DeployMeService.class);

    /**
     * XMLNS pattern
     */
    public static final Pattern XMLNS_PATTERN = Pattern.compile("xmlns.*");       

    /**
     * Deployme version which can manage the topology file
     */
    private DeploymeVersion deploymeVersion;

    /**
     * {@link IDeployme}
     */
    private IDeployme deployme;

    /**
     * List of available {@link IDeploymeExtension}
     */
    private List<IDeploymeExtension> availableExtensions;

    /**
     * List of available {@link IDeploymeModule}
     */
    private List<IDeploymeModule> availableModules;

    /**
     * Getter for topology.
     * 
     * @return the topology
     */
    //TODO used?
    @Deprecated
    public Object getTopology() {
        return topology;
    }

    /**
     * Default constructor
     */
    public DeployMeService() {
        this.namespaces = new LinkedHashMap<String, Node> ();
        this.availableExtensions = new ArrayList<IDeploymeExtension>();
        this.availableModules = new ArrayList<IDeploymeModule>();
    }

    /**
     * {@inheritDoc}
     */
    public void deploy(final URL topologyUrl, final String machine, final String domain, final String server,
                       final String cluster) throws SAXException, JAXBException, IOException {
        loadTopology(topologyUrl);
        init(machine, domain, server, cluster);
    }

    /**
     * {@inheritDoc}
     */
    public void deploy(final String topologyXml, final String machine, final String domain, final String server,
                       final String cluster) throws IOException, SAXException, JAXBException{
        loadTopology(topologyXml);
        init(machine, domain, server, cluster);
    }

    /**
     * Apply the config of the server (machine or domain/server/cluster)
     * @param machine The machine to use. if set domain or server or cluster
     *        name must not be set
     * @param domain The domain where found the server to use. If set, server or
     *        cluster has to be set, and machine must no be set
     * @param server The server to use. If set, domain has to be set and cluster
     *        or machine must not be set
     * @param cluster The cluster daemon to use. If set, domain has to be set
     *        and server or machine must not be set
     */
    private void init(final String machine, final String domain, final String server, final String cluster) {
        initDeployme();
        this.deployme.applyConfiguration(machine, domain, server, cluster);
    }

    /**
     * Initialize {@link IDeployme} component
     */
    public void initDeployme() {
        if (DeploymeVersion.DEPLOYME_2.equals(deploymeVersion)) {
            this.deployme = new DeploymeV2((TopologyType) this.topology, this.availableExtensions, this.availableModules);
        } else {
            this.deployme = new DeploymeV1((Topology) this.topology);
        }
    }

    /**
     * @return {@link IDeployme}
     */
    public IDeployme getDeployme() {
        return this.deployme;
    }

    /**
     * Load the topology from the url urlTopology using the schema from
     * schemaUrl. Store the topology on DeployMeService.topology variable.
     * 
     * @param urlTopology Url of the topology on xml format
     * @throws JAXBException JAXB exception.
     * @throws SAXException SAX exception.
     * @throws IOException If urlTopology cannot be retrieved.
     */
    public void loadTopology(final URL urlTopology) throws SAXException, JAXBException, IOException {
        initDeploymeVersion(urlTopology);
        XmlLoader xmlLoader = new XmlLoader(urlTopology, this.deploymeVersion);
        this.topology = xmlLoader.getTopology();
    }

    /**
     * Load the topology from the xml string topology using the schema from
     * schemaUrl. Store the topology on DeployMeService.topology variable.
     *
     * @param topology Url of the topology on xml format
     * @throws JAXBException JAXB exception.
     * @throws SAXException SAX exception.
     * @throws IOException If urlTopology cannot be retrieved.
     */
    public void loadTopology(final String topology) throws SAXException, JAXBException, IOException {
        initDeploymeVersion(topology);
        XmlLoader xmlLoader = new XmlLoader(topology, this.deploymeVersion);
        this.topology = xmlLoader.getTopology();
    }

    /**
     * Initialize the deployme version which is matching the given xml file
     * @param topologyURL The topology URL
     */
    private void initDeploymeVersion(final URL topologyURL) {
        try {
            initDeploymeVersion(new File(topologyURL.toURI()));
        } catch (URISyntaxException e) {
            logger.error("Cannot get the URI of the URL " + topologyURL.getFile(), e);
        }
    }

    /**
     * Initialize the deployme version which is matching the given xml file
     * @param topology The content of the topology
     */
    private void initDeploymeVersion(final String topology) {
        DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
        DocumentBuilder documentBuilder = null;
        Document document = null;
        try {
            documentBuilder = documentBuilderFactory.newDocumentBuilder();
        } catch (ParserConfigurationException e) {
            logger.error("Cannot get the instance of the DocumentBuilder", e);
        }
        if (documentBuilder != null) {
            try {
                document = documentBuilder.parse(new InputSource(new StringReader(topology.trim())));
            } catch (SAXException e) {
                logger.error("Cannot parse XML content of " + topology, e);
            } catch (IOException e) {
                logger.error("Cannot parse XML content of " + topology, e);
            }
        }
        initDeploymeVersion(document);
    }

    /**
     * Initialize the deployme version which is matching the given xml file
     * @param topologyFile The topology file
     */
    private void initDeploymeVersion(final File topologyFile) {
        DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
        DocumentBuilder documentBuilder = null;
        Document document = null;
        try {
            documentBuilder = documentBuilderFactory.newDocumentBuilder();
        } catch (ParserConfigurationException e) {
            logger.error("Cannot get the instance of the DocumentBuilder", e);
        }
        if (documentBuilder != null) {
            try {
                document = documentBuilder.parse(topologyFile);
            } catch (SAXException e) {
                logger.error("Cannot parse XML file " + topologyFile.getAbsolutePath(), e);
            } catch (IOException e) {
                logger.error("Cannot parse XML file " + topologyFile.getAbsolutePath(), e);
            }
        }
        initDeploymeVersion(document);
    }

    /**
     * Initialize the deployme version which is matching the given xml file
     * @param document The {@link Document} associated to the topology
     */
    private void initDeploymeVersion(final Document document) {
        if (document != null) {
            getNamespace(document);
        }
        this.deploymeVersion = DeployMEPropertiesManager.getDeploymeVersion(new ArrayList<String>(this.namespaces.keySet()));
    }

    /**
     * Get all namespaces of the document
     * @param node The node to check
     */
    private void getNamespace(final Node node) {
        NodeList nodeList = node.getChildNodes();
        for (int i = 0; i < nodeList.getLength(); i++) {
            Node child = nodeList.item(i);
            if (child.getNodeType() == Node.ELEMENT_NODE) {
                Element element = (Element) child;
                NamedNodeMap namedNodeMap = element.getAttributes();
                for (int y = 0; y < namedNodeMap.getLength(); y++) {
                    Attr attribute = Attr.class.cast(namedNodeMap.item(y));
                    if (XMLNS_PATTERN.matcher(attribute.getName()).matches()) {
                        String namespace = attribute.getValue();
                        if (namespace != null && !namespace.isEmpty()) {
                            this.namespaces.put(namespace, child);
                        }
                        getNamespace(child);
                    }
                }
            }
        }
    }

    /**
     * @param deploymeExtension The {@link IDeploymeExtension} to add
     */
    public void bindDeploymeExtension(final IDeploymeExtension deploymeExtension) {
        this.availableExtensions.add(deploymeExtension);
    }

    /**
     * @param deploymeExtension The {@link IDeploymeExtension} to remove
     */
    public void unbindDeploymeExtension(final IDeploymeExtension deploymeExtension) {
        this.availableExtensions.remove(deploymeExtension);
    }

    /**
     * @param deploymeModule The {@link IDeploymeModule} to add
     */
    public void bindDeploymeModule(final IDeploymeModule deploymeModule) {
        this.availableModules.add(deploymeModule);
    }

    /**
     * @param deploymeModule The {@link IDeploymeModule} to remove
     */
    public void unbindDeploymeModule(final IDeploymeModule deploymeModule) {
        this.availableModules.remove(deploymeModule);
    }

    /**
     * @return the deployme instance
     */
    public IDeployme getInstance() {
        return getDeployme();
    }

    /**
     * {@inheritDoc}
     */
    public IServer getServer(final String domainName, final String serverName) {
        if (this.deployme != null) {
            if (this.deployme instanceof DeploymeV2) {
                DeploymeV2 deploymeV2 = (DeploymeV2) this.deployme;
                return deploymeV2.getServer(domainName, serverName);
            }
        }
        return null;
    }
}
