/**
 * JASMINe Deploy ME [Managed Element]
 * Copyright (C) 2012 Bull S.A.S.
 * Copyright (C) 2012 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: AbstractDeploymeExtension.java 9990 2012-04-25 09:15:49Z cazauxj $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.deployme.extension;

import org.ow2.jasmine.deployme.api.DeployMEPropertiesManager;
import org.ow2.jasmine.deployme.api.DeploymeVersion;
import org.ow2.jasmine.deployme.api.extensions.DeploymeExtensionException;
import org.ow2.jasmine.deployme.api.extensions.IDeploymeExtension;
import org.ow2.jasmine.deployme.v2.generated.TopologyType;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

import javax.xml.XMLConstants;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

/**
 * Abstract deployme extension
 * @author Jeremy Cazaux
 */
public abstract class AbstractDeploymeExtension implements IDeploymeExtension {

    /**
     * EOL
     */
    public final static String EOL = "\n";

    /**
     * XSD URL
     */
    protected List<URL> xsdUrls;

    /**
     * Root class
     */
    protected Class rootClass = TopologyType.class;

    /**
     * Default constructor
     */
    protected AbstractDeploymeExtension() {
        this.xsdUrls = new ArrayList<URL>();
    }

    /*
    * @param resources List of resources
    * @return the list of Source associate to the given resources list
    */
    protected List<Source> getXsdSources(final List<String> resources) throws DeploymeExtensionException {
        List<URL> urls = getXsdURL(resources);
        List<Source> sources = new ArrayList<Source>();

        for (URL url: urls) {
            sources.add(getSource(url));
        }
        return sources;
    }

    /**
    * @param resources List of resources
    * @return the list of Source associate to the given resources list
    */
    protected List<URL> getXsdURL(final List<String> resources) throws DeploymeExtensionException {
        List<URL> urls = new ArrayList<URL>();
        for (String resource: resources) { 
            URL url = getXsdURL(resource);
            if (url != null) {
                urls.add(url);
            }
        } 
        return urls;
    }

    /** 
     * @param resource Resource
     * @return the URL of the given resource
     * @throws org.ow2.jasmine.deployme.api.extensions.DeploymeExtensionException
     */
    protected URL getXsdURL(final String resource) throws DeploymeExtensionException {
        return getXsdURL(resource, getClass());
    }

    /**
     * @param resource The resource to retrieve
     * @param clazz A {@link Class} associated to the given resource (they should share the same classloader)
     * @return the URL of the given resource
     * @throws DeploymeExtensionException
     */
    protected URL getXsdURL(final String resource, final Class clazz) throws DeploymeExtensionException {
        URL url = clazz.getClassLoader().getResource(resource);
        if (url == null) {
            url = clazz.getResource(resource);
            if (url == null) {
                throw new DeploymeExtensionException("Cannot get the URL of the resource " + resource
                        + EOL);
            }
        }
        return url;
    }

    /**
     *
     * @param url The {@link java.net.URL}
     * @return the {@link javax.xml.transform.Source} associated to the given URL
     * @throws org.ow2.jasmine.deployme.api.extensions.DeploymeExtensionException
     */
    protected Source getSource(final URL url) throws DeploymeExtensionException {
        if (url != null) {
            InputStream inputStream = null;
            try {
                inputStream = url.openStream();
            } catch (IOException e) {
                throw new DeploymeExtensionException("Cannot get the inpustream of the URL " + url.getPath() + EOL);
            }

            try {
                return new StreamSource(inputStream);
            } catch (Exception e) {
                throw new DeploymeExtensionException("Cannot create a new StreamSource for the URL : " + url.getPath()
                        + EOL, e);
            }
        } else {
            return null;
        }

    }

    /**
     * @param urls List of URL
     * @return the list of {@link javax.xml.transform.Source} associated to the list of url
     */
    protected List<Source> getSources(final List<URL> urls) throws DeploymeExtensionException {
        List<Source> sources = new ArrayList<Source>();
        for (URL url: urls) {
            Source source = getSource(url);
            if (source != null) {
                sources.add(source);
            }
        }
        return sources;
    }

    /**
     * @param node The node to unmarshall
     * @param rootClass The root class of the node
     * @param xsdSources XSD sources
     * @param classLoader The classloader of the extension
     * @return a JAXB object associated to the given node
     * @throws {@link org.ow2.jasmine.deployme.api.extensions.DeploymeExtensionException} if unmarshall operation has failed
     */
    protected Object unmarshall(final Node node, Class<?> rootClass, final List<Source> xsdSources,
                                final ClassLoader classLoader)
            throws DeploymeExtensionException {
        JAXBContext jaxbContext  = null;
        try {
            jaxbContext = JAXBContext.newInstance(rootClass.getPackage().getName(), classLoader);
        } catch (JAXBException e) {
            throw new DeploymeExtensionException("Cannot create a new instance of JAXBContext object with rootClass "
                    + rootClass.getPackage().getName(), e);
        }
        SchemaFactory schemaFactory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
        Schema schema = null;
        try {
            schema = schemaFactory.newSchema(xsdSources.toArray(new Source[xsdSources.size()]));
        } catch (SAXException e) {
            throw new DeploymeExtensionException("Cannot create a new instance of Schema object with the given XSD " +
                    "sources", e);
        }
        Unmarshaller unmarshaller = null;
        try {
            unmarshaller = jaxbContext.createUnmarshaller();
        } catch (JAXBException e) {
            throw new DeploymeExtensionException("Cannot instanciate an Unmarshaller object from the JAXBContext "
                    + jaxbContext, e);
        }
        unmarshaller.setSchema(schema);
        JAXBElement root = null;
        try {
            root = (JAXBElement) unmarshaller.unmarshal(node);
        } catch (JAXBException e) {
            throw new DeploymeExtensionException("Cannot unmarshall node " + node.getNodeName() + " of the namespace "
                    + node.getNamespaceURI(), e);
        }
        return root.getValue();
    }

    /**
     * {@inheritDoc}
     */
    public String getNamespace() {
        return DeployMEPropertiesManager.getTopologyXMLNS(DeploymeVersion.DEPLOYME_2);
    }

    /**
     *
     * @param localNode The node to unmarshall
     * @param rootClass The root class of a node
     * @param classLoader The classloader of the extension
     * @return a JAXB object associated to the given node
     * @throws DeploymeExtensionException
     */
    public Object convert(final Node localNode, final Class rootClass, final ClassLoader classLoader)
            throws DeploymeExtensionException {
        URL parentXSD = TopologyType.class.getClassLoader().getResource(
                DeployMEPropertiesManager.getXsdTopologyPath(DeploymeVersion.DEPLOYME_2));
        if (parentXSD == null) {
            throw new DeploymeExtensionException("Cannot get the URL of the topology XSD "  +
                    DeployMEPropertiesManager.getXsdTopologyPath(DeploymeVersion.DEPLOYME_2));
        }
        this.xsdUrls.add(parentXSD);

        //get XSD sources
        List<Source> sources = getSources(this.xsdUrls);

        //get tomcat7 local configuration
        return unmarshall(localNode, rootClass, sources, classLoader);
    }
}
