/**
 * JASMINe Deploy ME [Managed Element]
 * Copyright (C) 2008-2012 Bull S.A.S.
 * Copyright (C) 2008-2012 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DeploymeV1.java 9900 2012-02-22 15:21:10Z cazauxj $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.deployme.v1;

import org.ow2.jasmine.deployme.api.IDeployme;
import org.ow2.jasmine.deployme.v1.configApply.Configurator;
import org.ow2.jasmine.deployme.v1.util.ServerDomain;

import java.util.ArrayList;
import java.util.List;
import org.ow2.jasmine.deployme.v1.generated.Topology;
import org.ow2.jasmine.deployme.v1.generated.Topology.Domains.Domain.ClusterDaemons.ClusterDaemon;
import org.ow2.jasmine.deployme.v1.generated.Configuration;
import org.ow2.jasmine.deployme.v1.generated.Topology.Domains.Domain;
import org.ow2.jasmine.deployme.v1.generated.Topology.Domains.Domain.Servers.Server;

/**
 * Apply the config of the servers(machine or domain/server/cluster)
 * @author Remy Bresson
 */
public class DeploymeV1 implements IDeployme {

    /**
     * The loaded from XML topology
     */
    private Topology topology;

    /**
     * List of the servers witch deployMe will work
     */
    protected List<ServerDomain> serversDomains;    
    
    /**
     * Default constructor   *
     */
    public DeploymeV1(Topology topology) {
        this.topology = topology;
    }

    /**
     * Getter for topology.
     *
     * @return the topology
     */
    public Topology getTopology() {
        return this.topology;
    }

    /**
     * {@inheritDoc}
     */
    public void applyConfiguration(final String machine, final String domain, final String server, final String cluster) {

        boolean clusterMode = cluster != null;
        ClusterDaemon clusterDaemon = null;

        if (clusterMode) {
            clusterDaemon = this.getClusterDaemonConf(domain, cluster);
            if (clusterDaemon == null) {
                throw new IllegalArgumentException("Cannot find the specified cluster daemon '" + cluster
                        + "' on specified domain '" + domain + "'");
            }
        }

        this.serversDomains = this.buildServerDomainList(machine, domain, server, cluster);

        if (clusterMode) {
            /* cluster mode */
            /* Merge configuration with domain */
            clusterDaemon = this.mergeClusterDaemonAndDomainConf(clusterDaemon, this.serversDomains.get(0).getDomain());

            /* Configure cluster */
            Configurator.setClusterAConfiguration(this.serversDomains.get(0).getDomain().getName(), clusterDaemon);
        }

        int i = 1;
        for (ServerDomain sd : this.serversDomains) {
            Configuration confToApply = new ConfiguratorBuilder(sd).getConfig();

            if (clusterMode) {
                /* Create the requested JONAS_BASE */
                confToApply.getGlobalOptions().setJonasBase(clusterDaemon.getJonasBasePrefix() + i++);
            }

            Configurator.setAConfiguration(confToApply, sd);
        }
    }

    public ClusterDaemon getClusterDaemonConf(final String domain, final String clusterName) {
        ClusterDaemon workingCluster = null;

        for (Domain d : this.topology.getDomains().getDomain()) {
            if (d.getName().equals(domain)) {
                for (ClusterDaemon cd : d.getClusterDaemons().getClusterDaemon()) {
                    if (cd.getName().equals(clusterName)) {
                        workingCluster = cd;
                    }
                }
            }
        }

        return workingCluster;
    }

    /**
     * Build the server list that will be used.
     *
     * @param machine host name
     * @param domain domain name
     * @param server server name
     * @param clusterName cluster daemon name
     * @return A list with the server to configure
     */
    public List<ServerDomain> buildServerDomainList(final String machine, final String domain, final String server,
                                                    final String clusterName) {
        List<ServerDomain> ret;

        if (machine != null) {
            ret = this.buildServerListWithHostname(machine);
            if (ret == null || ret.size() < 1) {
                throw new IllegalArgumentException("The machine named '" + machine + "' does not appear in the topology file!");
            }
        } else if (clusterName == null) {
            ret = this.buildServerListWithServer(domain, server);
            if (ret == null || ret.size() < 1) {
                throw new IllegalArgumentException("The server named '" + domain + "." + server
                        + "' does not appear in the topology file!");
            }
        } else {
            ret = this.buildServerListWithCluster(domain, clusterName);
            if (ret == null || ret.size() < 1) {
                throw new IllegalArgumentException("The cluster daemon named '" + domain + "." + clusterName
                        + "' does not appear in the topology file!");
            }
        }

        return ret;
    }

    /**
     * Build the servers list (in this case, only 1 server in the list) defined
     * by domain.server
     *
     * @param domain the domain
     * @param cluster the cluster
     * @return A list with the server to configure
     */
    protected List<ServerDomain> buildServerListWithCluster(final String domain, final String cluster) {
        List<ServerDomain> ret = new ArrayList<ServerDomain>();

        ClusterDaemon workingCluster = null;
        Domain workingDomain = null;
        List<Server> servers = new ArrayList<Server>();

        for (Domain d : this.topology.getDomains().getDomain()) {
            if (d.getName().equals(domain)) {
                workingDomain = d;
                for (ClusterDaemon cd : d.getClusterDaemons().getClusterDaemon()) {
                    if (cd.getName().equals(cluster)) {
                        workingCluster = cd;
                        for (String serverName : cd.getServers().getServerName()) {
                            boolean found = false;
                            for (Server s : workingDomain.getServers().getServer()) {
                                if (s.getName().equals(serverName)) {
                                    if (servers.contains(s)) {
                                        throw new IllegalArgumentException("Server '" + serverName
                                                + "' appears twice in the cluster daemon '" + cluster + "' server list!");
                                    }
                                    servers.add(s);
                                    found = true;
                                    break;
                                }
                            }
                            if (!found) {
                                throw new IllegalArgumentException("Server '" + serverName + "' in the cluster daemon '"
                                        + cluster + "' server list does not appear in the topology!");
                            }
                        }
                    }
                }
            }
            if (workingCluster != null) {
                break;
            }
        }

        for (Server s : servers) {
            ret.add(new ServerDomain(s, workingDomain));
        }

        return ret;
    }

    /**
     * Build the servers list (in this case, only 1 server in the list) defined
     * by domain.server
     *
     * @param domain the domain
     * @param server the server
     * @return A list with the server to configure
     */
    private List<ServerDomain> buildServerListWithServer(final String domain, final String server) {
        List<ServerDomain> ret = new ArrayList<ServerDomain>();

        for (Domain d : this.topology.getDomains().getDomain()) {
            if (d.getName().equals(domain)) {
                for (Server s : d.getServers().getServer()) {
                    if (s.getName().equals(server)) {
                        ret.add(new ServerDomain(s, d));
                        break;
                    }
                }
                if (!ret.isEmpty()) {
                    break;
                }
            }
        }

        return ret;
    }

    /**
     * Build the servers list hosted by the machine machine
     *
     * @param machine The machine where are located the servers to configure
     * @return A list with the servers to configure
     */
    private List<ServerDomain> buildServerListWithHostname(final String machine) {
        List<ServerDomain> ret = new ArrayList<ServerDomain>();

        for (Domain d : this.topology.getDomains().getDomain()) {
            for (Server s : d.getServers().getServer()) {
                if (s.getMachine().equals(machine)) {
                    ret.add(new ServerDomain(s, d));
                }
            }
        }

        return ret;
    }

    /**
     * Merge the configuration of the domain with the configuration of the
     * cluster daemon
     *
     * @param cd cluster daemon configuration
     * @param domain the domain
     * @return the merged configuration
     */
    public ClusterDaemon mergeClusterDaemonAndDomainConf(final ClusterDaemon cd, final Domain domain) {
        if (cd.getJonasRoot() == null) {
            cd.setJonasRoot(domain.getConfiguration().getGlobalOptions().getJonasRoot());
        }
        if (cd.getJavaHome() == null) {
            cd.setJavaHome(domain.getConfiguration().getGlobalOptions().getJavaHome());
        }

        return cd;
    }
}
