/**
 * JASMINe Deploy ME [Managed Element]
 * Copyright (C) 2012 Bull S.A.S.
 * Copyright (C) 2012 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DeploymeV2.java 10005 2012-05-02 21:53:08Z cazauxj $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.deployme.v2;

import org.ow2.jasmine.deployme.api.IDeployme;
import org.ow2.jasmine.deployme.api.extensions.IDeploymeExtension;
import org.ow2.jasmine.deployme.api.modules.IDeploymeModule;
import org.ow2.jasmine.deployme.v2.configApply.Configurator;
import org.ow2.jasmine.deployme.v2.generated.TopologyType;
import org.ow2.jasmine.deployme.v2.generated.ServerType;
import org.ow2.jasmine.deployme.v2.generated.DomainType;
import org.ow2.jasmine.deployme.v2.generated.DomainsType;
import org.ow2.jasmine.deployme.v2.util.Agent;
import org.ow2.jasmine.deployme.v2.util.Server;
import org.ow2.jasmine.deployme.v2.generated.AgentType;
import org.ow2.jasmine.deployme.v2.generated.AgentsType;
import org.ow2.jasmine.deployme.v2.generated.ServerNameType;

import java.util.ArrayList;
import java.util.List;

/**
 * Manage Deployme 2.0 XML resources
 * @author Jeremy Cazaux
 */
public class DeploymeV2 implements IDeployme {

    /**
     * The loaded topology
     */
    private TopologyType topology;

    /**
     * List of available {@link IDeploymeExtension}
     */
    private List<IDeploymeExtension> availableExtensions;

    /**
     * List of available {@link org.ow2.jasmine.deployme.api.modules.IDeploymeModule}
     */
    private List<IDeploymeModule> availableModules;

    /**
     * Default constructor
     * @param topology The {@link TopologyType} to set
     * @param deploymeExtensions List of extensions to set
     */
    public DeploymeV2(final TopologyType topology, final List<IDeploymeExtension> deploymeExtensions,
                      final List<IDeploymeModule> deploymeModules) {
        this.topology = topology;
        this.availableExtensions = deploymeExtensions;
        this.availableModules = deploymeModules;
    }


    /**
     * List of {@link Server}
     */
    private List<Server> servers;

    /**
     * The agent
     */
    private Agent agent;

    /**
     * {@inheritDoc}
     */
    public void applyConfiguration(final String machine, final String domainName, final String serverName, 
                                   final String clusterName) {
        this.servers = buildServerList(machine, domainName, serverName, clusterName);
        Boolean clusterMode = (clusterName != null);
        if (clusterMode) {
            this.agent = getAgent(domainName, clusterName);
            Configurator.setAConfiguration(this.agent);
        }
        int i = 1;
        for (Server server: this.servers) {
            if (clusterMode) {
                server.setJonasBase(this.agent.getJonasBase() + i++);
            }
            Configurator.setAConfiguration(server);
        }
    }

    /**
     * Build the server list that will be used.
     *
     * @param machine host name
     * @param domainName domain name
     * @param serverName server name
     * @param clusterName cluster daemon name
     * @return A list with the server to configure
     */
    public List<Server> buildServerList(final String machine, final String domainName, final String serverName, 
                                              final String clusterName) {
        List<Server> servers;
        if (machine != null) {
            servers = this.buildServerListWithHostname(machine);
            if (servers == null || servers.size() < 1) {
                throw new IllegalArgumentException("The machine named '" + machine + "' does not appear in the topology file!");
            }
        } else if (clusterName == null) {
            servers = this.buildServerListWithServer(domainName, serverName);
            if (servers == null || servers.size() < 1) {
                throw new IllegalArgumentException("The server named '" + domainName + "." + serverName
                        + "' does not appear in the topology file!");
            }
        } else {
            servers = this.buildServerListManagedByAgent(domainName, clusterName);
            if (servers == null || servers.size() < 1) {
                throw new IllegalArgumentException("The cluster daemon named '" + domainName + "." + clusterName
                        + "' does not appear in the topology file!");
            }
        }

        return servers;
    }

    /**
     * Build the servers list hosted by the machine machine
     *
     * @param machine The machine where are located the servers to configure
     * @return A list with the servers to configure
     */
    protected List<Server> buildServerListWithHostname(final String machine) {
        List<Server> servers = new ArrayList<Server>();
        DomainsType domains = this.topology.getDomains();
        for (DomainType domain: domains.getDomain()) {
            for (ServerType server: domain.getServers().getServer()) {
                if (server.getMachine().equals(machine)) {
                    servers.add(new Server(server, domain, domains, true, this.availableExtensions, this.availableModules));
                }
            }
        }
        return servers;  
    }

    /**
     * Build the servers list (in this case, only 1 server in the list) defined
     * by domain.server
     *
     * @param domainName the name of the target domain
     * @param serverName the name of the target server
     * @return A list with the server to configure
     */
    protected List<Server> buildServerListWithServer(final String domainName, final String serverName) {
        List<Server> servers = new ArrayList<Server>();
        DomainsType domains = this.topology.getDomains();
        for (DomainType domain: domains.getDomain()) {
            if (domain.getName().equals(domainName)) {
                for (ServerType server: domain.getServers().getServer()) {
                    if (server.getName().equals(serverName)) {
                        servers.add(new Server(server, domain, domains, true, this.availableExtensions, this.availableModules));
                        break;
                    }
                }
            }
        }
        return servers;
    }


    /**
     * Build the agent
     * @param domainName the name of the domain
     * @param clusterName the name of the agent
     * @return the associated {@link Agent}
     */
    protected Agent getAgent(final String domainName, final String clusterName) {
        DomainsType domains = this.topology.getDomains();
        for (DomainType domain: domains.getDomain()) {
            if (domain.getName().equals(domainName)) {
                for (AgentType agent: domain.getAgents().getAgent()) {
                    if (agent.getName().equals(clusterName)) {
                        return new Agent(agent, domain, domains, true, this.availableExtensions, this.availableModules);
                    }
                }
            }
        }
        return null;
    }

    /**
     * Build the servers list (in this case, only 1 server in the list) defined
     * by domain.server
     *
     * @param domainName the domain
     * @param clusterName the cluster
     * @return A list with the server to configure
     */
    protected List<Server> buildServerListManagedByAgent(final String domainName, final String clusterName) {
        List<Server> servers = new ArrayList<Server>();
        DomainsType domains = this.topology.getDomains();
        for (DomainType domain: domains.getDomain()) {
            if (domain.getName().equals(domainName)) {
                AgentsType agents = domain.getAgents();
                if (agents != null) {
                    for (AgentType agent: agents.getAgent()) {
                        if (agent.getName().equals(clusterName)) {
                            for (ServerNameType serverName: agent.getManagedServers().getServerName()) {
                                boolean found = false;
                                for (ServerType server: domain.getServers().getServer()) {
                                    if (server.getName().equals(serverName.getName())) {
                                        if (servers.contains(server)) {
                                            throw new IllegalArgumentException("Server '" + serverName.getName()
                                                    + "' appears twice in the cluster daemon '" + clusterName + "' server list!");
                                        }
                                        servers.add(new Server(server, domain, domains, true, this.availableExtensions,
                                                this.availableModules));
                                        found = true;
                                        break;
                                    }
                                }
                                if (!found) {
                                    throw new IllegalArgumentException("Server '" + serverName.getName() + "' in the cluster daemon '"
                                            + clusterName + "' server list does not appear in the topology!");
                                }
                            }
                        }
                    }
                }
            }
        }

        return servers;
    }
    
    /**
     * @param domainName the domain name
     * @param serverName the server name
     * @return the {@link Server}
     */
    public Server getServer(final String domainName, final String serverName) {
        for (Server server: this.servers) {
           if (server.getServerName().equals(serverName) && server.getDomainName().equals(domainName)) {
               return server;
           }
        }
        return null;
    }
}
