/**
 * JASMINe Deploy ME [Managed Element]
 * Copyright (C) 2012 Bull S.A.S.
 * Copyright (C) 2012 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Configurator.java 10000 2012-05-02 14:52:40Z cazauxj $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.deployme.v2.configApply;

import org.ow2.jasmine.deployme.api.extensions.IDeploymeExtension;
import org.ow2.jasmine.deployme.api.modules.IDeploymeModule;
import org.ow2.jasmine.deployme.v2.util.Agent;
import org.ow2.jasmine.deployme.v2.util.Server;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.util.List;
import java.util.Map;

/**
 * Configure a JOnAS server
 * @author Jeremy Cazaux
 */
public class Configurator {

    /**
     * The logger
     */
    protected static Log logger = LogFactory.getLog(Configurator.class);

    /**
     * Set the configuration of the given server
     * @param server The {@link Server}
     */
    public static void setAConfiguration(final Server server) {
        final String domainName = server.getDomainName();
        final String serverName = server.getServerName();
        final String jonasRoot = server.getJonasRoot();
        final String jonasBase = server.getJonasBase();

        if (jonasRoot == null) {
            throw new IllegalArgumentException("The topology file doesn't define any JONAS_ROOT for the server " + server);
        }
        
        final ConfigJonasApply configJonasApply = new ConfigJonasApply(jonasRoot);

        logger.info("Starting configuration set up for server " + server + " in directory " + jonasBase);
        if (jonasBase != null && jonasRoot.equals(jonasBase)) {
            throw new IllegalArgumentException("JONAS_BASE must be different from JONAS_ROOT for server " + server);
        }

        //global options
        configJonasApply.setJOnASBase(jonasBase, server.isJonasBaseToUpdate());
        configJonasApply.setServices(getJOnASServices(server));
        configJonasApply.setDomainName(domainName);
        configJonasApply.setJOnASName(serverName);
        configJonasApply.applyModuleConfiguration(server.getModules());
        configJonasApply.applyExtensionConfiguration(server.getExtensions());
        configJonasApply.execute();

        logger.info("Server " + server + " configured successfully");
    }

    /**
     * Set the configuration of the given agent
     * @param agent The {@link Agent}
     */
    public static void setAConfiguration(final Agent agent) {
        final String domainName = agent.getDomainName();
        final String serverName = agent.getServerName();
        final String jonasRoot = agent.getJonasRoot();
        final String jonasBase = agent.getJonasBase();

        if (jonasRoot == null) {
            throw new IllegalArgumentException("The topology file doesn't define any JONAS_ROOT for the agent " + agent);
        }

        final ConfigJonasApply configJonasApply = new ConfigJonasApply(jonasRoot);

        logger.info("Starting configuration set up for the agent " + agent + " in directory " + jonasBase);
        if (jonasBase != null && jonasRoot.equals(jonasBase)) {
            throw new IllegalArgumentException("JONAS_BASE must be different from JONAS_ROOT for the agent " + agent);
        }

        //global options
        configJonasApply.setJOnASBase(jonasBase, agent.isJonasBaseToUpdate());
        configJonasApply.setServices(getJOnASServices(agent));
        configJonasApply.setDomainName(domainName);
        configJonasApply.setJOnASName(serverName);
        configJonasApply.setClusterConfiguration(domainName, agent.getAgent());
        configJonasApply.applyModuleConfiguration(agent.getModules());
        configJonasApply.applyExtensionConfiguration(agent.getExtensions());
        configJonasApply.execute();

        logger.info("Agent " + agent + " configured successfully");
    }

    /**
     *
     * @param server The {@link Server}
     * @return the list of JOnAS services
     */
    private static String getJOnASServices(final Server server) { 
        StringBuilder stringBuilder = new StringBuilder();
        
        Map<IDeploymeModule, Object> modules = server.getModules();
        if (modules != null) {
            for (IDeploymeModule module: modules.keySet()) {
                if (module != null) {
                    String service = module.getJOnASService();
                    if (service != null && !stringBuilder.toString().contains(service)) {
                        stringBuilder.append(service + ",");
                    }
                } 
            }
        }
        
        Map<IDeploymeExtension, Object> extensions = server.getExtensions();
        if (extensions != null) {
            for (IDeploymeExtension extension: extensions.keySet()) {
                if (extension != null) {
                    String service = extension.getJOnASService();
                    if (service != null && !stringBuilder.toString().contains(service)) {
                        stringBuilder.append(service + ",");
                    }
                }
            }
        }

        String servicesAsString = stringBuilder.toString();
        if (servicesAsString.length() > 0) {
            return servicesAsString.substring(0, servicesAsString.length() - 1);
        } else {
            return servicesAsString.toString();
        }
    }

    /**
     * @param agent The {@link Agent}
     * @return the list of JOnAS services
     */
    private static String getJOnASServices(final Agent agent) {
        StringBuilder stringBuilder = new StringBuilder();

        Map<IDeploymeModule, Object> modules = agent.getModules();
        if (modules != null) {
            for (IDeploymeModule module: modules.keySet()) {
                if (module != null) {
                    String service = module.getJOnASService();
                    if (service != null && !stringBuilder.toString().contains(service)) {
                        stringBuilder.append(service + ",");
                    }
                }
            }
        }

        Map<IDeploymeExtension, Object> extensions = agent.getExtensions();
        if (extensions != null) {
            for (IDeploymeExtension extension: extensions.keySet()) {
                if (extension != null) {
                    String service = extension.getJOnASService();
                    if (service != null && !stringBuilder.toString().contains(service)) {
                        stringBuilder.append(service + ",");
                    }
                }
            }
        }

        String servicesAsString = stringBuilder.toString();
        if (stringBuilder.length() > 0) {
            return servicesAsString.substring(0, servicesAsString.length() - 1);
        } else {
            return servicesAsString.toString();
        }
    }
}
