/**
 * JASMINe Deploy ME [Managed Element]
 * Copyright (C) 2012 Bull S.A.S.
 * Copyright (C) 2012 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: AbstractServer.java 10009 2012-05-04 11:42:14Z cazauxj $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.deployme.v2.util;

import org.ow2.jasmine.deployme.api.IServer;
import org.ow2.jasmine.deployme.api.extensions.DeploymeExtensionException;
import org.ow2.jasmine.deployme.api.extensions.IDeploymeExtension;
import org.ow2.jasmine.deployme.api.modules.IDeploymeModule;
import org.ow2.jasmine.deployme.v2.generated.ConfigurationType;
import org.ow2.jasmine.deployme.v2.generated.PropertiesType;
import org.ow2.jasmine.deployme.v2.generated.PropertyType;
import org.ow2.util.substitution.engine.DefaultSubstitutionEngine;
import org.ow2.util.substitution.resolver.PropertiesResolver;
import org.w3c.dom.Node;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Pattern;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.merge.object.MergeObject;

import javax.xml.bind.JAXBElement;

/**
 * Abstract class in order to factorize some Agent and Server methods
 * @author Jeremy Cazaux
 */
public abstract class AbstractServer implements IServer {

    /**
     * Servers configuration
     */
    protected ConfigurationType serversConfiguration;

    /**
     * The domain configuration
     */
    protected ConfigurationType domainConfiguration;

    /**
     * Domains configuration
     */
    protected ConfigurationType domainsConfiguration;

    /**
     * List of server properties
     */
    protected PropertiesType serverProperties;

    /**
     * List of servers properties
     */
    protected PropertiesType serversProperties;

    /**
     * List of domain properties
     */
    protected PropertiesType domainProperties;

    /**
     * List of domains properties
     */
    protected PropertiesType domainsProperties;

    /**
     * JOnAS ROOT
     */
    protected String jonasRoot;

    /**
     * JOnAS Base
     */
    protected String jonasBase;

    /**
     * True if JONAS_BASE need only to be updated
     */
    protected Boolean updateJonasBase;

    /**
     * The name of the server
     */
    protected String serverName;

    /**
     * The name of the domain
     */
    protected String domainName;

    /**
     * Map between a {@link IDeploymeExtension} and its associated topology element
      */
    protected Map<IDeploymeExtension, Object> extensions;

    /**
     * List of available extensions
     */
    protected List<IDeploymeExtension> availableExtensions;

    /**
     * Map between an {@link IDeploymeModule} and its associated topology element
     */
    private Map<IDeploymeModule, Object> modules;


    /**
     * List of available {@link IDeploymeModule}
     */
    protected List<IDeploymeModule> availableModules;

    /**
     * The logger
     */
    protected static Log logger = LogFactory.getLog(AbstractServer.class);

    /**
     * Java lang pattern
     */
    private static final Pattern pattern = Pattern.compile("java.lang.+");

    /**
     * The {@link DefaultSubstitutionEngine}
     */
    protected DefaultSubstitutionEngine engine;

    /**
     * Default constructor
     */
    public AbstractServer() {
        this.extensions = new HashMap<IDeploymeExtension, Object>();
        this.modules = new HashMap<IDeploymeModule, Object>();
    }

    protected void merge() {
        //TODO jvm-properties, deployables
        mergeConfiguration();
    }

    /**
     * Merge the configuration of a server
     */
    protected void mergeConfiguration() {
        ConfigurationType configuration = getServerConfiguration();
        Boolean inherit = true;
        if (configuration != null && configuration.isInherit() != null) {
            inherit = configuration.isInherit();
        }
        if (inherit) {
            if (this.serversConfiguration != null) {
                //merge server configuration and servers configuration
                mergeConfiguration(configuration, this.serversConfiguration);

                if (this.serversConfiguration.isInherit() != null) {
                    inherit = this.serversConfiguration.isInherit();
                }
            }

            if (inherit && this.domainConfiguration != null) {
                //merge server configuration and domain configuration
                mergeConfiguration(configuration, this.domainConfiguration);

                if (this.domainConfiguration.isInherit() != null) {
                    inherit = this.domainConfiguration.isInherit();
                }
            }

            if (inherit && this.domainsConfiguration != null) {
                //merge server configuration and domains configuration
                mergeConfiguration(configuration, this.domainsConfiguration);
            }
        }
    }

    /**
     * Initialize properties
     */
    protected void initProperties() {
        boolean inherit = true;
        Properties properties = new Properties();
        properties.putAll(System.getProperties());
        properties.putAll(System.getenv());

        if (this.serverProperties != null) {
            addProperties(properties, this.serverProperties.getProperty());
            inherit = this.serverProperties.isInherit();
        }

        if (inherit && this.serversProperties != null) {
            addProperties(properties, this.serversProperties.getProperty());
            inherit = this.serversProperties.isInherit();
        }

        if (inherit && this.domainProperties != null) {
            addProperties(properties, this.domainProperties.getProperty());
            inherit = this.domainProperties.isInherit();
        }

        if (inherit && this.domainsProperties != null) {
            addProperties(properties, this.domainsProperties.getProperty());
            inherit = this.domainsProperties.isInherit();
        }

        //initialize the substitution engine
        this.engine = new DefaultSubstitutionEngine();
        this.engine.setResolver(new PropertiesResolver(properties));
    }

    /**
     * @param properties The {@link Properties}
     * @param propertiesType The {@link PropertiesType}
     */
    private void addProperties(final Properties properties, final List<PropertyType> propertiesType) {
        if (propertiesType != null) {
            for (PropertyType property: propertiesType) {
                String key = property.getName();
                String value = property.getValue();
                if (key != null && value != null && !properties.contains(key)) {
                    properties.put(key, value);
                }
            }
        }
    }

    /**
     * Substitute all properties of all DeployME modules & extensions
     */
    protected void substituteProperties() {
        if (this.modules != null) {
            for(Object object: this.modules.values()) {
                if (object != null) {
                    substituteProperties(object);
                }
            }
        }
        if (this.extensions != null) {
            for(Object object: this.extensions.values()) {
                if (object != null) {
                    substituteProperties(object);
                }
            }
        }
    }

    /**
     * Substitute all properties of the given object
     * @param object An object
     */
    private void substituteProperties(final Object object) {
        if (object != null) {
            List<Field> fields = MergeObject.getAllField(object.getClass());          
            if (fields != null) {
                for (Field field: fields) {
                    int modifier = field.getModifiers();
                    if (!Modifier.isFinal(modifier) && !Modifier.isStatic(modifier)) {
                        Object value = MergeObject.getValue(field, object);
                        if (value instanceof String) {
                            String initialValue = String.valueOf(value);
                            String substitutedString = this.engine.substitute(initialValue);
                            if (substitutedString != null && !initialValue.equals(substitutedString)) {
                                MergeObject.setValue(field, object, substitutedString);
                            }
                        } else {
                            boolean javaLangClass = pattern.matcher(field.getType().getName()).matches();
                            if (value != null && !value.getClass().isPrimitive() && !javaLangClass) {
                                substituteProperties(value);
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * Build modules & extensions of a server
     */
    protected void buildModulesAndExtensions(final ConfigurationType configuration) {
        if (configuration != null) {
            List<Object> objects = configuration.getAny();
            if (objects != null) {
                for (Object object: objects) {
                    if (object instanceof Node) {
                        buildExtension((Node) object);
                    } else if (object instanceof JAXBElement) {
                        buildModule(JAXBElement.class.cast(object).getValue());
                    }
                }
            }
        }
    }

    /**
     * Add a topology object to the list of extensions of the server
     * @param node Node which represents an extension
     */
    private void buildExtension(final Node node) {
        IDeploymeExtension extension;
        if (node != null) {
            final String namespace = node.getNamespaceURI();            
            extension = getAvailableExtension(namespace);
            if (extension != null) {
                try {
                    Object object = extension.convert(node);
                    if (object != null) {
                        this.extensions.put(extension, object);
                    }
                } catch (DeploymeExtensionException e) {
                    logger.error("Cannot get extension object of namespace " + namespace, e);
                }
            } else {
                logger.error("Cannot find a DeployME extension associated to the node " + node.getNodeName()
                        + " with namespace " + namespace);
            }
        }
    }

    /*
     * Add a topolog object to the list of topology modules of the server
     * @param object An object
     */
    private void buildModule(final Object object) {
        IDeploymeModule module = getAvailableModule(object); 
        if (module != null) {
            this.modules.put(module, object);
        } else {
            logger.error("Cannot find a DeployME module associated to the object " + object.getClass().getName());
        }
    }
    
    /**
     * @param object An object
     * @return the associated module if found
     */
    private IDeploymeModule getAvailableModule(final Object object) {
        if (object != null) {
            for (IDeploymeModule module: this.availableModules) {
                if (module.isSupport(object)) {
                    return module;
                }
            }
        }
        return null;
    }
    
    /**
     * @param object An object
     * @return the associated module if found
     */
    private Map.Entry<IDeploymeModule, Object> getModule(final Object object) {
        if (object != null) {
            for (Map.Entry<IDeploymeModule, Object> module: this.modules.entrySet()) {
                IDeploymeModule deploymeModule = module.getKey();                
                if (deploymeModule != null && deploymeModule.isSupport(object)) {
                    return module;
                }
            }
        }
        return null;
    }

    /**
     * @param namespace A namespace
     * @return the first registered extension which is matching the namespace
     */
    private Map.Entry<IDeploymeExtension, Object> getExtension(final String namespace) {
        for (Map.Entry<IDeploymeExtension,Object> extension: this.extensions.entrySet()) {
            IDeploymeExtension deploymeExtension = extension.getKey();
            if (deploymeExtension != null) {
                final String ns = deploymeExtension.getNamespace();
                if (ns != null && ns.equals(namespace)) {
                    return extension;
                }                
            }
        }
        return null;
    }

    /**
     * @param namespace A namespace
     * @return the first available extension which is matching the namespace
     */
    private IDeploymeExtension getAvailableExtension(final String namespace) {
        for (IDeploymeExtension extension: this.availableExtensions) {
            final String ns = extension.getNamespace();
            if (ns != null && ns.equals(namespace)) {
                return extension;
            }
        }
        return null;
    }

    /**                                                    *
     * @return {@link ConfigurationType} of the server
     */
    protected abstract ConfigurationType getServerConfiguration();


    /**
     * Merge the server configuration with an other configuration
     * @param localConfiguration Local configuration
     * @param configuration Another configuration
     */
    protected void mergeConfiguration(ConfigurationType localConfiguration, final ConfigurationType configuration) {
        if (localConfiguration == null) {
            localConfiguration = configuration;
        } else {
            //merge extensions configuration
            List<Object> objects = configuration.getAny();
            if (objects != null) {
                for (Object object: objects) {
                    if (object instanceof Node) {
                        Node node = (Node) object;
                        final String namespace = node.getNamespaceURI();

                        //try to find a local extension which match
                        Map.Entry<IDeploymeExtension, Object> extension = getExtension(namespace);
                        if (extension != null) {
                            IDeploymeExtension deploymeExtension = extension.getKey();
                            Object localTopologyObject = extension.getValue();
                            //convert the Node to a topology extension object and merge it
                            try {
                                Object globalTopologyObject = deploymeExtension.convert(node);
                                MergeObject.mergeObject(localTopologyObject, globalTopologyObject);                                 
                            } catch (DeploymeExtensionException e) {
                                logger.error("Cannot convert node " + node.getNodeName() + " to a Topology object.", e);
                            }
                        } else {
                            buildExtension(node);
                        }
                    } else if (object instanceof JAXBElement) {
                        Object globalTopologyObject = JAXBElement.class.cast(object).getValue();
                        
                        //try to find a module which support the given configuration
                        Map.Entry<IDeploymeModule, Object> module = getModule(globalTopologyObject);
                        if (module != null) {
                            //merge configuration
                            Object localTopologyObject = module.getValue();
                            IDeploymeModule deploymeModule = module.getKey();
                            Map<Class, Field> classIdField = null;
                            if (deploymeModule != null) {
                                classIdField = deploymeModule.getClassIdField();
                            }
                            MergeObject.mergeObject(localTopologyObject, globalTopologyObject, classIdField);
                        } else {
                            buildModule(globalTopologyObject);
                        }
                    }
                }
            }
        }
    }

    /**
     * @return the path to the JOnAS root
     */
    public String getJonasRoot() {
        return this.jonasRoot;
    }

    /**
     * @return the path to the JOnAS base
     */
    public String getJonasBase() {
        return this.jonasBase;
    }

    /**
     * @return the name of the server
     */
    public String getServerName() {
        return this.serverName;
    }

    /**
     * @return the name of the domain
     */
    public String getDomainName() {
        return this.domainName;
    }

    /**
     * @return true if the JONAS BASE need to be only updated
     */
    public Boolean isJonasBaseToUpdate() {
        return this.updateJonasBase;
    }

    /**
     * return "domainName.serverName"
     *
     * @return "domainName.serverName"
     */
    @Override
    public String toString() {
        return this.domainName + "." + this.serverName;
    }

    /**
     * @param jonasBase JONAS_BASE to set
     */
    public void setJonasBase(final String jonasBase) {
        this.jonasBase = this.engine.substitute(jonasBase);
    }

    /**
     * @param jonasRoot JONAS_ROOT to set
     */
    public void setJonasRoot(final String jonasRoot) {
        this.jonasRoot = this.engine.substitute(jonasRoot);
    }

    /**
     * @return the map of pair of <extension, topologyObject> of the server
     */
    public Map<IDeploymeExtension, Object> getExtensions() {
        return this.extensions;
    }

    /**
     * @return the map of pair of <module, topologyObject> of the server
     */
    public Map<IDeploymeModule, Object> getModules() {
        return this.modules;
    }
}
