/**
 * JASMINe Deploy ME [Managed Element]
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DeployMe.java 2730 2008-11-07 14:32:05Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.deployme;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.JAXBException;

import org.ow2.jasmine.deployme.configApply.Configurator;
import org.ow2.jasmine.deployme.generated.Configuration;
import org.ow2.jasmine.deployme.generated.Topology;
import org.ow2.jasmine.deployme.generated.Topology.Domains.Domain;
import org.ow2.jasmine.deployme.generated.Topology.Domains.Domain.Servers.Server;
import org.ow2.jasmine.deployme.loadXml.XmlLoader;
import org.ow2.jasmine.deployme.utils.DeployMEPropertiesManager;
import org.ow2.jasmine.deployme.utils.ServerDomain;
import org.xml.sax.SAXException;

/**
 * The main class of deployMe project.
 * 
 * @author Remy Bresson
 */
public class DeployMe {

    /**
     * The loaded from xml topology
     */
    protected Topology topology;

    /**
     * List of the servers witch deployMe will work
     */
    protected List<ServerDomain> serversDomains;

    /**
     * Default constructor, do nothing <br />
     * Only for test
     */
    protected DeployMe() {

    }

    /**
     * JASMINe-CJC constructor : <br />
     * - Load the topology <br />
     * - Apply the config of the servers(hostname or domain/server/cluster)
     * 
     * @param topologyUrl topology xml file to use url
     * @param hostname The hostname (machine) to use. if set domain or server or
     *        cluster name must not be set
     * @param domain The domain where found the server to use. If set, server or
     *        cluster has to be set, and hostname must no be sett
     * @param server The server to use. If set, domain has to be sett and
     *        cluster or hostname must not be set
     * @param cluster The cluster daemon to use. If set, domain has to be sett
     *        and server or hostname must not be set
     * @throws JAXBException JAXB exception.
     * @throws SAXException SAX exception.
     */
    public DeployMe(final URL topologyUrl, final String hostname, final String domain, final String server, final String cluster)
        throws SAXException, JAXBException {
        this.loadTopology(topologyUrl, DeployMEPropertiesManager.getXsdTopologyURL());

        boolean clusterMode = cluster != null;
        Domain.ClusterDaemons.ClusterDaemon clusterDaemon = null;

        if (clusterMode) {
            clusterDaemon = this.getClusterDaemonConf(domain, cluster);
            if (clusterDaemon == null) {
                throw new IllegalArgumentException("Unable to find cluster daemon : " + domain + "." + cluster);
            }
        }

        this.serversDomains = this.buildServerDomainList(hostname, domain, server, cluster);

        if (clusterMode) {
            /* cluster mode */
            /* Merge configuration with domain */
            clusterDaemon = this.mergeClusterDaemonAndDomainConf(clusterDaemon, this.serversDomains.get(0).getDomain());

            /* Configure cluster */
            Configurator.setClusterAConfiguration(this.serversDomains.get(0).getDomain().getName(), clusterDaemon);
        }

        int i = 1;
        for (ServerDomain sd : this.serversDomains) {
            Configuration confToApply = new ConfiguratorBuilder(sd).getConfig();

            if (clusterMode) {
                /* Create the requested JONAS_BASE */
                confToApply.getGlobalJonas().setJonasBase(clusterDaemon.getJonasBasePrefix() + i++);
            }

            Configurator.setAConfiguration(confToApply, sd);
        }
    }

    /**
     * Load the topology from the url urlTopology using the schema from
     * schemaUrl. Store the topology on DeployMe.topology variable.
     * 
     * @param urlTopology Url of the topology on xml format
     * @param schemaUrl Url of the schema to use (xsd)
     * @throws JAXBException JAXB exception.
     * @throws SAXException SAX exception.
     */
    protected void loadTopology(final URL urlTopology, final URL schemaUrl) throws SAXException, JAXBException {
        XmlLoader xmlLoader = new XmlLoader(urlTopology, schemaUrl);
        this.topology = xmlLoader.getTopology();
    }

    protected Domain.ClusterDaemons.ClusterDaemon getClusterDaemonConf(final String domain, final String clusterName) {
        Domain.ClusterDaemons.ClusterDaemon workingCluster = null;

        for (Domain d : this.topology.getDomains().getDomain()) {
            if (d.getName().compareTo(domain) == 0) {
                for (Domain.ClusterDaemons.ClusterDaemon cd : d.getClusterDaemons().getClusterDaemon()) {
                    if (cd.getClusterDaemonName().compareTo(clusterName) == 0) {
                        workingCluster = cd;
                    }
                }
            }
        }

        return workingCluster;
    }

    /**
     * Build the server list that will be used.
     * 
     * @param hostname host name
     * @param domain domain name
     * @param server server name
     * @param clusterName cluster daemon name
     * @return A list with the server to configure
     */
    protected List<ServerDomain> buildServerDomainList(final String hostname, final String domain, final String server,
        final String clusterName) {
        List<ServerDomain> ret;

        if (hostname != null) {
            ret = this.buildServerListWithHostname(hostname);
        } else if (clusterName == null) {
            ret = this.buildServerListWithServer(domain, server);
        } else {
            ret = this.buildServerListWithCluster(domain, clusterName);
        }

        return ret;
    }

    /**
     * Build the servers list (in this case, only 1 server in the list) defined
     * by domain.server
     * 
     * @param domain the domain
     * @param server the server
     * @return A list with the server to configure
     */
    private List<ServerDomain> buildServerListWithCluster(final String domain, final String cluster) {
        List<ServerDomain> ret = new ArrayList<ServerDomain>();

        Domain.ClusterDaemons.ClusterDaemon workingCluster = null;
        Domain workingDomain = null;
        List<Server> servers = new ArrayList<Server>();

        for (Domain d : this.topology.getDomains().getDomain()) {
            if (d.getName().compareTo(domain) == 0) {
                workingDomain = d;
                for (Domain.ClusterDaemons.ClusterDaemon cd : d.getClusterDaemons().getClusterDaemon()) {
                    if (cd.getClusterDaemonName().compareTo(cluster) == 0) {
                        workingCluster = cd;
                        for (String serverName : cd.getServers().getServerName()) {
                            boolean found = false;
                            for (Server s : workingDomain.getServers().getServer()) {
                                if (s.getName().compareTo(serverName) == 0) {
                                    if (servers.contains(s)) {
                                        throw new IllegalArgumentException("Server '" + serverName
                                            + "' appears twice in the cluster daemon '" + cluster + "' server list!");
                                    }
                                    servers.add(s);
                                    found = true;
                                    break;
                                }
                            }
                            if (!found) {
                                throw new IllegalArgumentException("Server '" + serverName + "' in the cluster daemon '"
                                    + cluster + "' server list does not appear in the topology!");
                            }
                        }
                    }
                }
            }
            if (workingCluster != null) {
                break;
            }
        }

        for (Server s : servers) {
            ret.add(new ServerDomain(s, workingDomain));
        }

        return ret;
    }

    /**
     * Build the servers list (in this case, only 1 server in the list) defined
     * by domain.server
     * 
     * @param domain the domain
     * @param server the server
     * @return A list with the server to configure
     */
    private List<ServerDomain> buildServerListWithServer(final String domain, final String server) {
        List<ServerDomain> ret = new ArrayList<ServerDomain>();

        for (Domain d : this.topology.getDomains().getDomain()) {
            if (d.getName().compareTo(domain) == 0) {
                for (Server s : d.getServers().getServer()) {
                    if (s.getName().compareTo(server) == 0) {
                        ret.add(new ServerDomain(s, d));
                        break;
                    }
                }
                if (!ret.isEmpty()) {
                    break;
                }
            }
        }

        return ret;
    }

    /**
     * Build the servers list hosted by the machine hostname
     * 
     * @param hostname The machine where are located the servers to configure
     * @return A list with the servers to configure
     */
    private List<ServerDomain> buildServerListWithHostname(final String hostname) {
        List<ServerDomain> ret = new ArrayList<ServerDomain>();

        for (Domain d : this.topology.getDomains().getDomain()) {
            for (Server s : d.getServers().getServer()) {
                if (s.getMachine().compareTo(hostname) == 0) {
                    ret.add(new ServerDomain(s, d));
                }
            }
        }

        return ret;
    }

    /**
     * Merge the configuration of the domain with the configuration of the
     * cluster daemon
     * 
     * @param cd cluster daemon configuration
     * @param domain the domain
     * @return the merged configuration
     */
    protected Domain.ClusterDaemons.ClusterDaemon mergeClusterDaemonAndDomainConf(final Domain.ClusterDaemons.ClusterDaemon cd,
        final Domain domain) {
        if (cd.getJonasRoot() == null) {
            cd.setJonasRoot(domain.getConfiguration().getGlobalJonas().getJonasRoot());
        }
        if (cd.getJavaHome() == null) {
            cd.setJavaHome(domain.getConfiguration().getGlobalJonas().getJdk());
        }

        return cd;
    }

}
